/*
* Tyr is an AI for StarCraft: Broodwar, 
* 
* Please visit https://github.com/SimonPrins/Tyr for further information.
* 
* Copyright 2015 Simon Prins
*
* This file is part of Tyr.
* Tyr is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 2 of the License, or
* (at your option) any later version.
* Tyr is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
* You should have received a copy of the GNU General Public License
* along with Tyr.  If not, see http://www.gnu.org/licenses/.
*/


package com.tyr.unitgroups;

import java.util.ArrayList;

import com.tyr.Tyr;
import com.tyr.agents.Agent;

import bwapi.Game;
import bwapi.Player;


/**
 * This class manages a group of units.
 * @author Simon
 *
 */
public abstract class UnitGroup 
{
	/**
	 * The units managed by this unit group.
	 */
	public ArrayList<Agent> units = new ArrayList<Agent>();
	
	/**
	 * The OutOfJob object to which agents will be added when they are no longer needed by this unit group.
	 */
	OutOfJob rejects;
	
	/**
	 * This class manages a group of units.
	 * @param rejects The OutOfJob object to which agents will be added when they are no longer needed by this unit group.
	 */
	public UnitGroup(OutOfJob rejects)
	{
		this.rejects = rejects;
	}
	
	/**
	 * Removes all dead agents from this unit group.
	 */
	public void cleanup()
	{
		for(int i=0; i<units.size(); i++)
			if(units.get(i) == null || units.get(i).isDead())
			{
				units.remove(i);
				i--;
			}
	}
	
	/**
	 * This method is assumed to be called once per frame.
	 * @param game The game.
	 * @param self The player controlled by our bot.
	 * @param bot The instance of Tyr that is currently playing the game.
	 */
	public abstract void onFrame(Game game, Player self, Tyr bot);
	
	/**
	 * Adds an agent to this unit group.
	 * @param agent The agent to be added to this group.
	 */
	public void add(Agent agent)
	{
		units.add(agent);
	}
	
	/**
	 * Remove an agent from this unit group.
	 * @param agent The agent to be removed.
	 */
	public void remove(Agent agent)
	{
		units.remove(agent);
	}
	
	/**
	 * Remove an agent from this unit group at a certain position in the array.
	 * @param pos The position in the array from which the unit is to be removed.
	 */
	public void remove(int pos) 
	{
		units.remove(pos);
	}
	
	/**
	 * Removes an agent from this unit group and returns it.
	 * @return The removed agent. This may be null if no agent is available. This does not necessarily mean that this unit group contains no more agents.
	 */
	public Agent pop()
	{
		if(units.size() == 0)
			return null;
		Agent result = units.get(units.size()-1);
		remove(units.size()-1);
		return result;
	}
	
	/**
	 * Returns true if the unit group wants to take control of the newly created agent.
	 * @param agent The agent for which the UnitGroup must determine if it wants the control.
	 * @return Return the priority with which the unit group wishes to take the unit, -1 means doesn't want it, otherwise the unit group with the highest priority gets it.
	 */
	public int takeAgent(Agent agent)
	{
		return -1;
	}
}
