/*
* Tyr is an AI for StarCraft: Broodwar, 
* 
* Please visit https://github.com/SimonPrins/Tyr for further information.
* 
* Copyright 2015 Simon Prins
*
* This file is part of Tyr.
* Tyr is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 2 of the License, or
* (at your option) any later version.
* Tyr is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
* You should have received a copy of the GNU General Public License
* along with Tyr.  If not, see http://www.gnu.org/licenses/.
*/


package com.tyr.tasks;

import java.util.ArrayList;
import java.util.List;

import com.tyr.DebugMessages;
import com.tyr.OrderedExpands;
import com.tyr.PositionUtil;
import com.tyr.Tyr;
import com.tyr.agents.Agent;
import com.tyr.unitgroups.ZealotDropGroup;

import bwapi.Game;
import bwapi.Player;
import bwapi.Position;
import bwapi.UnitType;

/**
 * This class implements a solution for constantly pushing the opponent.
 */
public class ZealotDropSolution extends Solution
{
	/**
	 * The OrderedExpands object for our clockwise scouting observer.
	 */
	private OrderedExpands cwExpands = new OrderedExpands(true, true, true, true);
	
	List<ZealotDropGroup> zealotDropGroups = new ArrayList<ZealotDropGroup>();
	
	private int mode = GATHER;
	private static final int GATHER = 0;
	private static final int MOVE = 1;
	private static final int DROP = 2;
	
	/**
	 * This class implements a solution for constantly pushing the opponent.
	 * @param task The task that started this solution.
	 */
	public ZealotDropSolution(Task task, int shuttleCount) 
	{
		super(task);
		for (int i=0; i<shuttleCount; i++)
		{
			ZealotDropGroup group = new ZealotDropGroup();
			group.attackMode = ZealotDropGroup.ATTACK_BASE;
			zealotDropGroups.add(group);
		}
	}
	
	@Override
	public void onFrame(Game game, Player self, Tyr bot)
	{
		if (mode == GATHER)
			gather();
		else if (mode == MOVE)
			move();
		else
			drop();
		
		DebugMessages.addMessage("Going for drop. Mode: " + mode);
		
		for (ZealotDropGroup drop : zealotDropGroups)	
		{
			drop.cleanup();
			drop.onFrame(game, self, bot);
		}
	}
	
	private void gather()
	{
		boolean finished = true;
		for (ZealotDropGroup drop : zealotDropGroups)
		{
			if (drop.needsShuttle() || drop.getZealots() < 4)
			{
				finished = false;
				break;
			}
		}
		
		if (finished)
			mode = MOVE;
	}
	
	private void move()
	{
		if (Tyr.bot.suspectedEnemy.size() == 0)
			return;
		final Position target = getTarget();
		if (target == null)
			return;
		final Position enemyMain = Tyr.bot.suspectedEnemy.get(0).getPosition();
		if (PositionUtil.distanceSq(target, enemyMain) <= 100 * 100)
			mode = DROP;
		for (ZealotDropGroup drop : zealotDropGroups)
			drop.setTarget(target);
	}
	
	private void drop()
	{
		if (Tyr.bot.suspectedEnemy.size() == 0)
			return;
		final Position target = Tyr.bot.suspectedEnemy.get(0).getPosition();
		for (ZealotDropGroup drop : zealotDropGroups)
			drop.setTarget(target);
	}
	
	/**
	 * Add an agent to the attack group with the attacking agents.
	 * @param unit The agent to be added.
	 */
	public void add(Agent agent)
	{
		if (agent.unit.getType() == UnitType.Protoss_Shuttle)
		{
			for (ZealotDropGroup drop : zealotDropGroups)
				if (drop.needsShuttle())
				{
					drop.add(agent);
					return;
				}
		}
		else
		{
			for (ZealotDropGroup drop : zealotDropGroups)
				if (drop.getZealots() < 4)
				{
					drop.add(agent);
					return;
				}
		}
		Tyr.bot.hobos.add(agent);
	}
	
	/**
	 * Gets us the next target to harass.
	 */
	private Position getTarget()
	{
		if (!cwExpands.tryInitialize())
			return null;

		final Position current = cwExpands.getPos();
		
		if (current == null)
			cwExpands.next();
		else
		{
			for (ZealotDropGroup drop : zealotDropGroups)
			{
				if (drop.getShuttle() != null && drop.getShuttle().distanceSquared(current) <= 200 * 200)
				{
					cwExpands.next();
					break;
				}
			}
		}
		
		return cwExpands.getPos();
	}
	
	public int getZealots()
	{
		int zealots = 0;
		for (ZealotDropGroup drop : zealotDropGroups)
			zealots += drop.getZealots();
		return zealots;
	}
	
	public int getShuttles()
	{
		int shuttles = 0;
		for (ZealotDropGroup drop : zealotDropGroups)
			if (!drop.needsShuttle())
				shuttles++;
		return shuttles;
	}
}
