/*
* Tyr is an AI for StarCraft: Broodwar, 
* 
* Please visit https://github.com/SimonPrins/Tyr for further information.
* 
* Copyright 2015 Simon Prins
*
* This file is part of Tyr.
* Tyr is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 2 of the License, or
* (at your option) any later version.
* Tyr is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
* You should have received a copy of the GNU General Public License
* along with Tyr.  If not, see http://www.gnu.org/licenses/.
*/


package com.tyr.tasks;

import java.util.ArrayList;

import com.tyr.Tyr;
import com.tyr.agents.Agent;
import com.tyr.agents.None;
import com.tyr.unitgroups.MarineHarassGroup;
import com.tyr.unitgroups.UnitGroup;

import bwapi.Color;
import bwapi.Game;
import bwapi.Player;
import bwapi.Position;
import bwapi.UnitType;

/**
 * This class implements a solution for attacking the opponent.
 * During the attack it will periodically regroup to make sure attacking units stay together.
 */
public class MarineHarassSolution extends Solution
{
	/**
	 *  The group of attacking units.
	 */
	private MarineHarassGroup attackGroup;
	
	/**
	 *  The target of the attack.
	 */
	private Position attackTarget;
	
	/**
	 * This class implements a solution for attacking the opponent.
	 * During the attack it will periodically regroup to make sure attacking units stay together.
	 * @param task The task that started this solution.
	 * @param attackTarget The position which we should attack.
	 */
	public MarineHarassSolution(Task task, Position attackTarget) 
	{
		super(task);
		
		this.attackTarget = attackTarget;
		
		this.attackGroup = new MarineHarassGroup(attackTarget);
	}
	
	/**
	 * Sets the target which the units will attack.
	 * @param attackTarget
	 */
	public void setTarget(Position attackTarget)
	{
		this.attackTarget = attackTarget;
	}

	@Override
	public void onFrame(Game game, Player self, Tyr bot)
	{
		// Draw some debug info.
		Tyr.drawCircle(attackTarget, Color.Orange, 64);

		// Manage the attackGroup.
		attackGroup.cleanup();
		
		attackGroup.setTarget(attackTarget);
		attackGroup.onFrame(game, self, bot);
		

		// If we have no more attacking units, the medics can return home.
		boolean attackerRemaining = false;
		for(Agent agent : attackGroup.units){
			if (agent.unit.getType() == UnitType.Terran_Medic)
				continue;
			attackerRemaining = true;
			break;
		}
		
		if (!attackerRemaining)
			attackGroup.clear();
	}
	
	/**
	 * Add an agent to the attack group with the attacking agents.
	 * @param unit The agent to be added.
	 */
	public void add(Agent agent)
	{
		attackGroup.add(agent);
	}
	
	/**
	 * This method is called when this solution is no longer needed.
	 */
	public void done(UnitGroup unitGroup) 
	{
		for(Agent unit : attackGroup.units)
		{
			unitGroup.add(unit);
			unit.order(new None(unit));
		}
		attackGroup.units = new ArrayList<Agent>();
	}

	/**
	 * Clears the solution so that it can be removed.
	 */
	public void clear() 
	{
		attackGroup.clear();
	}

	/**
	 * Returns the number if agents sent out by this solution.
	 * @return Returns the number if agents sent out by this solution.
	 */
	public int size() 
	{
		return attackGroup.units.size();
	}
}
