/*
* Tyr is an AI for StarCraft: Broodwar, 
* 
* Please visit https://github.com/SimonPrins/Tyr for further information.
* 
* Copyright 2015 Simon Prins
*
* This file is part of Tyr.
* Tyr is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 2 of the License, or
* (at your option) any later version.
* Tyr is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
* You should have received a copy of the GNU General Public License
* along with Tyr.  If not, see http://www.gnu.org/licenses/.
*/


package com.tyr.tasks;

import java.util.ArrayList;

import com.tyr.DebugMessages;
import com.tyr.EnemyManager;
import com.tyr.EnemyPosition;
import com.tyr.Tyr;
import com.tyr.agents.Agent;
import com.tyr.unitgroups.MineralWorkers;

import bwapi.Color;
import bwapi.Game;
import bwapi.Order;
import bwapi.Player;
import bwapi.Position;
import bwapi.UnitType;
import bwta.BaseLocation;

/**
 * This class implements a solution for harassing the opponents expands with vultures.
 */
public class DoomDropSolution extends Solution 
{
	/**
	 * The unit group of units that perform the harass.
	 */
	private AttackRegroupSolution attackRegroup;
	
	/**
	 * Unit groups that were previously used for the attack.
	 */
	private ArrayList<AttackRegroupSolution> previousAttacks = new ArrayList<AttackRegroupSolution>();
	
	/**
	 * The position of the base we are currently attacking in the list of bases returned by EnemyManager.getOrderedexpands().
	 */
	private int currentPos;
	
	/**
	 * The direction in which we are currently going through the list of bases returned by EnemyManager.getOrderedexpands().
	 */
	private int currentDir = 1;
	
	/**
	 * The position of the base we currently intend to harass.
	 */
	private Position target;
	
	/**
	 * The position where our forces will attack next.
	 */
	private Position attackTarget;
	
	/**
	 * The list of dropships we will use for the harass.
	 */
	private ArrayList<Agent> dropships;
	
	/**
	 * The list of units which will perform the drop.
	 */
	private ArrayList<Agent>[] units;
	
	/**
	 * List of all expands, in the order that they are encountered in a sweep around the map.
	 * Includes opponents main.
	 */
	private ArrayList<Position> orderedExpands;
	
	/**
	 * Have the dropships returned to the base?
	 */
	public boolean returned;
	
	/**
	 * The mode our drop is in.
	 * Either load, moveOut or attack.
	 */
	private int mode = load;
	
	private static final int load = 0;
	private static final int moveOut = 1;
	private static final int attack = 2;
	
	/**
	 * This class implements a solution for harassing the opponents expands with vultures.
	 * @param task The task that started this solution.
	 * @param dropships The list of dropships we will use for the harass.
	 * @param units The list of units which will perform the drop.
	 * @param currentDir The direction in which the drop will move, either 1 or -1. 
	 */
	public DoomDropSolution(Task task, ArrayList<Agent> dropships, ArrayList<Agent>[] units, int currentDir) 
	{
		super(task);
		this.dropships = dropships;
		this.units = units;
		this.currentDir = currentDir;
		determineTarget();
	}
	
	@Override
	public void onFrame(Game game, Player self, Tyr bot)
	{
		// If we find a new target to attack, we set that new target.
		boolean newTarget = acquireTarget(Tyr.bot);

		if (attackRegroup != null)
			Tyr.drawCircle(attackRegroup.getCenter(), Color.Blue, 20);
		if (attackTarget != null)
			Tyr.drawCircle(attackTarget, Color.Red, 20);
			
		if (newTarget && attackRegroup != null)
			attackRegroup.setTarget(attackTarget);
		
		for (AttackRegroupSolution sol : previousAttacks)
		{
			if (attackTarget != null)
				sol.setTarget(attackTarget);
			sol.onFrame(game, self, bot);
		}
		
		DebugMessages.addMessage("Previous attacks remaining: " + previousAttacks.size());
		
		for (int i=previousAttacks.size()-1; i >= 0; i--)
			if (previousAttacks.get(i).done())
				previousAttacks.remove(i);
		
		// Cleanup the lists of units.
		for (int i = dropships.size()-1; i>= 0; i--)
			if(dropships.get(i) != null && dropships.get(i).isDead())
				dropships.set(i, null);
		
		for (ArrayList<Agent> passengers : units)
			for (int i = passengers.size()-1; i>= 0; i--)
				if(passengers.get(i).isDead())
					passengers.remove(i);
		
		DebugMessages.addMessage("Dropship mode = " + mode);
		
		if (attackRegroup != null)
			attackRegroup.onFrame(game, self, bot);
		
		if (mode == load)
			load();
		else if (mode == moveOut)
			moveOut();
		else if (mode == attack)
			attack();
	}
	
	/**
	 * attack mode, where the actual attack takes place.
	 */
	@SuppressWarnings("unchecked")
	private void attack() 
	{	
		if (dropships.size() == 0)
			return;
		
		returned = true;
		
		for (int listIt = 0; listIt < units.length; listIt++)
		{
			Agent dropship = dropships.get(listIt);
			boolean isCarrying = false;
			ArrayList<Agent> passengers = units[listIt];
			for (Agent passenger : passengers)
			{
				if(passenger.unit.isLoaded())
				{
					dropship.unit.unload(passenger.unit);
					isCarrying = true;
					returned = false;
					break;
				}
			}

			if (isCarrying && Tyr.bot.spaceManager.map[dropship.unit.getTilePosition().getX()][dropship.unit.getTilePosition().getY()] != 1)
				dropship.unit.move(target);
			
			if(isCarrying && Tyr.bot.spaceManager.map[dropship.unit.getTilePosition().getX()][dropship.unit.getTilePosition().getY()] == 1)
			{
				if(dropship.unit.getOrder() == Order.Move)
					dropship.unit.stop();
			}
			
			if (dropship != null && ! dropship.isDead() && !isCarrying)
			{
				dropship.unit.move(Tyr.tileToPosition(Tyr.self.getStartLocation()));
				if (dropship.distanceSquared(Tyr.tileToPosition(Tyr.self.getStartLocation())) > 200*200 )
					returned = false;
			}
		}
		
		if (returned)
		{
			if (attackRegroup != null)
			{
				previousAttacks.add(attackRegroup);
				attackRegroup = null;
			}

			units = new ArrayList[dropships.size()];
			for (int i=0; i<dropships.size(); i++)
				units[i] = new ArrayList<Agent>();
			mode = load;
		}
	}
	
	/**
	 * moveOut mode. The dropships move out to an enemy base.
	 */
	private void moveOut()
	{
		// See if we need to change the target.
		determineTarget();
		int dist = 0;
		for(Agent agent : dropships)
		{
			if (agent == null)
				continue;
			agent.unit.move(target);
			Tyr.game.drawLineMap(agent.unit.getX(), agent.unit.getY(), target.getX(), target.getY(), Color.Blue);
			dist = Math.min(dist, agent.distanceSquared(target));
		}
		
		if (dist > 300*300)
			return;

		if (currentPos >= orderedExpands.size() - 1)
		{
			startAttack();
			return;
		}
		
		// See if we have found an enemy base.
		for (EnemyPosition enemy : EnemyManager.getManager().enemyBuildingMemory)
		{
			if (!enemy.type.isResourceDepot())
				continue;
			if (enemy.pos.getDistance(target) < 100)
			{
				startAttack();
				break;
			}
		}
	}
	
	/**
	 * Starts the attack mode.
	 */
	public void startAttack()
	{
		mode = attack;
		if (attackRegroup == null)
			attackRegroup = new AttackRegroupSolution(this.task, target);
		for (ArrayList<Agent> passengers : units)
			for (Agent agent : passengers)
				attackRegroup.add(agent);
	}

	/**
	 * Load mode, loads the units into the dropship.
	 */
	private void load()
	{
		returned = true;
		for (int listIt = 0; listIt < units.length; listIt++)
			if (units[listIt].size() > 0)
				returned = false;
		int totalPassengers = 0;
		boolean done = true;
		for (int listIt = 0; listIt < units.length; listIt++)
		{
			ArrayList<Agent> passengers = units[listIt];
			Agent dropship = dropships.get(listIt);
			
			if (dropship == null || dropship.isDead())
				continue;
			totalPassengers += passengers.size();
			for (Agent agent : passengers)
			{
				if (agent != null && !agent.isDead() && !agent.unit.isLoaded())
				{
					done = false;
					dropship.unit.load(agent.unit);
					if (agent.unit.getType() == UnitType.Terran_Siege_Tank_Siege_Mode)
						agent.unit.unsiege();
					else if (agent.unit.getTargetPosition() == null || agent.unit.getTargetPosition().getDistance(dropship.unit.getPosition()) >= 100)
						agent.unit.move(dropship.unit.getPosition());
				}
			}
		}
		if (done && totalPassengers > 0)
		{
			mode = moveOut;
			target = null;
			return;
		}
	}
	
	/**
	 * Determines the base we want to attack.
	 * @return Whether the method call actually changes the target.
	 */
	private boolean determineTarget()
	{
		boolean result = false;
		
		ArrayList<Position> orderedExpands = EnemyManager.getManager().getOrderedExpands();
		
		if (orderedExpands != null)
		{
			for(int i=0; i<orderedExpands.size(); i++)
			{
				Position pos = orderedExpands.get(i);
				Tyr.drawCircle(pos, Color.Red, 32);
				Tyr.game.drawTextMap(pos.getX(), pos.getY(), i + "");
			}
			
			if (target != null)
				Tyr.drawCircle(target, Color.Red);
		}
		// If the target has not previously been initialized, initalize it.
		if (target == null)
		{
			if (orderedExpands == null)
			{
				DebugMessages.addMessage("orderedExpands is null.");
				return false;
			}
			
			currentPos = EnemyManager.getManager().getSelfPos();
			if(currentPos != -1)
			{
				target = orderedExpands.get(currentPos);
				result = true;
			}
		}
		
		// Determine whether we are done harassing the current target.
		for(int i=0; i < 2*orderedExpands.size(); i++)
		{
			boolean isIsland = false;
			for (BaseLocation loc : Tyr.bot.expands)
			{
				if (!loc.isIsland())
					continue;
				if (loc.getDistance(target) <= 128)
					isIsland = true;
			}
			
			// If the target is invisible, we are not done.
			if (!isIsland && target != null && !Tyr.game.isVisible(Tyr.positionToTile(target)))
				break;
			
			// If the vultures are too far away from that base, we do not yet look for a new target.
			boolean closeEnough = false;
			
			// If we have a base there, we do not need to harass it.
			for (MineralWorkers base : Tyr.bot.workForce.mineralWorkers)
				if (base.resourceDepot != null && base.resourceDepot.getDistance(target) < 100)
					closeEnough = true;
			
			if (!isIsland && !closeEnough && dropships != null)
			{
				for(Agent agent : dropships)
				{
					if (agent == null)
						continue;
					
					if (agent.distanceSquared(target) < 128 * 128)
						closeEnough = true;
				}
			}
			if (!isIsland && !closeEnough)
				break;
			
			// We are done with the current target, get the next one.
			nextTarget();
			
			result = true;
		}
		
		return result;
	}
	
	/**
	 * Gets us the next target to harass.
	 */
	@SuppressWarnings("unchecked")
	private void nextTarget()
	{
		if (orderedExpands == null)
		{
			orderedExpands = (ArrayList<Position>)EnemyManager.getManager().getOrderedExpands().clone();
			if (orderedExpands == null)
				return;
			if (Tyr.bot.suspectedEnemy.size() == 1)
				orderedExpands.add(Tyr.bot.suspectedEnemy.get(0).getPosition());
		}
		
		if (currentPos >= orderedExpands.size() - 1)
			return;
		
		currentPos = (currentPos + currentDir + orderedExpands.size()) % orderedExpands.size();
		target = orderedExpands.get(currentPos);
	}

	/**
	 * Whether the harass is done.
	 * @return Whether the harass is done.
	 */
	public boolean done() 
	{
		for (ArrayList<Agent> passengers : units)
			if (passengers.size() > 0)
				return false;
		return true;
	}

	/**
	 * Clears the solution once it is done.
	 */
	public void clear() 
	{
		for (Agent dropship : dropships)
			Tyr.bot.hobos.add(dropship);
		for (ArrayList<Agent> passengers : units)
			for (Agent agent : passengers)
				Tyr.bot.hobos.add(agent);
	}

	
	/**
	 * See if we need to find a new target to attack.
	 * If so we return true, if the current target is already fine, we return false.
	 */
	public boolean acquireTarget(Tyr bot)
	{
		if (attackTarget != null)
	    	for(EnemyPosition p : EnemyManager.getManager().enemyBuildingMemory)
	    		if (p.pos.getX() == attackTarget.getX() && p.pos.getY() == attackTarget.getY())
	    			return false;
		
		Position newTarget = null;
    	for(EnemyPosition p : EnemyManager.getManager().enemyBuildingMemory)
    	{
    		newTarget = p.pos;
			break;
    	}

    	if (newTarget == null)
    		for (BaseLocation b : bot.suspectedEnemy) 
    		{
    			newTarget = b.getPosition();
    			break;
    		}
    	
    	if (newTarget == null)
    		return false;
    	
    	if (attackTarget == null)
    	{
    		attackTarget = newTarget;
    		return true;
    	}
    	
    	
    	if (attackTarget.getX() == newTarget.getX() && attackTarget.getY() == newTarget.getY())
    	{
			DebugMessages.addMessage("Target has not changed.");
    		return false;
    	}
    	
    	attackTarget = newTarget;
    	return true;
	}

	/**
	 * How many dropships are left alive?
	 * @return How many dropships are left alive?
	 */
	public int dropshipsLeft()
	{
		int result = 0;
		for (Agent dropship : dropships)
			if (dropship != null && !dropship.isDead())
				result++;
		return result;
	}

	/**
	 * Removes all null dropships.
	 */
	public void clearDropships() {
		ArrayList<Agent> newShips = new ArrayList<Agent>();
		for (Agent dropship : dropships)
			if (dropship != null)
				newShips.add(dropship);
		dropships = newShips;
	}

	/**
	 * The list of dropships we will use for the harass.
	 * @return The list of dropships we will use for the harass.
	 */
	public ArrayList<Agent> getDropships() {
		return dropships;
	}

	/**
	 * The list of units which will perform the drop.
	 * @return The list of units which will perform the drop.
	 */
	public ArrayList<Agent>[] getUnits() {
		return units;
	}

	public boolean attackersRemain()
	{
		return previousAttacks.size() > 0;
	}	
}
