/*
* Tyr is an AI for StarCraft: Broodwar, 
* 
* Please visit https://github.com/SimonPrins/Tyr for further information.
* 
* Copyright 2015 Simon Prins
*
* This file is part of Tyr.
* Tyr is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 2 of the License, or
* (at your option) any later version.
* Tyr is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
* You should have received a copy of the GNU General Public License
* along with Tyr.  If not, see http://www.gnu.org/licenses/.
*/


package com.tyr.builds;

import com.tyr.EnemyManager;
import com.tyr.Tyr;
import com.tyr.UnitTracker;
import com.tyr.buildingplacement.BaseBuildSite;
import com.tyr.buildingplacement.BuildCommand;
import com.tyr.unitgroups.MineralWorkers;

import bwapi.Game;
import bwapi.Player;
import bwapi.Race;
import bwapi.Unit;
import bwapi.UnitType;


/**
 * This BuildOrder chooses when to expand to new resource locations.
 * This is meant to be used as a part of a larger build using the CompositeBuildOrder class.
 */
public class ExpandPart extends BuildOrder
{
	/**
	 * Are we building a Command Center, a Nexus or a Hatchery? 
	 */
	private UnitType building;
	
	/**
	 * How many minerals are required to build the building?
	 */
	private int requiredMinerals;
	
	/**
	 * The maximum number of ccs we will build.
	 */
	public static int maximumCcs = 100;
	
	/**
	 * Do we take a gas expand next?
	 */
	public boolean takeGasNext = false;

	/**
	 * This BuildOrder chooses when to expand to new resource locations.
	 * This is meant to be used as a part of a larger build using the CompositeBuildOrder class.
	 */
	public ExpandPart()
	{
		super();
	}
	/**
	 * This BuildOrder chooses when to expand to new resource locations.
	 * This is meant to be used as a part of a larger build using the CompositeBuildOrder class.
	 * @param takeGasNext Do we take a gas expand next?
	 */
	public ExpandPart(boolean takeGasNext)
	{
		super();
		this.takeGasNext = takeGasNext;
	}
	
	@Override
	public void initialize(Game game, Player self, Tyr bot)
	{
		if (self.getRace() == Race.Terran)
			building = UnitType.Terran_Command_Center;
		else if (self.getRace() == Race.Protoss)
			building = UnitType.Protoss_Nexus;
		else
			building = UnitType.Zerg_Hatchery;
		
		requiredMinerals = building.mineralPrice() - 50;
	}
	
	@Override
	public void onFrame(Game game, Player self, Tyr bot)
	{
		// Do not build more expands than we are allowed.
		if (UnitTracker.getCcCount() >= maximumCcs)
			return;
		
		// Do not expand if we are under attack.
		// If there are only workers attacking, we do expand though.
		if (!EnemyManager.getManager().isInvaderWorker())
			return;
		
		// If an expansion is already being taken, don't take another.
		for(BuildCommand command : bot.buildCommands)
			if (command.building.isResourceDepot())
				return;
		
		// How many mineral patches and minerals are still available to us?
		int patchCount = 0;
		int totalMineralsLeft = bot.getAvailableMinerals();
		for(MineralWorkers base : bot.workForce.mineralWorkers)
		{
			patchCount += base.minerals.size();
			for (Unit patch : base.minerals)
				totalMineralsLeft += patch.getResources();
		}
		
		
		
		
		
		if (
				// Do we have enough minerals?
				(bot.getAvailableMinerals() >= requiredMinerals || patchCount <= 6 || totalMineralsLeft <= 800) 
				&&
				(
					// First two expands can always be taken.
					UnitTracker.getCcCount() < 3 
					// Later expands are taken only if we are running out of mineral patches.
						|| patchCount * 2 + 4 < 
							bot.workForce.units.size() + bot.builders.units.size()
						|| totalMineralsLeft <= 800
				)
			)
		{
			bot.spaceManager.build(building, null, new BaseBuildSite(takeGasNext));
			takeGasNext = false;
		}
	}
}
