/*
* Tyr is an AI for StarCraft: Broodwar, 
* 
* Please visit https://github.com/SimonPrins/Tyr for further information.
* 
* Copyright 2015 Simon Prins
*
* This file is part of Tyr.
* Tyr is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 2 of the License, or
* (at your option) any later version.
* Tyr is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
* You should have received a copy of the GNU General Public License
* along with Tyr.  If not, see http://www.gnu.org/licenses/.
*/


package com.tyr.builds;

import com.tyr.DebugMessages;
import com.tyr.Settings;
import com.tyr.StopWatch;
import com.tyr.Tyr;
import com.tyr.UnitTracker;
import com.tyr.agents.Agent;
import com.tyr.buildingplacement.DefensiveStructures;
import com.tyr.requirements.CostRequirement;
import com.tyr.requirements.DisjRequirement;
import com.tyr.requirements.LinearMax;
import com.tyr.requirements.UnitRequirement;
import com.tyr.tasks.AssignScience;
import com.tyr.tasks.AttackTask;
import com.tyr.unitgroups.MineralWorkers;

import bwapi.Game;
import bwapi.Player;
import bwapi.Race;
import bwapi.TechType;
import bwapi.UnitType;
import bwapi.UpgradeType;


/**
 * Implements a bio build.
 * @author Simon
 *
 */
public class Bio extends CompositeBuildOrder
{
	/**
	 * Are we ready to take expands?
	 */
	private boolean takeExpands = false;
	
	/**
	 * The mode in which the build currently is.
	 */
	private int mode = GREEDY;
	
	/**
	 * The modes the build can be in.
	 * Determines how soon it will expand and how many bunkers will be built.
	 */
	public static int GREEDY = 0, DEFENSIVE = 1, ULTRA_DEFENSIVE = 2, RUSH = 3;
	
	/**
	 * The minimum size the army needs to be for the first attack.
	 */
	private int minimumArmySize = 30;
	
	/**
	 * The maximum size the army needs to be for an attack.
	 */
	private int maximumArmySize = 60;
	
	/**
	 * Should we build turrets?
	 */
	private boolean buildTurrets = false;
	
	/**
	 * Should we build a single turret at our natural?
	 */
	private boolean buildNaturalTurret = false;
	
	/**
	 * Do we use science vessels?
	 */
	private boolean vesselSupport = true;

	/**
	 * Index of the base where we last tried to place a turret.
	 */
	private int turretPos;
	
	/**
	 * How many turrets do we build near our supply depots?
	 */
	private int depotTurrets = 0;
	
	/**
	 * Implements a bio build.
	 */
	public Bio()
	{}
	
	/**
	 * Implements a bio build.
	 * @param mode The mode in which the build will be, options are GREEDY, DEFENSIVE and ULTRA_DEFENSIVE.
	 */
	public Bio(int mode)
	{
		this.mode = mode;
		if (mode == RUSH)
			minimumArmySize = 15;
	}
	
	/**
	 * Implements a bio build.
	 * @param mode The mode in which the build will be, options are GREEDY, DEFENSIVE and ULTRA_DEFENSIVE.
	 * @param minimumArmySize The minimum size the army needs to be for the first attack.
	 */
	public Bio(int mode, int minimumArmySize)
	{
		this.mode = mode;
		this.minimumArmySize = minimumArmySize;
	}
	
	/**
	 * Implements a bio build.
	 * @param mode The mode in which the build will be, options are GREEDY, DEFENSIVE and ULTRA_DEFENSIVE.
	 * @param minimumArmySize The minimum size the army needs to be for the first attack.
	 * @param buildTurrets Should we build turrets?
	 * @param buildNaturalTurret Should we build a single turret at our natural?
	 * @param depotTurrets How many turrets do we build near our supply depots?
	 */
	public Bio(int mode, int minimumArmySize, boolean buildTurrets, boolean buildNaturalTurret, int depotTurrets)
	{
		this.mode = mode;
		this.minimumArmySize = minimumArmySize;
		this.buildTurrets = buildTurrets;
		this.buildNaturalTurret = buildNaturalTurret;
		this.depotTurrets = depotTurrets;
	}
	
	/**
	 * Implements a bio build.
	 * @param mode The mode in which the build will be, options are GREEDY, DEFENSIVE and ULTRA_DEFENSIVE.
	 * @param minimumArmySize The minimum size the army needs to be for the first attack.
	 * @param maximumArmySize The maximum size the army needs to be for an attack.
	 */
	public Bio(int mode, int minimumArmySize, int maximumArmySize) {
		this.mode = mode;
		this.minimumArmySize = minimumArmySize;
		this.maximumArmySize = maximumArmySize;
	}

	@Override
	public void initialize(Game game, Player self, Tyr bot)
	{
		if (mode == DEFENSIVE || mode == ULTRA_DEFENSIVE)
			Settings.setLargeInvasionDist(720);
		
		Settings.setRequiredSize(minimumArmySize);
		Settings.setMaximumSize(maximumArmySize);
		
		Settings.setWorkersPerGas(1);
		if(mode == RUSH)
			Settings.setMaximumWorkers(20);
		
		if (game.enemy().getRace() == Race.Zerg && mode == DEFENSIVE)
			bot.bunkers.maximumRepair = 1;
		
		this.add(new SupplyDepotPart());
		this.add(new WorkerScoutPart(1600));

		this.add(new GenericPartFactory(UnitType.Terran_Barracks)
			.setCost(mode == DEFENSIVE?100:150, 0)
			.create());
		
		GenericPartFactory barracks2Factory = new GenericPartFactory(UnitType.Terran_Barracks)
			.setMax(2)
			.setCost(mode == DEFENSIVE?100:150, 0);
		this.add(barracks2Factory.create());
		if (mode == GREEDY)
			barracks2Factory.add(new UnitRequirement(UnitType.Terran_Command_Center, 2));
		if (mode != GREEDY && mode != RUSH)
			barracks2Factory.add(new UnitRequirement(UnitType.Terran_Bunker));
		
		GenericPartFactory barracksFactory = new GenericPartFactory(UnitType.Terran_Barracks)
			.setMax(8)
			.setCost(mode == DEFENSIVE?100:150, 0)
			.add(new LinearMax(UnitType.Terran_Command_Center, UnitType.Terran_Barracks, 3, 0, false))
			.add(new UnitRequirement(UnitType.Terran_Academy));
		if (mode == GREEDY)
			barracksFactory.add(new UnitRequirement(UnitType.Terran_Command_Center, 2));
		if (mode != GREEDY && mode != RUSH)
			barracksFactory.add(new UnitRequirement(UnitType.Terran_Bunker));
		if (vesselSupport)
			barracksFactory.add(new DisjRequirement()
				.addRequirement(new UnitRequirement(UnitType.Terran_Science_Facility))
				.addRequirement(new CostRequirement(350, 0)));
		this.add(barracksFactory.create());
		
		GenericPartFactory academyFactory = new GenericPartFactory(UnitType.Terran_Academy)
			.add(new UnitRequirement(UnitType.Terran_Barracks))
			.add(new UnitRequirement(UnitType.Terran_Refinery));
		if (mode == GREEDY)
			academyFactory.add(new UnitRequirement(UnitType.Terran_Command_Center, 2));
		this.add(academyFactory.create());
		
		this.add(new GenericPartFactory(UnitType.Terran_Engineering_Bay)
			.setMax(2)
			.add(new UnitRequirement(UnitType.Terran_Marine, 10))
			.add(new UnitRequirement(UnitType.Terran_Academy))
			.create());
		
		this.add(new GenericPartFactory(UnitType.Terran_Factory)
			.add(new UnitRequirement(UnitType.Terran_Barracks, vesselSupport?2:6))
			.add(new UnitRequirement(UnitType.Terran_Engineering_Bay, 2))
			.create());
		
		this.add(new GenericPartFactory(UnitType.Terran_Starport)
			.add(new UnitRequirement(UnitType.Terran_Barracks, vesselSupport?2:6))
			.add(new UnitRequirement(UnitType.Terran_Engineering_Bay, 2))
			.add(new UnitRequirement(UnitType.Terran_Factory, 1, true))
			.create());
		
		this.add(new GenericPartFactory(UnitType.Terran_Science_Facility)
			.add(new UnitRequirement(UnitType.Terran_Barracks, vesselSupport?2:6))
			.add(new UnitRequirement(UnitType.Terran_Engineering_Bay, 2))
			.add(new UnitRequirement(UnitType.Terran_Starport, 1, true))
			.create());
		
		
		
		bot.taskManager.potentialTasks.add(new AssignScience());
		
		super.initialize(game, self, bot);
	}
	
	@Override
	public void onFrame(Game game, Player self, Tyr bot)
	{
		DebugMessages.addMessage("Going bio.");
		
		StopWatch watch = new StopWatch();
		long time;
		watch.start();
		
		super.onFrame(game, self, bot);
		
		time = watch.time();
		if (time > 55)
			DebugMessages.addMessage("Bio super time: " + time); 
		
		if (UnitTracker.count(UnitType.Terran_Engineering_Bay) > 0 && vesselSupport && bot.getAvailableGas() < 400)
			Settings.setWorkersPerGas(3);
		else if (UnitTracker.count(UnitType.Terran_Barracks) > 5 || UnitTracker.count(UnitType.Terran_Engineering_Bay) > 0)
			Settings.setWorkersPerGas(2);
		
		// We only take an expand after the first attack.
		if(!takeExpands && (mode != RUSH || bot.getAvailableMinerals() >= 600) && (AttackTask.getAttackSent() || (mode != ULTRA_DEFENSIVE)))
		{
			if (mode == RUSH)
				Settings.setMaximumWorkers(45);
			this.add(new ExpandPart(true));
			takeExpands = true;
		}
		
		time = watch.time();
		if (time > 55)
			DebugMessages.addMessage("Bio tier 1 time: " + time);
		
		if (bot.getAvailableMinerals()>= 100 
				&& UnitTracker.getGeyserCount() > 0 
				&& UnitTracker.count(UnitType.Terran_Barracks) >= 1
				&& (UnitTracker.count(UnitType.Terran_Refinery) == 0 || UnitTracker.count(UnitType.Terran_Starport) > 0)
				&& (UnitTracker.getCcCount() >= 2 || mode != GREEDY)
				&& (UnitTracker.count(UnitType.Terran_Bunker) > 0 || mode == GREEDY || mode == RUSH)
			)
		{
			bot.spaceManager.build(UnitType.Terran_Refinery);
		}

		time = watch.time();
		if (time > 55)
			DebugMessages.addMessage("Bio refinery: " + time + " geyserCount: " + UnitTracker.getGeyserCount());
		
		if (buildNaturalTurret && bot.getAvailableMinerals() >= 75 && self.completedUnitCount(UnitType.Terran_Engineering_Bay) > 0)
		{
			for(DefensiveStructures structures : bot.defensiveStructures)
			{
				if (structures.defendedPosition.getDistance(Tyr.tileToPosition(self.getStartLocation())) <= 200)
					continue;
				if (structures.getUnitCount(UnitType.Terran_Missile_Turret) == 0)
					bot.spaceManager.buildDefensive(UnitType.Terran_Missile_Turret, structures);
				break;
			}
		}
		
		// Build bunkers, unless we are in greedy mode.
		if (mode != GREEDY && mode != RUSH && bot.getAvailableMinerals() >= 100 && UnitTracker.count(UnitType.Terran_Barracks) >= 1)
		{
			for(DefensiveStructures structures : bot.defensiveStructures)
			{
				int count = structures.getUnitCount(UnitType.Terran_Bunker);

				// We do not build a bunker at our main if there already is one at our natural.
				if (bot.defensiveStructures.size() > 1 
						&& structures.defendedPosition.getDistance(Tyr.tileToPosition(self.getStartLocation())) <= 200)
					continue;
				
				// Ultra defensive mode goes for double bunkers.
				if(count < ((mode == ULTRA_DEFENSIVE)?2:1))
				{
					bot.spaceManager.buildDefensive(UnitType.Terran_Bunker, structures);
		  			break;
				}
			}
		}

		time = watch.time();
		if (time > 55)
			DebugMessages.addMessage("Bio bunker time: " + time);
		
		if(buildTurrets && self.completedUnitCount(UnitType.Terran_Engineering_Bay) >= 1 && bot.getAvailableMinerals() >= 75)
		{
			boolean built = false;
			
			for (int i=0; i<bot.workForce.mineralWorkers.size(); i++)
			{
				MineralWorkers workers = bot.workForce.mineralWorkers.get((i + turretPos)%bot.workForce.mineralWorkers.size());
				if (bot.spaceManager.build(UnitType.Terran_Missile_Turret, null, workers.turretSite))
				{
					built = true;
					turretPos = (i + turretPos + 1)%bot.workForce.mineralWorkers.size();
					break;
				}
			}
			if (!built)
				DebugMessages.addMessage("No placement found for any turret.");
		}
		
		if (bot.getAvailableMinerals() >= 75
				&& self.completedUnitCount(UnitType.Terran_Missile_Turret) >= 4
				&& depotTurrets > 0)
		{
			bot.spaceManager.build(UnitType.Terran_Missile_Turret, null, bot.spaceManager.depotBuildSite);
			depotTurrets--;
		}

		time = watch.time();
		if (time > 55)
			DebugMessages.addMessage("Bio turrets time: " + time);
	}

	@Override
	public boolean overrideStructureOrder(Game game, Player self, Tyr bot, Agent agent)
	{
		if(super.overrideStructureOrder(game, self, bot, agent))
			return true;
		
		if (agent.unit.getType() == UnitType.Terran_Barracks)
		{
			/*
			if (vesselSupport
					&& self.completedUnitCount(UnitType.Terran_Science_Facility) > 0
					&& bot.getAvailableMinerals() <= 300
					&& !self.hasResearched(TechType.Irradiate)
					&& !self.isResearching(TechType.Irradiate))
				return true;
			*/
			
			// In greedy mode, we do not train marines until we have our expand started.
			if(mode == GREEDY && UnitTracker.getCcCount() <= 1)
				return true;
			if (!agent.unit.isTraining() 
					&& Tyr.bot.getAvailableGas() >= 25 && Tyr.bot.getAvailableMinerals() >= 50
					&& self.completedUnitCount(UnitType.Terran_Academy) > 0)
			{
				if (UnitTracker.count(UnitType.Terran_Medic) * 4 < 
						UnitTracker.count(UnitType.Terran_Marine) - 4 * UnitTracker.count(UnitType.Terran_Bunker))
				{
					agent.unit.train(UnitType.Terran_Medic);
					return true;
				}
			}
		}
		else if (agent.unit.getType() == UnitType.Terran_Academy)
		{
			if (bot.getAvailableMinerals() >= 100 && bot.getAvailableGas() >= 100 
					&& !agent.unit.isResearching() && !self.hasResearched(TechType.Stim_Packs))
			{
				agent.unit.research(TechType.Stim_Packs);
				return true;
			}
			else if (bot.getAvailableGas() >= 150 && bot.getAvailableMinerals() >= 150
					&& !agent.unit.isResearching())
			{
				agent.unit.upgrade(UpgradeType.U_238_Shells);
			}
		}
		else if (agent.unit.getType() == UnitType.Terran_Starport)
		{
			if (vesselSupport
					&& agent.unit.getAddon() == null 
					&& bot.getAvailableMinerals() >= 50 && bot.getAvailableGas() >= 50)
				agent.unit.buildAddon(UnitType.Terran_Control_Tower);
			if (vesselSupport 
					&& !agent.unit.isTraining()
					&& bot.getAvailableMinerals() >= 100 
					&& bot.getAvailableGas() >= 225)
			{
				agent.unit.train(UnitType.Terran_Science_Vessel);
			}
			return true;
		}
		else if (agent.unit.getType() == UnitType.Terran_Factory)
		{
			if (!agent.unit.isTraining() && bot.getAvailableMinerals() >= 100 && Tyr.game.enemy().getRace() == Race.Protoss)
				agent.unit.train(UnitType.Terran_Vulture);
			return true;
		}
		else if (agent.unit.getType() == UnitType.Terran_Science_Facility)
		{
			if (vesselSupport 
					&& !self.hasResearched(TechType.Irradiate)
					&& bot.getAvailableMinerals() >= 200 
					&& bot.getAvailableGas() >= 200)
			{
				agent.unit.research(TechType.Irradiate);
				return true;
			}
			return false;
		}
		
		return false;
	}
}
