/*
* Tyr is an AI for StarCraft: Broodwar, 
* 
* Please visit https://github.com/SimonPrins/Tyr for further information.
* 
* Copyright 2015 Simon Prins
*
* This file is part of Tyr.
* Tyr is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 2 of the License, or
* (at your option) any later version.
* Tyr is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
* You should have received a copy of the GNU General Public License
* along with Tyr.  If not, see http://www.gnu.org/licenses/.
*/


package com.tyr.buildingplacement;

import java.util.ArrayList;

import com.tyr.BWTAProxy;
import com.tyr.DebugMessages;
import com.tyr.Tyr;
import com.tyr.agents.Agent;
import com.tyr.unitgroups.PatchWorkers;

import bwapi.TilePosition;
import bwapi.Unit;
import bwapi.UnitType;
import bwta.Region;

/**
 * Build site locator for finding a placement for a supply depot or other building of the same size.
 * @author Simon
 *
 */
public class DepotBuildSite extends BuildSiteLocator 
{
	/**
	 * A list of potential places where we may want to build supply depots.
	 */
	ArrayList<TilePosition> potentialDepots = new ArrayList<TilePosition>();
	
	/**
	 * x position on the map where we are in determining potential supply depot placements.
	 */
	private int x = 0;
	
	public void onFrame()
	{
		// See if we have already gone over the entire map.
		if (x >= Tyr.game.mapWidth())
			return;
		
		// If BWTA hasn't been initialized yet we cannot determine the depot placements.
		if (!BWTAProxy.initialized)
			return;
		
		// Avoid crash when no workers are available.
		if (Tyr.bot.workForce.mineralWorkers.size() == 0)
			return;
		
		// We get a worker which will be passed to the methods determining whether a building placements is allowed.
		// This worker will not actually be sent to construct the supplydepot though.
		Unit worker = null;
		try
		{
			if (Tyr.bot.workForce.mineralWorkers.get(0).units.size() > 0)
				worker = Tyr.bot.workForce.mineralWorkers.get(0).units.get(0).unit;
			for (PatchWorkers patch : Tyr.bot.workForce.mineralWorkers.get(0).patchWorkers)
			{
				if (worker != null)
					break;
				if (patch.units.size() > 0)
					worker = patch.units.get(0).unit;
			}
		}
		catch(NullPointerException e)
		{
			return;
		}
		
		TilePosition start = Tyr.self.getStartLocation();
		if (start == null)
		{
			DebugMessages.addMessage("Startlocation is null!");
			return;
		}
		
		Region startRegion = BWTAProxy.getRegion(start);
		if (startRegion == null)
		{
			DebugMessages.addMessage("Startregion is null!");
			return;
		}
		
		// Go over the map at from top to bottom at the x position.
		for (int y = 0; y <Tyr.game.mapHeight(); y += 2)
		{
			TilePosition pos = new TilePosition(x, y);
			
			Region depotRegion = BWTAProxy.getRegion(pos);
			
			if (depotRegion == null)
				continue;
			
			// We only place the depot in our starting region.
			if (!depotRegion.equals(startRegion))
				continue;
			
			// Determine whether we can actually build a supply depot here.
			if (!Tyr.bot.spaceManager.canBuildHere(worker, pos, UnitType.Terran_Supply_Depot, true))
				continue;
			
			// Add the depot if it can be built.
			potentialDepots.add(pos);
		}
		
		// Increment the x value for the next iteration which will take place the next frame.
		x+=3;
	}
	
	
	@Override
	public TilePosition findPlacement(UnitType building, TilePosition preferredTile, Agent worker)
	{
		if (building != UnitType.Terran_Supply_Depot 
				&& building != UnitType.Terran_Armory 
				&& building != UnitType.Terran_Academy)
			return null;
		
		if (!BWTAProxy.initialized)
			return null;
		
		if (potentialDepots.size() == 0)
			return null;
		
		// If the start location is not visible, that means the opponent destroyed our main base. 
		// In this case a position for the depot is determined through the regular building placement method.
		// This means the supply depot will be built at one of our remaining expands.
		if (!Tyr.game.isVisible(Tyr.game.self().getStartLocation()))
			return null;
		
		int bestDist = 0;
		TilePosition result = null;
		
		TilePosition mainExit = Tyr.positionToTile(SpaceManager.getMainExit());
		
		// Go through the list of placements to determine a suitable one.
		for(TilePosition pos : potentialDepots)
		{
			// If we can't build on this location we try the next location.
			if (!SpaceManager.canBuildHere(worker.unit, pos.getX(), pos.getY(), UnitType.Terran_Supply_Depot))
				continue;
			
			// We try to find the one whose Manhattan distance to the exit is as large as possible.
			// This to prevent blocking paths through our base.
			int newDist = Math.abs(pos.getX() - mainExit.getX()) + Math.abs(pos.getY() - mainExit.getY());
			if (newDist > bestDist)
			{
				bestDist = newDist;
				result = pos;
			}
		}
		
		// Remove the result from the list of candidates and return it.
		if(result != null)
			potentialDepots.remove(result);
		
		return result;
	}
}
