/*
* Tyr is an AI for StarCraft: Broodwar, 
* 
* Please visit https://github.com/SimonPrins/Tyr for further information.
* 
* Copyright 2015 Simon Prins
*
* This file is part of Tyr.
* Tyr is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 2 of the License, or
* (at your option) any later version.
* Tyr is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
* You should have received a copy of the GNU General Public License
* along with Tyr.  If not, see http://www.gnu.org/licenses/.
*/


package com.tyr.buildingplacement;

import com.tyr.Tyr;
import com.tyr.agents.Agent;
import com.tyr.agents.None;
import com.tyr.agents.WorkerAgent;

import bwapi.Position;
import bwapi.TilePosition;
import bwapi.UnitType;

/**
 * An object instance of this class keeps track of a command
 *  for building a building that has been issued.  
 * @author Simon
 *
 */
public class BuildCommand 
{
	/**
	 * The worker who will build the building.
	 */
	public Agent worker;
	
	/**
	 * The type of building we want to build.
	 */
	public UnitType building;
	
	/**
	 * The (top left corner) position where we want to build the building.
	 */
	public TilePosition position;
	
	/**
	 * Is the position fixed, or are we allowed to change it if it turns out to be blocked?
	 */
	public boolean fixed;
	
	/**
	 * This variable is set to true when we remove the BuildCommand.
	 */
	public boolean removing = false;
	
	public BuildCommand(Agent worker, UnitType building, TilePosition position)
	{
		this.worker = worker;
		this.building = building;
		this.position = position;
		
		WorkerAgent wa = (WorkerAgent)worker;
		wa.blockedTimer = getTimer();
	}
	
	public BuildCommand(Agent worker, UnitType building, TilePosition position, boolean fixed)
	{
		this.worker = worker;
		this.building = building;
		this.position = position;
		this.fixed = fixed;
		
		WorkerAgent wa = (WorkerAgent)worker;
		wa.blockedTimer = getTimer();
	}
	
	/**
	 * Gets the amount of minerals required to build the building.
	 * @return The amount of minerals required to build the building.
	 */
	public int mineralCost()
	{
		return building.mineralPrice();
	}
	
	/**
	 * Gets the amount of gas required to build the building.
	 * @return The amount of gas required to build the building.
	 */
	public int gasCost()
	{
		return building.gasPrice();
	}
	
	/**
	 * Removes the build command from the list of commands to be built.
	 */
	public void remove()
	{
		if(removing)
			return;
		
		removing = true;
		
		Tyr.bot.reservedMinerals -= mineralCost();
		Tyr.bot.reservedGas -= gasCost();

		
		if (!worker.isDead())
		{
			if(((WorkerAgent)worker).blockedTimer == Tyr.game.getFrameCount())
				((WorkerAgent)worker).isBlocked = true;
			
			if(((WorkerAgent)worker).isBlocked)
				worker.unit.stop();
			((WorkerAgent)worker).blockedTimer = -1;
			worker.order(new None(worker));
		}
	}
	
	/**
	 * Determines the center of the building that is to be built.
	 * @return The center of the building that is to be built.
	 */
	public Position getCenter()
	{
		Position buildPos = Tyr.tileToPosition(position);
		return new Position(
				buildPos.getX() + building.tileWidth() * 16 - 16,
				buildPos.getY() + building.tileHeight() * 16 - 16);
	}
	
	/**
	 * Determines the amount of time (in frames) the worker is allowed to take to move to the building position and start building.
	 * After that time it is assumed that the worker is unable to reach its destination and a new worker is sent instead.
	 * @return Returns the amount of time a worker is allowed to take to reach the build location.
	 */
	public long getTimer() 
	{
		return Tyr.game.getFrameCount() + 400 + worker.unit.getDistance(Tyr.tileToPosition(position))/3;
	}
}
