/*
* Tyr is an AI for StarCraft: Broodwar, 
* 
* Please visit https://github.com/SimonPrins/Tyr for further information.
* 
* Copyright 2015 Simon Prins
*
* This file is part of Tyr.
* Tyr is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 2 of the License, or
* (at your option) any later version.
* Tyr is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
* You should have received a copy of the GNU General Public License
* along with Tyr.  If not, see http://www.gnu.org/licenses/.
*/


package com.tyr.agents;

import com.tyr.Tyr;

import bwapi.Color;
import bwapi.Game;
import bwapi.Player;
import bwapi.Position;
import bwapi.TechType;
import bwapi.Unit;


/**
 * This is an agent that helps control a vulture.
 * @author Simon
 *
 */
public class VultureAgent extends Agent
{
	/**
	 * The time until the vulture is considered blocked in, or -1 if no timer is active.
	 * This is used when placing a spider mine.
	 * The vluture is given a certain amount of time to place the mine.
	 * If it does not place the mine within this time it is considered blocked in.
	 */
	public long blockedTimer = -1;
	
	/**
	 * When isBlocked is true, the vulture is assumed to be blocked in.
	 */
	public boolean isBlocked = false;

	/**
	 * The number of times the unit was ordered to place a mine.
	 */
	private int placeMineOrders;
	
	public VultureAgent(Unit unit)
	{
		super(unit);
	}
	
	/**
	 * This method is expected to be called every frame.
	 * @param game The game.
	 * @param self The player controlled by our bot.
	 * @param bot The instance of Tyr that is currently playing the game.
	 */
	public void onFrame(Game game, Player self, Tyr bot) 
	{
		// If the time runs out, the vulture is considered blocked.
		if(game.getFrameCount() == blockedTimer)
		{
			blockedTimer = -1;
			isBlocked = true;
		}
		
		if(blockedTimer != -1)
			game.drawTextMap(unit.getX(), unit.getY()+10, (blockedTimer - game.getFrameCount()) + "");
		
		if(isBlocked)
			drawCircle(Color.Red, 6);
	}
	
	/**
	 * Order the vulture to place a mine.
	 * @param target The position where we should place the mine.
	 */
	public void placeMine(Position target)
	{
		placeMineOrders++;
		unit.useTech(TechType.Spider_Mines, target);
	}
	
	/**
	 * Getter for the number of times the agent was ordered to place a mine.
	 * @return The number of times the agent was ordered to place a mine.
	 */
	public int getPlaceMineOrders()
	{
		return placeMineOrders;
	}
}