/*
* Tyr is an AI for StarCraft: Broodwar, 
* 
* Please visit https://github.com/SimonPrins/Tyr for further information.
* 
* Copyright 2015 Simon Prins
*
* This file is part of Tyr.
* Tyr is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 2 of the License, or
* (at your option) any later version.
* Tyr is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
* You should have received a copy of the GNU General Public License
* along with Tyr.  If not, see http://www.gnu.org/licenses/.
*/


package com.tyr.agents;

import com.tyr.EnemyManager;
import com.tyr.Tyr;

import bwapi.Color;
import bwapi.Game;
import bwapi.Order;
import bwapi.Player;
import bwapi.Position;
import bwapi.Unit;
import bwapi.UnitType;
import bwapi.WeaponType;


/**
 * Order a unit to attack toward a target position.
 * @author Simon
 *
 */
public class MedicAttack extends Command
{
	/**
	 * The position where the unit will attack.
	 */
	private Position target;
	
	private Unit defendingUnit;
	
	/**
	 * Order a medic to assist an attack toward a target position.
	 * @param agent The medic who will assist in the attack.
	 * @param target The position where the medic will attack.
	 */
	public MedicAttack(Agent agent, Position target)
	{
		super(agent);
		this.target = target;
	}
	
	@Override
	public void execute(Game game, Player self, Tyr bot) 
	{
		agent.drawCircle(Color.White);
		

		if (target == null || agent.distanceSquared(target) <= 128*128)
			return;
		
		Order order = agent.unit.getOrder(); 
		Position orderTarget = agent.unit.getOrderTargetPosition();
		
		if (agent.unit.isHoldingPosition() || agent.unit.isIdle())
		{
			// If the unit is idle we need to send a new attack.
			Tyr.drawCircle(agent.unit.getPosition(), Color.Green, 6);
			agent.attack(target);
		}
		else 
		{
			// See if there is an enemy closeby.
			boolean enemyClose = false;
			for (Unit unit : EnemyManager.getEnemyUnits())
			{
				if (unit.getType().groundWeapon() == WeaponType.None)
					continue;
				
				if (agent.distanceSquared(unit) <= 128*128)
				{
					enemyClose = true;
					break;
				}
			}
			
			if (!enemyClose)
				defendingUnit = null;
			
			// If there is an enemy close, we need to see if there is an allied unit we can go to.
			if (defendingUnit == null && enemyClose)
			{
				for(Unit unit : self.getUnits())
				{
					if (!unit.getType().isOrganic() || unit.getType() == UnitType.Terran_Medic)
						continue;
					
					if (agent.distanceSquared(unit) <= 256*256)
					{
						defendingUnit = unit;
						agent.unit.follow(defendingUnit);
						break;
					}
				}
			}
			
			// If the attack target changed, we need to send a new attack.
			double dist = target.getDistance(orderTarget);
			if (order == Order.AttackMove 
					&& orderTarget != null
					&& agent.unit.getTarget()== null
					&&
				(Math.abs(target.getX() - orderTarget.getX()) >= 10 
					|| Math.abs(target.getY() - orderTarget.getY()) >= 10
					|| dist >= 10))
			{
				// Attack the target position.
				Tyr.drawCircle(agent.unit.getPosition(), Color.White, 6);
				agent.attack(target);
			}
		}
	}
	
	
	@Override
	public boolean replace(Command command) 
	{
		if (!command.getClass().equals(MedicAttack.class))
			return true;
		
		MedicAttack atkCom = ((MedicAttack)command);
		agent.attack(target);
		
		this.target = atkCom.target;
		
		return false;
	}
	
}
