/*
* Tyr is an AI for StarCraft: Broodwar, 
* 
* Please visit https://github.com/SimonPrins/Tyr for further information.
* 
* Copyright 2015 Simon Prins
*
* This file is part of Tyr.
* Tyr is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 2 of the License, or
* (at your option) any later version.
* Tyr is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
* You should have received a copy of the GNU General Public License
* along with Tyr.  If not, see http://www.gnu.org/licenses/.
*/


package com.tyr.unitgroups;

import java.util.ArrayList;

import com.tyr.DebugMessages;
import com.tyr.EnemyManager;
import com.tyr.ScanEvent;
import com.tyr.Tyr;
import com.tyr.agents.Agent;

import bwapi.Color;
import bwapi.Game;
import bwapi.Player;
import bwapi.Position;
import bwapi.TechType;
import bwapi.Unit;
import bwapi.UnitType;


/**
 * This unit group manages the Comsat Stations to scan for invisible units.
 * @author Simon
 *
 */
public class ComsatNetwork extends UnitGroup
{
	/**
	 * The recorded scans that are currently active.
	 */
	ArrayList<ScanEvent> scanEvents = new ArrayList<ScanEvent>();
	
	/**
	 * This unit group manages the Comsat Stations to scan for invisible units.
	 * @param rejects The OutOfJob unit group to which unwanted units are sent.
	 */
	public ComsatNetwork(OutOfJob rejects) 
	{
		super(rejects);
	}

	@Override
	public void onFrame(Game game, Player self, Tyr bot)
	{
		Position target = null;
		
		// Clear scan events that have ended.
		for(int i=0; i<scanEvents.size(); i++)
		{
			if (game.getFrameCount() - scanEvents.get(i).time >= 15*15)
			{
				scanEvents.remove(i);
				i--;
			}
			else
				Tyr.drawCircle(scanEvents.get(i).pos, Color.Orange, 320);
		}
		Unit scannee = null;
		
		for (Unit enemy : EnemyManager.getEnemyUnits())
			if (!enemy.isDetected())
				Tyr.drawCircle(enemy.getPosition(), Color.Orange);
		
		// See if there are enemy units which we need to scan.
		for(Unit enemy : EnemyManager.getEnemyUnits())
		{
			if (! enemy.isDetected() && enemy.getType() != UnitType.Protoss_Observer)
			{
				boolean alreadyScanned = false;
				for(ScanEvent event : scanEvents)
					if (event.inRange(enemy.getPosition()))
						alreadyScanned = true;
				if(alreadyScanned)
					continue;
				target = enemy.getPosition();
				scannee = enemy;
				break;
			}
		}
		
		// If we find an enemy unit to scan, try to scan it.
		if(target != null)
		{
			if (target.getX() >= game.mapWidth()*32 - 128 && target.getY() >= game.mapWidth()*32 - 128 )
			{
				System.out.println("Error, non existant unit?");
				System.out.println("Unit type: " + scannee.getType());
				System.out.println("Position: (" + target.getX() + ", " + target.getY() + ")");
				DebugMessages.addMessage("Scanner error: " + scannee.getType() + " (" + target.getX() + ", " + target.getY() + ")");
				return;
			}
			
			for(Agent myUnit : units)
			{
				if (myUnit.unit.getEnergy() > 50 && myUnit.unit.isCompleted())
				{
					scanEvents.add(new ScanEvent(target, game.getFrameCount()));
					
					myUnit.unit.useTech(TechType.Scanner_Sweep, target);
					break;
				}
			}
		}
	}
	
	@Override
	public int takeAgent(Agent agent)
	{
		if (agent.unit.getType() == UnitType.Terran_Comsat_Station)
			return 10;
		else
			return -1;
	}
}
