/*
* Tyr is an AI for StarCraft: Broodwar, 
* 
* Please visit https://github.com/SimonPrins/Tyr for further information.
* 
* Copyright 2015 Simon Prins
*
* This file is part of Tyr.
* Tyr is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 2 of the License, or
* (at your option) any later version.
* Tyr is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
* You should have received a copy of the GNU General Public License
* along with Tyr.  If not, see http://www.gnu.org/licenses/.
*/


package com.tyr.tasks;

import com.tyr.DebugMessages;
import com.tyr.EnemyManager;
import com.tyr.PositionUtil;
import com.tyr.Settings;
import com.tyr.Tyr;
import com.tyr.agents.Agent;
import com.tyr.unitgroups.HomeGroup;

import bwapi.Game;
import bwapi.Player;
import bwapi.Unit;
import bwapi.UnitType;
import bwapi.WeaponType;


/**
 * A task that determines whether our main base needs defending and if so will try to defend it.
 */
public class DefendMainTask extends Task
{
	/**
	 * The closest invading enemy unit.
	 */
	private Unit invader;
	
	/**
	 * Do we defend against flyers only?
	 * Otherwise we defend against ground units only.
	 */
	private boolean flyers;
	
	/**
	 * Do we need to defend against a worker rush?
	 */
	public static boolean workerRush = false;
	
	/**
	 * Do we keep zealots back against enemy units?
	 */
	public static boolean keepZealotsBack = false;
	
	/**
	 * Do we defend against air units with everything we have?
	 */
	public static boolean maxFlyerDefense = false;
	
	/**
	 * A task that determines whether our main base needs defending and if so will try to defend it.
	 * @param flyers Do we defend against flyers only?
	 * Otherwise we defend against ground units only.
	 */
	public DefendMainTask(boolean flyers)
	{
		this.flyers = flyers;
	}
	
	@Override
	public boolean isRequired(Game game, Player self, Tyr bot) 
	{
		if (!Settings.getDefendMain())
		{
			if (!flyers)
				DebugMessages.addMessage("Not defending main!");
			return false;
		}
		
		if (!flyers && workerRush)
			return true;
		
		if (flyers)
			invader = EnemyManager.getManager().getFlyingInvader();
		else
			invader = EnemyManager.getManager().getInvader();
		
		if (invader == null && solution != null)
		{
			getSolution().done(bot.homeGroup);
			
			solution = null;
		}
		
		// If we have found an invader, we will need to respond!
		return invader != null;
	}

	@Override
	public void solve(Game game, Player self, Tyr bot)
	{
		if (workerRush && !(solution instanceof DefendWorkerRushSolution))
		{
			((SendTroopsSolution)solution).done(bot.homeGroup);
			solution = new DefendWorkerRushSolution(this, null);
		}
		
		final int required;
		if (flyers && !maxFlyerDefense)
		{
			final int invaders = EnemyManager.getManager().getFlyingInvaderCount(); 
			required = invaders + (invaders == 1 ? 2 : 3);
		}
		else
			required = Integer.MAX_VALUE;
		
		final Unit invader = flyers ? EnemyManager.getManager().getFlyingInvader() : EnemyManager.getManager().getInvader();
		final int invadingDist;
		if (invader != null)
			invadingDist = PositionUtil.distanceSq(Tyr.tileToPosition(self.getStartLocation()), invader);
		else
			invadingDist = 1000000;
		
		// Get all available units to defend.
		for (int i=bot.homeGroup.units.size() - 1; i >= 0; i--)
		{
			if (EnemyManager.getManager().getInvaderCount() == 1 
					&& EnemyManager.getManager().getInvader() != null
					&& EnemyManager.getManager().getInvader().getType().isWorker() 
					&& getSolution().getCount() >= 1)
				break;
			
			if (getSolution().getCount() >= required)
				break;
				
			Agent agent = bot.homeGroup.units.get(i);
			
			if (agent.unit.getType() == UnitType.Protoss_Zealot 
					&& keepZealotsBack
					&& invadingDist >= 400 * 400)
				continue;
			
			if (HomeGroup.keepTanksSieged && agent.isTank())
				continue;
			
			if (agent.unit.getType() != UnitType.Terran_Battlecruiser 
					&& ((flyers && agent.unit.getType().airWeapon() != WeaponType.None))
						|| (!flyers && (agent.unit.getType().groundWeapon() != WeaponType.None || agent.unit.getType() == UnitType.Protoss_Reaver)))
			{
				getSolution().add(agent);
				bot.homeGroup.units.remove(i);
			}
		}
		
		// We set the position where we want the units to defend.
		if (solution != null && invader != null)
			getSolution().setTarget(invader.getPosition());
		
		// We call the super class, which will send the units out to defend.
		super.solve(game, self, bot);
	}

	@Override
	public void findSolution(Game game, Player self, Tyr bot)
	{
		if (invader != null)
			solution = new SendTroopsSolution(this, invader.getPosition());
	}
	
	public TargetableSolution getSolution()
	{
		return (TargetableSolution) solution;
	}
}
