/*
* Tyr is an AI for StarCraft: Broodwar, 
* 
* Please visit https://github.com/SimonPrins/Tyr for further information.
* 
* Copyright 2015 Simon Prins
*
* This file is part of Tyr.
* Tyr is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 2 of the License, or
* (at your option) any later version.
* Tyr is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
* You should have received a copy of the GNU General Public License
* along with Tyr.  If not, see http://www.gnu.org/licenses/.
*/


package com.tyr.builds;

import com.tyr.DebugMessages;
import com.tyr.EnemyManager;
import com.tyr.Settings;
import com.tyr.StopWatch;
import com.tyr.Tyr;
import com.tyr.UnitTracker;
import com.tyr.agents.Agent;
import com.tyr.buildingplacement.DefensiveStructures;
import com.tyr.requirements.UnitRequirement;

import bwapi.Game;
import bwapi.Player;
import bwapi.Race;
import bwapi.Unit;
import bwapi.UnitType;


/**
 * This build is supposed to work well against almost any opponent.
 * @author Simon
 *
 */
public class DefensiveTerran extends CompositeBuildOrder
{
	private Race opponentRace = Race.Unknown;
	private BuildOrder counter;
	@Override
	public void initialize(Game game, Player self, Tyr bot)
	{
		Settings.setRequiredSize(20);
		Settings.setMaximumSize(40);
		
		Settings.setWorkersPerGas(1);
		
		this.add(new SupplyDepotPart());
		this.add(new WorkerScoutPart(1600));
		
		this.add(new GenericPartFactory(UnitType.Terran_Barracks)
			.setCost(100, 0)
			.create());
		
		this.add(new GenericPartFactory(UnitType.Terran_Engineering_Bay)
			.setMax(2)
			.add(new UnitRequirement(UnitType.Terran_Marine, 20))
			.add(new UnitRequirement(UnitType.Terran_Academy))
			.create());
		
		super.initialize(game, self, bot);
	}
	
	@Override
	public void onFrame(Game game, Player self, Tyr bot)
	{
		if (opponentRace == Race.Unknown)
		{
			if (game.enemy().getRace() == Race.Zerg)
				opponentRace = Race.Zerg;
			else if (game.enemy().getRace() == Race.Terran)
				opponentRace = Race.Terran;
			else if (game.enemy().getRace() == Race.Protoss)
				opponentRace = Race.Protoss;
			if (opponentRace == Race.Unknown)
			{
				for (Unit unit : EnemyManager.getEnemyUnits())
				{
					if (unit.getType().getRace() == Race.Zerg)
						opponentRace = Race.Zerg;
					else if (unit.getType().getRace() == Race.Terran)
						opponentRace = Race.Terran;
					else if (unit.getType().getRace() == Race.Protoss)
						opponentRace = Race.Protoss;
					
					if (opponentRace != Race.Unknown)
						break;
				}
			}
			
			if (opponentRace == Race.Zerg)
				counter = new Bio(Bio.DEFENSIVE);
			else if (opponentRace == Race.Terran)
				counter = new TvPGeneric();
			else if (opponentRace == Race.Protoss)
				counter = new TvPGeneric();
			if (opponentRace != Race.Unknown)
				counter.initialize(game, self, bot);
		}
		
		if (opponentRace != Race.Unknown)
		{
			counter.onFrame(game, self, bot);
			return;
		}
		
		DebugMessages.addMessage("TvX build.");
		
		StopWatch watch = new StopWatch();
		long time;
		watch.start();
		
		super.onFrame(game, self, bot);
		
		time = watch.time();
		if (time > 55)
			DebugMessages.addMessage("TvX super time: " + time); 
		
		// Build bunkers.
		if (bot.getAvailableMinerals() >= 100 && UnitTracker.count(UnitType.Terran_Barracks) >= 1)
		{
			for(DefensiveStructures structures : bot.defensiveStructures)
			{
				int count = structures.getUnitCount(UnitType.Terran_Bunker);

				// We do not build a bunker at our main if there already is one at our natural.
				if (bot.defensiveStructures.size() > 1 
						&& structures.defendedPosition.getDistance(Tyr.tileToPosition(self.getStartLocation())) <= 200)
					continue;
				
				if(count < 1)
				{
					bot.spaceManager.buildDefensive(UnitType.Terran_Bunker, structures);
		  			break;
				}
			}
		}

		time = watch.time();
		if (time > 55)
			DebugMessages.addMessage("TvX bunker time: " + time);
	}

	@Override
	public boolean overrideStructureOrder(Game game, Player self, Tyr bot, Agent agent)
	{
		if (opponentRace != Race.Unknown)
			return counter.overrideStructureOrder(game, self, bot, agent);
		
		if(super.overrideStructureOrder(game, self, bot, agent))
			return true;
		
		if (agent.unit.getType()== UnitType.Terran_Barracks
				&& UnitTracker.count(UnitType.Terran_Marine) < 4 * UnitTracker.count(UnitType.Terran_Bunker))
			return false;
		
		return false;
	}
}
