/*
* Tyr is an AI for StarCraft: Broodwar, 
* 
* Please visit https://github.com/SimonPrins/Tyr for further information.
* 
* Copyright 2015 Simon Prins
*
* This file is part of Tyr.
* Tyr is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 2 of the License, or
* (at your option) any later version.
* Tyr is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
* You should have received a copy of the GNU General Public License
* along with Tyr.  If not, see http://www.gnu.org/licenses/.
*/


package com.tyr.builds;

import com.tyr.DebugMessages;
import com.tyr.EnemyManager;
import com.tyr.Settings;
import com.tyr.StopWatch;
import com.tyr.Tyr;
import com.tyr.UnitTracker;
import com.tyr.agents.Agent;
import com.tyr.buildingplacement.DefensiveStructures;
import com.tyr.tasks.AssignScience;
import com.tyr.tasks.AttackTask;
import com.tyr.tasks.LayMinesTask;
import com.tyr.tasks.VultureHarassTask;

import bwapi.Game;
import bwapi.Player;
import bwapi.Race;
import bwapi.TechType;
import bwapi.Unit;
import bwapi.UnitType;
import bwapi.UpgradeType;


/**
 * A defensive mech build.
 * @author Simon
 *
 */
public class DefensiveMech extends CompositeBuildOrder
{
	/**
	 * The mode the build is in. Either NORMAL or EARLY_PUSH.
	 */
	private int mode = NORMAL;
	
	public static final int NORMAL = 0, EARLY_PUSH = 1;
	
	/**
	 * Do we prefer tanks early on?
	 */
	private boolean preferTanks = false;
	
	/**
	 * Will we harass the opponent with vultures?
	 */
	private boolean vultureHarass = true;
	
	/**
	 * Do we lay mines?
	 */
	private boolean layMines = true;
	
	/**
	 * Will the build take expands?
	 */
	public boolean expandingEnabled = false;
	
	/**
	 * Have we reserved minerals and gas to build a science vessel?
	 */
	private boolean reserveScience;
	
	/**
	 * Has the opponent made dts yet?
	 */
	private boolean dtsEncountered;
	
	/**
	 * A defensive mech build.
	 */
	public DefensiveMech()
	{}
	
	/**
	 * A defensive mech build.
	 * @param preferTanks Do we prefer tanks early on?
	 */
	public DefensiveMech(boolean preferTanks)
	{
		this.preferTanks = preferTanks;
	}
	
	/**
	 * A defensive mech build.
	 * @param preferTanks Do we prefer tanks early on?
	 * @param vultureHarass Will we harass the opponent with vultures?
	 */
	public DefensiveMech(boolean preferTanks, boolean vultureHarass)
	{
		this.preferTanks = preferTanks;
		this.vultureHarass = vultureHarass;
	}
	
	/**
	 * A defensive mech build.
	 * @param mode The mode the build is in. Either NORMAL or EARLY_PUSH.
	 */
	public DefensiveMech(int mode)
	{
		this.mode = mode;
	}
	
	/**
	 * A defensive mech build.
	 * @param preferTanks Do we prefer tanks early on?
	 * @param vultureHarass Will we harass the opponent with vultures?
	 * @param layMines Do we lay mines?
	 */
	public DefensiveMech(boolean preferTanks, boolean vultureHarass, boolean layMines)
	{
		this.preferTanks = preferTanks;
		this.vultureHarass = vultureHarass;
		this.layMines = layMines;
	}
	
	/**
	 * A defensive mech build.
	 * @param mode The mode the build is in. Either NORMAL or EARLY_PUSH.
	 * @param preferTanks Do we prefer tanks early on?
	 * @param vultureHarass Will we harass the opponent with vultures?
	 * @param layMines Do we lay mines?
	 */
	public DefensiveMech(int mode, boolean preferTanks, boolean vultureHarass, boolean layMines)
	{
		this.mode = mode;
		this.preferTanks = preferTanks;
		this.vultureHarass = vultureHarass;
		this.layMines = layMines;
	}

	@Override
	public void initialize(Game game, Player self, Tyr bot)
	{
		Settings.setRequiredSize(mode==EARLY_PUSH?15:40);
		Settings.setMaximumSize(60);
		
		Settings.setWorkersPerGas(3);
		ExpandPart.maximumCcs = 2;
		
		
		this.add(new SupplyDepotPart());
		this.add(new WorkerScoutPart(1600));
		if (mode != EARLY_PUSH)
		{
			this.add(new ExpandPart());
			expandingEnabled = true;
		}

		if (layMines)
			bot.taskManager.potentialTasks.add(new LayMinesTask());
		if (vultureHarass)
			bot.taskManager.potentialTasks.add(new VultureHarassTask(true));
		VultureHarassTask.defendingUnitsNeeded = 20;
		
		bot.taskManager.potentialTasks.add(new AssignScience());
		
		super.initialize(game, self, bot);
	}
	
	@Override
	public void onFrame(Game game, Player self, Tyr bot)
	{
		DebugMessages.addMessage("Going defensive mech.");
		
		StopWatch watch = new StopWatch();
		long time;
		watch.start();
		
		// If we are floating minerals, start taking expands.
		// This will also allow the build to  build more structures, giving a chance to spend the resources better.
		if (!expandingEnabled && bot.getAvailableMinerals() >= 800)
		{
			this.add(new ExpandPart());
			expandingEnabled = true;
		}
		
		if (ExpandPart.maximumCcs == 2 
				&& UnitTracker.count(UnitType.Terran_Factory) >= 5 
				&& (bot.homeGroup.units.size() >= 20 || AttackTask.getAttackSent()))
			ExpandPart.maximumCcs = 100;
		
		super.onFrame(game, self, bot);
		
		if (UnitTracker.count(UnitType.Terran_Factory)  < 1 && (bot.getAvailableGas() >= 100 || bot.getAvailableMinerals() < 100))
			Settings.setWorkersPerGas(1);
		else if ( (UnitTracker.count(UnitType.Terran_Factory)  <= 2 || bot.getAvailableGas() >= 300 )  
				&& (bot.getAvailableGas() >= 100 || bot.getAvailableMinerals() < 150))
			Settings.setWorkersPerGas(2);
		else
			Settings.setWorkersPerGas(3);
		
		time = watch.time();
		if (time > 55)
			DebugMessages.addMessage("Mech super time: " + time);
		
		if (!dtsEncountered)
			for (Unit enemy : EnemyManager.getEnemyUnits())
				if (enemy.getType() == UnitType.Protoss_Dark_Templar)
					dtsEncountered = true;
		
		//if we've the resources to build a barracks ...
		if(bot.getAvailableMinerals() >= 100 
				&& UnitTracker.count(UnitType.Terran_Refinery) > 0
				&& (bot.getAvailableMinerals() >= 150 || UnitTracker.getCcCount() > 1)
				&& (UnitTracker.count(UnitType.Terran_Barracks) == 0)) 
		{
			bot.spaceManager.build(UnitType.Terran_Barracks);
		}
		
		if(UnitTracker.count(UnitType.Terran_Refinery) < UnitTracker.count(UnitType.Terran_Command_Center)
				&& (UnitTracker.count(UnitType.Terran_Refinery) == 0 || UnitTracker.count(UnitType.Terran_Factory) >= 1)
				&& bot.getAvailableMinerals() >= 100 
				&& UnitTracker.getGeyserCount() > 0)
		{
			bot.spaceManager.build(UnitType.Terran_Refinery);
		}
		
		if(bot.getAvailableMinerals() >= 200 && bot.getAvailableGas() >= 100
				&& UnitTracker.count(UnitType.Terran_Barracks) >= 1 
				&& UnitTracker.count(UnitType.Terran_Factory) < Math.min(9, (mode==EARLY_PUSH?0:-1) + 3*UnitTracker.count(UnitType.Terran_Command_Center))
				&& (UnitTracker.count(UnitType.Terran_Factory) == 0
						|| bot.getAvailableMinerals() >= 350 && bot.getAvailableGas() >= 200
					)
			)
		{
			bot.spaceManager.build(UnitType.Terran_Factory);
		}

		time = watch.time();
		if (time > 55)
			DebugMessages.addMessage("Mech tier 1 time: " + time);
				time = watch.time();
		
		if (bot.getAvailableMinerals() >= 150 
				&& UnitTracker.count(UnitType.Terran_Barracks) != 0 && UnitTracker.count(UnitType.Terran_Refinery) != 0
				&& UnitTracker.count(UnitType.Terran_Academy) == 0
				&& (bot.homeGroup.units.size() >= 10 || UnitTracker.count(UnitType.Terran_Vulture) >= 6))
		{
			bot.spaceManager.build(UnitType.Terran_Academy);
		}
		
		if (UnitTracker.count(UnitType.Terran_Siege_Tank_Siege_Mode) + UnitTracker.count(UnitType.Terran_Siege_Tank_Tank_Mode) >= 4
				&& UnitTracker.count(UnitType.Terran_Armory) < 2
				&& bot.getAvailableGas() >= 50
				&& bot.getAvailableMinerals() >= 100)
		{
			bot.spaceManager.build(UnitType.Terran_Armory);
		}
		
		if (UnitTracker.count(UnitType.Terran_Armory) >= 2
				&& self.completedUnitCount(UnitType.Terran_Factory) >= 4
				&& UnitTracker.count(UnitType.Terran_Starport) == 0
				&& bot.getAvailableGas() >= 100
				&& bot.getAvailableMinerals() >= 150)
		{
			bot.spaceManager.build(UnitType.Terran_Starport);
		}
		
		if (UnitTracker.count(UnitType.Terran_Armory) >= 2
				&& UnitTracker.count(UnitType.Terran_Science_Facility) < 1
				&& self.completedUnitCount(UnitType.Terran_Starport) > 0
				&& bot.getAvailableGas() >= 150
				&& bot.getAvailableMinerals() >= 100)
		{
			bot.spaceManager.build(UnitType.Terran_Science_Facility);
		}
		
		time = watch.time();
		if (time > 55)
			DebugMessages.addMessage("Mech tier 2 time: " + time);
		
		
		// Build bunkers.
		if (bot.getAvailableMinerals() >= 50 
				&& (bot.getAvailableMinerals() >= 100 ||  UnitTracker.count(UnitType.Terran_Factory) > 1)
				&& (UnitTracker.count(UnitType.Terran_Factory) >= 1) 
				&& self.completedUnitCount(UnitType.Terran_Barracks) >= 1
				&& UnitTracker.count(UnitType.Terran_Bunker) < 2)
		{
			for(DefensiveStructures structures : bot.defensiveStructures)
			{
				int count = structures.getUnitCount(UnitType.Terran_Bunker);
				
				if(count < 1)
				{
					// We do not build a bunker at our main if there already is one at our natural.
					if (bot.defensiveStructures.size() > 1 && structures.defendedPosition.getDistance(Tyr.tileToPosition(self.getStartLocation())) <= 200)
					{
						if (UnitTracker.count(UnitType.Terran_Bunker) == 0)
							continue;
						else
							break;
					}
					
					bot.spaceManager.buildDefensive(UnitType.Terran_Bunker, structures);
		  			break;
				}
			}
		}

		time = watch.time();
		if (time > 55)
			DebugMessages.addMessage("Mech bunker time: " + time);
	}

	@Override
	public boolean overrideStructureOrder(Game game, Player self, Tyr bot, Agent agent)
	{
		if(super.overrideStructureOrder(game, self, bot, agent))
			return true;
		
		if(agent.unit.getType() == UnitType.Terran_Barracks)
		{
			if (UnitTracker.count(UnitType.Terran_Factory) == 0 || UnitTracker.count(UnitType.Terran_Marine) >= Math.max(4, UnitTracker.count(UnitType.Terran_Bunker)*4))
				return true;
		}
		
		if(agent.unit.getType() == UnitType.Terran_Starport)
		{
			if (agent.unit.isTraining())
				return true;
			
			if (Tyr.game.enemy().getRace() != Race.Protoss)
				return true;
			
			if (agent.unit.getAddon() == null
					&& bot.getAvailableGas() >= 50
					&& bot.getAvailableMinerals() >= 50
					&& UnitTracker.count(UnitType.Terran_Science_Facility) > 0)
				agent.unit.buildAddon(UnitType.Terran_Control_Tower);
			else if (UnitTracker.count(UnitType.Terran_Science_Vessel) < 2
					&& self.completedUnitCount(UnitType.Terran_Science_Facility) > 0)
			{
				if (bot.getAvailableMinerals() >= 100
						&& bot.getAvailableGas() >= 225
						&& !reserveScience)
					agent.unit.train(UnitType.Terran_Science_Vessel);
				else if (!reserveScience && dtsEncountered)
				{
					reserveScience = true;
					bot.reservedMinerals += 100;
					bot.reservedGas += 225;
				}
				else if (reserveScience
						&& self.minerals() >= 100
						&& self.gas() >= 225)
				{
					agent.unit.train(UnitType.Terran_Science_Vessel);
					reserveScience = false;
					bot.reservedMinerals -= 100;
					bot.reservedGas -= 225;
				}
			}
			return true;
		}
		
		if (agent.unit.getType() == UnitType.Terran_Factory)
		{
			if (agent.unit.getAddon() != null)
			{
				if (UnitTracker.count(UnitType.Terran_Factory) <= 1 
						&& UnitTracker.count(UnitType.Terran_Siege_Tank_Siege_Mode) + UnitTracker.count(UnitType.Terran_Siege_Tank_Tank_Mode) >= 2)
					return false;
				return !self.isResearching(TechType.Tank_Siege_Mode) && !self.hasResearched(TechType.Tank_Siege_Mode);
			}

			boolean addonWanted = UnitTracker.usableAddOns(UnitType.Terran_Machine_Shop)*4 + 1 < self.completedUnitCount(UnitType.Terran_Factory)
					|| self.completedUnitCount(UnitType.Terran_Factory) <= 1;
			
			if(UnitTracker.count(UnitType.Terran_Armory) >= 1 && !agent.unit.isTraining()
					&& UnitTracker.count(UnitType.Terran_Siege_Tank_Siege_Mode) + UnitTracker.count(UnitType.Terran_Siege_Tank_Tank_Mode) 
						>= 5 + (UnitTracker.count(UnitType.Terran_Goliath)/2) 
					&& bot.getAvailableMinerals() >= 100 && bot.getAvailableGas() >= 50)
			{
				agent.unit.train(UnitType.Terran_Goliath);
				return true;
			}
			else if (bot.getAvailableMinerals() >= 75 
					&& !agent.unit.isTraining() 
					&& UnitTracker.count(UnitType.Terran_Vulture) <= 2*(UnitTracker.count(UnitType.Terran_Siege_Tank_Siege_Mode) + UnitTracker.count(UnitType.Terran_Siege_Tank_Tank_Mode)) - 1
					&& (!preferTanks || UnitTracker.count(UnitType.Terran_Siege_Tank_Siege_Mode) + UnitTracker.count(UnitType.Terran_Siege_Tank_Tank_Mode) >= 2
						|| bot.getAvailableMinerals() >= 300))
			{
				agent.unit.train(UnitType.Terran_Vulture);
				return true;
			}
			
			return !addonWanted;
		}
		else if(agent.unit.getType() == UnitType.Terran_Armory)
		{
			if(bot.getAvailableMinerals() >= UpgradeType.Terran_Vehicle_Weapons.mineralPrice()
					&& bot.getAvailableGas() >= UpgradeType.Terran_Vehicle_Weapons.gasPrice())
				agent.unit.upgrade(UpgradeType.Terran_Vehicle_Weapons);
			
			if(bot.getAvailableMinerals() >= UpgradeType.Terran_Vehicle_Plating.mineralPrice()
					&& bot.getAvailableGas() >= UpgradeType.Terran_Vehicle_Plating.gasPrice())
				agent.unit.upgrade(UpgradeType.Terran_Vehicle_Plating);
			
			return true;
		}
		else if (agent.unit.getType() == UnitType.Terran_Engineering_Bay)
		{
			return true;
		}
		else if (agent.unit.getType() == UnitType.Terran_Machine_Shop)
		{
			if (!agent.unit.isResearching() && self.hasResearched(TechType.Tank_Siege_Mode)
					&& !self.hasResearched(TechType.Spider_Mines)
					&& bot.getAvailableMinerals() >= 100
					&& bot.getAvailableGas() >= 100)
				agent.unit.research(TechType.Spider_Mines);
			else if (!agent.unit.isResearching() && self.hasResearched(TechType.Spider_Mines)
					&& bot.getAvailableMinerals() >= 100
					&& bot.getAvailableGas() >= 100)
				agent.unit.upgrade(UpgradeType.Charon_Boosters);
		}
		return false;
	}
}
