/*
* Tyr is an AI for StarCraft: Broodwar, 
* 
* Please visit https://github.com/SimonPrins/Tyr for further information.
* 
* Copyright 2015 Simon Prins
*
* This file is part of Tyr.
* Tyr is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 2 of the License, or
* (at your option) any later version.
* Tyr is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
* You should have received a copy of the GNU General Public License
* along with Tyr.  If not, see http://www.gnu.org/licenses/.
*/


package com.tyr.agents;

import java.util.List;

import com.tyr.BWTAProxy;
import com.tyr.EnemyManager;
import com.tyr.Tyr;

import bwapi.Color;
import bwapi.Game;
import bwapi.Order;
import bwapi.Player;
import bwapi.Position;
import bwapi.Unit;
import bwapi.UnitType;


/**
 * Order the agent to defend a location.
 * @author Simon
 *
 */
public class Defend extends Command
{
	/**
	 * The position the agent will defend.
	 */
	private Position target;
	
	/**
	 * A unit we will defend against.
	 */
	private Unit killTarget;
	
	/**
	 * Order the agent to defend a location.
	 * @param agent The agent to which we give this order.
	 * @param target The position the agent will defend.
	 */
	public Defend(Agent agent, Position target)
	{
		super(agent);
		this.target = target;
	}
	
	@Override
	public void execute(Game game, Player self, Tyr bot) 
	{
		agent.drawCircle(Color.Green);
		
		// If the defending agent is a tank...
		if (agent.unit.getType() == UnitType.Terran_Siege_Tank_Siege_Mode || agent.unit.getType() == UnitType.Terran_Siege_Tank_Tank_Mode )
		{
			// See if there is a unit within attack range.
			int radius = UnitType.Terran_Siege_Tank_Siege_Mode.groundWeapon().maxRange();
			boolean found = false;
			for(Unit unit : EnemyManager.getEnemyUnits())
			{
				if (unit.getType().isFlyer() || unit.isLifted())
					continue;
				if (agent.distanceSquared(unit) <= radius * radius)
				{
					found = true;
					break;
				}
			}
		
			if (agent.unit.getType() == UnitType.Terran_Siege_Tank_Siege_Mode)
			{
				// If there is no unit in attack range, we need to unsiege.
				if(!found)
				{
					agent.unit.unsiege();
					return;
				}
			}
			else if (agent.unit.getType() == UnitType.Terran_Siege_Tank_Tank_Mode)
			{
				// If there is a unit in attack range, we need to siege.
				if(found)
				{
					agent.unit.siege();
					return;
				}
			}
		}
		
		// Hold position so that we do not move beyond the walloff, making the walloff pointless.
		if(bot.wallOff != null && bot.wallOff.wall.size() > 1
				&& BWTAProxy.initialized
				&& BWTAProxy.getRegion(EnemyManager.getManager().getInvader().getPosition()) != BWTAProxy.getRegion(self.getStartLocation())
				&& agent.distanceSquared(target) <= 256*256
				&& agent.unit.getOrder() != Order.HoldPosition)
		{
			agent.unit.holdPosition();
		}
		
    	if (killTarget == null)
    	{
    		// See if we can find a unit to kill.
    		List<Unit> inRange = game.getUnitsInRadius(agent.unit.getPosition(), agent.unit.getType().groundWeapon().maxRange());
    		
    		for(Unit unit : inRange)
    		{
    			if (unit.getPlayer().isEnemy(self) && unit.getType() != UnitType.Protoss_Observer)
    			{
    				killTarget = unit;
    				break;
    			}
    		}
    	}
		
		if (killTarget != null)
		{
			boolean targetFlies = killTarget.getType().isFlyer() || killTarget.isLifted();
			int maxRangeSq = targetFlies?agent.unit.getType().airWeapon().maxRange(): agent.unit.getType().groundWeapon().maxRange();
			maxRangeSq = maxRangeSq * maxRangeSq;
			
			if (agent.distanceSquared(killTarget) >= maxRangeSq)
				killTarget = null;
			else
			{
				game.drawLineMap(agent.unit.getX(), agent.unit.getY(), killTarget.getX(), killTarget.getY(), Color.Red);
				
				int cooldown = targetFlies?agent.unit.getAirWeaponCooldown():agent.unit.getGroundWeaponCooldown();
			
				if (cooldown == 0)
				{
					// Attack the target.
					if (agent.unit.getOrder() != Order.AttackUnit)
						agent.unit.attack(killTarget);
				}
				else
				{
					// Move back when your weapon is on cooldown.
					Position fleeTarget = agent.retreatTarget(killTarget.getPosition(), 32);
					agent.unit.move(fleeTarget);
				}
				
				return;
			}
		}
		
		if (EnemyManager.getManager().getInvader() != null)
		{
			// Move in the direction of the invader.
			Position orderTarget = agent.unit.getOrderTargetPosition();
			if (agent.unit.isIdle()
					|| (killTarget != null &&(Math.abs(orderTarget.getX() - target.getX()) >= 128 || Math.abs(orderTarget.getY() - target.getY()) >= 128))
					|| (killTarget != null && (agent.unit.getGroundWeaponCooldown() == 1 || agent.unit.getAirWeaponCooldown() == 1)))
				
				agent.unit.attack(EnemyManager.getManager().getInvader().getPosition());
		}
		else
		{
			// If there is no invader, move in the direction of the position we want to defend.
			if ((agent.unit.getOrder() == Order.Move || agent.unit.isIdle()))
				agent.unit.attack(target);
		}
	}

	@Override
	public boolean replace(Command command) 
	{
		if (!command.getClass().equals(Defend.class))
			return true;
		
		return ((Defend)command).target.getX() == target.getX() && ((Defend)command).target.getY() == target.getY();
	}
}
