//////////////////////////////////////////////////////////////////////////
//
// This file is part of Iron's source files.
// Iron is free software, licensed under the MIT/X11 License. 
// A copy of the license is provided with the library in the LICENSE file.
// Copyright (c) 2016, Igor Dimitrijevic
//
//////////////////////////////////////////////////////////////////////////


#ifndef MY_H
#define MY_H

#include <BWAPI.h>
#include "bwapiUnits.h"
#include "expert.h"
#include "faceOff.h"
#include "../defs.h"
#include <memory>
#include <array>


namespace iron
{

class Production;
class ConstructingAddonExpert;
class EarlyRunBy;

//////////////////////////////////////////////////////////////////////////////////////////////
//                                                                                          //
//                                  class MyUnit
//                                                                                          //
//////////////////////////////////////////////////////////////////////////////////////////////
//

class MyUnit : public MyBWAPIUnit
{
public:
	static unique_ptr<MyUnit>		Make(BWAPI::Unit u);

									~MyUnit();

	void							Update() override;

	const MyUnit *					IsMyUnit() const override			{ return this; }
	MyUnit *						IsMyUnit() override					{ return this; }

	const vector<FaceOff> &			FaceOffs() const					{CI(this); return m_FaceOffs; }

	void							OnDangerHere();
	frame_t							LastDangerReport() const			{CI(this); return m_lastDangerReport; }

	bool							AllowedToChase() const;
	void							SetNoChaseFor(frame_t time);

	bool							StayInArea() const					{CI(this); return m_stayInArea; }
	void							SetStayInArea()						{CI(this); m_stayInArea = true; }
	void							UnsetStayInArea()					{CI(this); m_stayInArea = false; }

	virtual bool					WorthBeingRepaired() const			{CI(this); return true; }

protected:
									MyUnit(BWAPI::Unit u, unique_ptr<IBehavior> pBehavior);

private:
	vector<FaceOff>					m_FaceOffs;
	frame_t							m_lastDangerReport = 0;
	frame_t							m_noChaseUntil = 0;
	bool							m_stayInArea = false;
};



//////////////////////////////////////////////////////////////////////////////////////////////
//                                                                                          //
//                                  class MyBuilding
//                                                                                          //
//////////////////////////////////////////////////////////////////////////////////////////////
//

class MyBuilding : public MyBWAPIUnit
{
public:
	static unique_ptr<MyBuilding>		Make(BWAPI::Unit u);

	using BWAPIUnit::JustLifted;
	using BWAPIUnit::JustLanded;

										~MyBuilding();

	void								Update() override;

	const MyBuilding *					IsMyBuilding() const override			{ return this; }
	MyBuilding *						IsMyBuilding() override					{ return this; }

	TilePosition						Size() const							{CI(this); return m_size; }

	void								Train(BWAPI::UnitType type, bool noCheck = false);
	void								Research(TechType type, bool noCheck = false);
	void								Research(UpgradeType type, bool noCheck = false);
	void								BuildAddon(BWAPI::UnitType type, bool noCheck = false);
	void								Lift(bool noCheck = false);
	bool								Land(TilePosition location, bool noCheck = false);
	void								CancelConstruction(bool noCheck = false);
	void								CancelTrain(bool noCheck = false);
	void								CancelResearch(bool noCheck = false);

	void								RecordTrainingExperts(Production & Prod);
	void								RecordResearchingExperts(Production & Prod);

	virtual vector<ConstructingAddonExpert *>	ConstructingAddonExperts() { return {}; }
	virtual ConstructingAddonExpert *	ConstructingThisAddonExpert() { return nullptr; }

	const vector<unique_ptr<ResearchingExpert>> &	ResearchingExperts() const	{CI(this); return m_ResearchingExperts; }

	void								OnOtherBWAPIUnitDestroyed_v(BWAPIUnit * pOther) override;

	int									IdlePosition() const;

	// Returns n in [0..10] that tells how often a unit was trained in this building the last 2000 frames.
	// Training is checked every 100 frames.
	int									Activity() const;


protected:
										MyBuilding(BWAPI::Unit u, unique_ptr<IBehavior> pBehavior);

	template<tid_t tid>
	void								AddTrainingExpert()				{CI(this); m_TrainingExperts.push_back(make_unique<ExpertInTraining<tid>>(this)); }

	template<TechTypes::Enum::Enum val>
	void								AddResearchingExpert()			{CI(this); m_ResearchingExperts.push_back(make_unique<ExpertInResearching<val>>(this)); }

	template<UpgradeTypes::Enum::Enum val>
	void								AddUpgragingExpert()			{CI(this); m_ResearchingExperts.push_back(make_unique<ExpertInUpgraging<val>>(this)); }

	bool								DefaultBehaviorOnFrame_common();

	double								MinLifePercentageToRepair() const override	{ return 0.35; }
	double								MaxLifePercentageToRepair() const override	{ return 0.66; }

private:
	const TilePosition						m_size;
	vector<unique_ptr<TrainingExpert>>		m_TrainingExperts;
	vector<unique_ptr<ResearchingExpert>>	m_ResearchingExperts;
	uint_least32_t							m_activity = 0;
};




template<tid_t> class My;



} // namespace iron


#endif

