#include "GameState.h"
#include "InformationManager.h"
#include "CombatSimulatorBasic.h"
#include "CombatSimulatorDPS.h"

#define IDLE_TIME 400 //in frames
#define NOTHING_TIME 24*60*30 //in frames (30 minutes)

using namespace BWAPI;

// player = true  = friendly player
// player = false = enemy player

GameState::GameState()
	:_buildingTypes(NoneBuilding),
	_time(0),
	cs(new CombatSimulatorBasic)
{
	army.friendly.clear();
	army.enemy.clear();
}

GameState::GameState(CombatSimulator* combatSim)
	:_buildingTypes(NoneBuilding),
	_time(0),
	cs(combatSim)
{
	army.friendly.clear();
	army.enemy.clear();
}

GameState::~GameState()
{
	cleanArmyData();
	delete cs;
}

void GameState::cleanArmyData()
{
	for (auto unitGroup : army.friendly) delete unitGroup;
	army.friendly.clear();
	for (auto unitGroup : army.enemy) delete unitGroup;
	army.enemy.clear();
}

void GameState::changeCombatSimulator(CombatSimulator* newCS)
{
	delete cs;
	cs = newCS;
}

GameState::GameState(const GameState &gameState)
	:_buildingTypes(gameState._buildingTypes),
	regionsInCombat(gameState.regionsInCombat),
	combats(gameState.combats),
	_time(gameState._time)
{
	for (auto unitGroup : gameState.army.friendly) army.friendly.push_back(new unitGroup_t(*unitGroup));
	for (auto unitGroup : gameState.army.enemy) army.enemy.push_back(new unitGroup_t(*unitGroup));
	cs = gameState.cs->clone();
	updateCombatList();
}

const GameState& GameState::operator=(const GameState& gameState)
{
	this->_buildingTypes = gameState._buildingTypes;
	for (auto unitGroup : gameState.army.friendly) this->army.friendly.push_back(new unitGroup_t(*unitGroup));
	for (auto unitGroup : gameState.army.enemy) this->army.enemy.push_back(new unitGroup_t(*unitGroup));
	this->regionsInCombat = gameState.regionsInCombat;
	this->combats = gameState.combats;
	this->_time = gameState._time;
	this->cs = gameState.cs->clone();
	this->updateCombatList();

	return *this;
}

void GameState::loadIniConfig()
{
    std::string iniBuilding = LoadConfigString("high_level_search", "buildings", "RESOURCE_DEPOT");
    if (iniBuilding == "NONE") {
        _buildingTypes = NoneBuilding;
    } else if (iniBuilding == "ALL") {
        _buildingTypes = AllBuildings;
    } else {
        _buildingTypes = ResourceDepot;
    }
}

void GameState::importCurrentGameState()
{
	cleanArmyData();
	_time = Broodwar->getFrameCount();
	unitGroupVector* armySide;

	for (auto player : Broodwar->getPlayers()) {
		if (player == Broodwar->self()) armySide = &army.friendly;
		else if (Broodwar->self()->isEnemy(player)) armySide = &army.enemy;
		else continue; // skip neutral player

		for (auto unit : player->getUnits()) {
			// ignore buildings (except _buildingTypes)
			if ((_buildingTypes == ResourceDepot) && unit->getType().isBuilding() && !unit->getType().isResourceDepot() ) continue;
			// ignore units training
			if (!unit->isCompleted()) continue;
            // ignore spells
            if (unit->getType().isSpell()) continue;

			uint8_t regionId = informationManager->_regionIdMap[unit->getTilePosition().x][unit->getTilePosition().y];
			uint8_t targetRegionId = 0;
			Position targetPosition = unit->getTargetPosition();
			if (targetPosition != Positions::None) {
				TilePosition tile = (TilePosition)targetPosition;
				targetRegionId = informationManager->_regionIdMap[tile.x][tile.y];
			}

			// TODO: warning, possible collision with BWAPI::Orders::Nothing
			uint8_t orderId = abstractOrder::Nothing;
			if (!unit->getType().isBuilding()) {
				orderId = getAbstractOrderID(unit->getOrder().getID(), regionId, targetRegionId);
			}

			unitGroup_t* unitGroup = new unitGroup_t(unit->getType().getID(), 1, regionId, orderId, targetRegionId);

			addUnitToArmySide(unitGroup, *armySide);
		}
	}

	calculateExpectedEndFrameForAllGroups();
}

// returns the unitGroup_t* where the unit was added
unsigned short GameState::addUnitToArmySide(unitGroup_t* unit, unitGroupVector& armySide)
{
	// simplify unitTypeId
	if (unit->unitTypeId == UnitTypes::Terran_Siege_Tank_Siege_Mode) {
		unit->unitTypeId = UnitTypes::Terran_Siege_Tank_Tank_Mode;
	}

	// check for wrong unit
	UnitType unitType(unit->unitTypeId);
	if (unitType.isSpell()) DEBUG("Trying to add a spell");

	// search if unit is already in the vector
	for (std::vector<int>::size_type i = 0; i < armySide.size(); ++i) {
		if (unit->unitTypeId == armySide[i]->unitTypeId &&
			unit->regionId == armySide[i]->regionId &&
			unit->orderId == armySide[i]->orderId &&
			unit->targetRegionId == armySide[i]->targetRegionId) {
			armySide[i]->numUnits++;
			delete unit;
			return i;
		}
	}
	// if not, add it to the end
	armySide.push_back(unit);
	return armySide.size() - 1;
}

// assumes army.enmey is empty
void GameState::addAllEnemyUnits()
{
	_time = Broodwar->getFrameCount();
	Position targetPosition;

	for (auto unit : Broodwar->enemy()->getUnits()) {
		// ignore buildings (except _buildingTypes)
        if ((_buildingTypes == ResourceDepot) && unit->getType().isBuilding() && !unit->getType().isResourceDepot() ) continue;
		// ignore units training
		if (!unit->isCompleted()) continue;
		if (unit->getType().isWorker()) continue;
        // ignore spells
        if (unit->getType().isSpell()) continue;

		uint8_t regionId = informationManager->_regionIdMap[unit->getTilePosition().x][unit->getTilePosition().y];
		uint8_t orderId = abstractOrder::Unknown;
		if (unit->getType().isBuilding()) orderId = abstractOrder::Nothing;
		unitGroup_t* unitGroup = new unitGroup_t(unit->getType().getID(), 1, regionId, orderId, 0);

		addUnitToArmySide(unitGroup, army.enemy);
	}
}

void GameState::addSelfBuildings()
{
	for (auto unit : Broodwar->self()->getUnits()) {
        if ( !unit->getType().isBuilding() ) continue;
        if ( _buildingTypes == ResourceDepot && !unit->getType().isResourceDepot() ) continue;

		uint8_t regionId = informationManager->_regionIdMap[unit->getTilePosition().x][unit->getTilePosition().y];
		unitGroup_t* unitGroup = new unitGroup_t(unit->getType().getID(), 1, regionId, abstractOrder::Nothing, 0);

		addUnitToArmySide(unitGroup, army.friendly);
    }
}

unsigned short GameState::addFriendlyUnit(Unit unit)
{
		uint8_t regionId = informationManager->_regionIdMap[unit->getTilePosition().x][unit->getTilePosition().y];
		unitGroup_t* unitGroup = new unitGroup_t(unit->getType().getID(), 1, regionId, abstractOrder::Unknown, 0);

		return addUnitToArmySide(unitGroup, army.friendly);
}

void GameState::addGroup(int unitId, int numUnits, int regionId, int listID, int orderId, int targetRegion, int endFrame )
{
	UnitType unitType(unitId);
	if (unitType.isAddon() || unitType.isSpell()) return;

	if (unitType.isBuilding()) orderId = abstractOrder::Nothing;
	unitGroup_t* unitGroup = new unitGroup_t(unitId, numUnits, regionId, orderId, targetRegion, endFrame);

    if (listID == 1) {
		army.friendly.push_back(unitGroup);
    } else if (listID == 2) {
		army.enemy.push_back(unitGroup);
    }
}

void GameState::addUnit(int unitId, int regionId, int listID, int orderId)
{
	UnitType unitType(unitId);
	if (unitType.isAddon() || unitType.isSpell()) return;

	if (unitType.isBuilding()) orderId = abstractOrder::Nothing;
	unitGroup_t* unitGroup = new unitGroup_t(unitId, 1, regionId, orderId, 0);

	if (listID == 1) {
		addUnitToArmySide(unitGroup, army.friendly);
	} else if (listID == 2) {
		addUnitToArmySide(unitGroup, army.enemy);
	}
}



// Calculate the expected end frame for each order
void GameState::calculateExpectedEndFrameForAllGroups()
{
	combats.clear();
	regionsInCombat.clear();

	unitGroupVector* armySide;
	int timeToMove;

	for (int i=0; i<2; i++) {
		if (i == 0) armySide = &army.friendly;
		else armySide = &army.enemy;

		for (unsigned int j = 0; j < armySide->size(); j++) {
			armySide->at(j)->endFrame = _time + IDLE_TIME;
			if (armySide->at(j)->orderId == abstractOrder::Unknown) {
				armySide->at(j)->endFrame = _time + IDLE_TIME;
			} else if (armySide->at(j)->orderId == abstractOrder::Move) {
				timeToMove = getMoveTime(armySide->at(j)->unitTypeId, armySide->at(j)->regionId, armySide->at(j)->targetRegionId);
				armySide->at(j)->endFrame = _time + timeToMove;
			} else if (armySide->at(j)->orderId == abstractOrder::Attack) {
				// keep track of the regions with units in attack state
				regionsInCombat.insert(armySide->at(j)->regionId);
			} else if (armySide->at(j)->orderId == abstractOrder::Nothing) { // it's a building
				armySide->at(j)->endFrame = _time + NOTHING_TIME;
			}
		}
	}
	// WARNING some units aren't attacking in the same region!!!
    if (!regionsInCombat.empty()) {
		calculateCombatTimes(); // update combatList too
	}
}

int GameState::getMoveTime(int unitTypeId, int regionId, int targetRegionId)
{
	if (regionId == targetRegionId) return 0;

	UnitType unitType(unitTypeId);
	double speed = unitType.topSpeed(); //non-upgraded top speed in pixels per frame
    int distance = getRegionDistance(regionId, targetRegionId); // distance in pixels

	int timeToMove = (int)((double)distance / speed);
	if (timeToMove == INT_MAX || timeToMove < 0) {
		LOG("Wrong move time: " << timeToMove << " traveling from " << regionId << " to " << targetRegionId);
		LOG(toString());
	}
	return timeToMove;

// 	return  (int)((double)distance/speed);
}

// TODO move this to BWTA
BWAPI::Position GameState::getCenterRegionId(int regionId)
{
    // TODO use informationManager->_chokepointIdOffset
	BWTA::Region* region = informationManager->_regionFromID[regionId];
	if (region != NULL) {
		return region->getCenter();
	} else {
		BWTA::Chokepoint* cp = informationManager->_chokePointFromID[regionId];
		if (cp != NULL) {
			return cp->getCenter();
		} else {
			return BWAPI::Positions::None;
		}
	}
}

// TODO move this to BWTA
int GameState::getRegionDistance(int regId1, int regId2)
{
    if (!informationManager->_onlyRegions) {
        if (regId1 < informationManager->_chokepointIdOffset) {
            // regId1 is a region, and regId2 a chokepoint
            return informationManager->_distanceBetweenRegAndChoke[regId1][regId2];
        } else {
            // regId1 is a chokepoint, and regId2 a region
            return informationManager->_distanceBetweenRegAndChoke[regId2][regId1];
        }
    } else {
        //LOG("Distance from " << regId1 << " to " << regId2 << " = " << informationManager->_distanceBetweenRegions[regId1][regId2]);
        return informationManager->_distanceBetweenRegions[regId1][regId2];
    }

}

int GameState::getAbstractOrderID(int orderId, int currentRegion, int targetRegion)
{
 	if ( orderId == BWAPI::Orders::MoveToMinerals ||
		 orderId == BWAPI::Orders::WaitForMinerals ||
		 orderId == BWAPI::Orders::MiningMinerals ||
		 orderId == BWAPI::Orders::Harvest3 ||
		 orderId == BWAPI::Orders::Harvest4 ||
 		 orderId == BWAPI::Orders::ReturnMinerals )
		//return abstractOrder::Mineral;
		return abstractOrder::Unknown;
 	else if ( orderId == BWAPI::Orders::MoveToGas ||
			  orderId == BWAPI::Orders::Harvest1 ||
			  orderId == BWAPI::Orders::Harvest2 ||
			  orderId == BWAPI::Orders::WaitForGas ||
			  orderId == BWAPI::Orders::HarvestGas ||
 			  orderId == BWAPI::Orders::ReturnGas )
 		//return abstractOrder::Gas;
		return abstractOrder::Unknown;
	else if ( orderId == BWAPI::Orders::Move ||
			  orderId == BWAPI::Orders::Follow ||
			  orderId == BWAPI::Orders::ComputerReturn ) 
		return abstractOrder::Move;
	else if ( orderId == BWAPI::Orders::AttackUnit ||
			  orderId == BWAPI::Orders::AttackMove ||
			  orderId == BWAPI::Orders::AttackTile)
		  if (currentRegion == targetRegion) return abstractOrder::Attack;
		  else return abstractOrder::Move;
	else if ( orderId == BWAPI::Orders::Repair || 
			  orderId == BWAPI::Orders::MedicHeal )
			return abstractOrder::Heal;
    else if ( orderId == BWAPI::Orders::Nothing )
			return abstractOrder::Nothing;
    else {
        //LOG("Not detected action " << BWAPI::Order(orderId).getName() << " id: " << orderId);
		return abstractOrder::Unknown;
    }
}

const std::string GameState::getAbstractOrderName(BWAPI::Order order, int currentRegion, int targetRegion)
{
	int abstractId = getAbstractOrderID(order.getID(), currentRegion, targetRegion);
	return abstractOrder::name[abstractId];
}

const std::string GameState::getAbstractOrderName(int abstractId)
{
	return abstractOrder::name[abstractId];
}

std::string GameState::toString()
{
    std::stringstream tmp;
	tmp << "GameState: time " << _time << " ====================================\n";
	tmp << "TypeID \tNumber \tRegion \tOrder \tTarget \tEnd frame \tUnit name\n";
	for (unsigned int i=0; i < army.friendly.size(); i++) {
		tmp << intToString(army.friendly[i]->unitTypeId) << "\t";
		tmp << intToString(army.friendly[i]->numUnits) << "\t";
		tmp << intToString(army.friendly[i]->regionId) << "\t";
		tmp << getAbstractOrderName(army.friendly[i]->orderId) << "\t";
		tmp << intToString(army.friendly[i]->targetRegionId) << "\t";
		tmp << intToString(army.friendly[i]->endFrame - _time) << "\t";
		tmp << BWAPI::UnitType(army.friendly[i]->unitTypeId).getName() << "\n";
	}
	tmp << "(up)FRIENDS---------------ENEMIES(down)\n";
	for (unsigned int i = 0; i < army.enemy.size(); i++) {
		tmp << intToString(army.enemy[i]->unitTypeId) << "\t";
		tmp << intToString(army.enemy[i]->numUnits) << "\t";
		tmp << intToString(army.enemy[i]->regionId) << "\t";
		tmp << getAbstractOrderName(army.enemy[i]->orderId) << "\t";
		tmp << intToString(army.enemy[i]->targetRegionId) << "\t";
		tmp << intToString(army.enemy[i]->endFrame - _time) << "\t";
		tmp << BWAPI::UnitType(army.enemy[i]->unitTypeId).getName() << "\n";
	}

    return tmp.str();
}

void GameState::execute(playerActions_t playerActions, bool player)
{
	unitGroupVector* armySide;
	if (player) armySide = &army.friendly;
	else armySide = &army.enemy;

	uint8_t orderId;
	uint8_t targetRegionId;
	unitGroup_t* unitGroup;
	int timeToMove;

	for (unsigned int i=0; i < playerActions.size(); i++) {
		orderId = playerActions[i].action.orderID;
		targetRegionId = playerActions[i].action.targetRegion;
		unitGroup = (*armySide)[playerActions[i].pos];

#ifdef DEBUG_ORDERS
		bool printError = false;
		if (playerActions[i].isFriendly != player) { DEBUG("Picking the wrong army"); printError = true; }
		if (playerActions[i].unitTypeId != unitGroup->unitTypeId) { DEBUG("UnitTypeID mismatch"); printError = true; }
		if (playerActions[i].regionId != unitGroup->regionId) { DEBUG("regionID mismatch"); printError = true; }
		if (printError) {
			std::string tmp = "";
			tmp += "[" + intToString(playerActions[i].pos) + "]" + BWAPI::UnitType(playerActions[i].unitTypeId).getName() + " (" + intToString(playerActions[i].regionId) + "): ";
			tmp += abstractOrder::name[orderId] + " (" + intToString(targetRegionId) + ")\n";
			LOG(tmp);
			LOG("State before trying to update the actions to perform");
			LOG(toString());
		}
#endif
		unitGroup->orderId = orderId;
		unitGroup->targetRegionId = targetRegionId;

		if (orderId == abstractOrder::Move) {
			timeToMove = getMoveTime(unitGroup->unitTypeId, unitGroup->regionId, targetRegionId);
			unitGroup->endFrame = _time + timeToMove;
		} else if (orderId == abstractOrder::Attack) {
			// keep track of the regions with units in attack state
			regionsInCombat.insert(unitGroup->regionId);

			updateCombatList(); // TODO can we optimize this?
			updateCombatLengthAtRegion(unitGroup->regionId);
		} else { // Unknown or Idle
			unitGroup->endFrame = _time + IDLE_TIME;
		}
	}
}

void GameState::updateCombatList() 
{
	combats.clear();
	// add all units to the combat list since some units can attack units that they aren't in attack state
	unitGroupVector* armySide;
	for (int i = 0; i < 2; i++) {
		if (i == 0) armySide = &army.friendly;
		else armySide = &army.enemy;
		for (unitGroupVector::size_type j = 0; j < armySide->size(); j++) {
			unitGroup_t* unitState = armySide->at(j);
			if (i == 0) combats[unitState->regionId].friendly.push_back(unitState);
			else combats[unitState->regionId].enemy.push_back(unitState);
		}
	}
}

void GameState::calculateCombatTimes() 
{
	updateCombatList();

	// calculate combatTime for each region in combat
	for (auto regionID : regionsInCombat) {
		updateCombatLengthAtRegion(regionID, true);
	}

	if (!regionsToDelete.empty()) {
		for (auto regionID : regionsToDelete) {
			combats.erase(regionID);
			regionsInCombat.erase(regionID);
		}
		regionsToDelete.clear();
	}
}

void GameState::updateCombatLengthAtRegion(int regionID, bool lateDelete)
{
	// if one of the armies is empty combat is impossible
	if (combats[regionID].friendly.empty() || combats[regionID].enemy.empty()) {
		if (combats[regionID].friendly.empty()) setAttackOrderToIdle(combats[regionID].enemy);
		else setAttackOrderToIdle(combats[regionID].friendly);

		if (lateDelete) {
			regionsToDelete.push_back(regionID);
		} else {
			combats.erase(regionID);
			regionsInCombat.erase(regionID);
		}
		return;
	}

	// if combatLenght is INT_MAX combat is impossible
	int combatLenght = cs->getCombatLength(&combats[regionID]);
	if (combatLenght == INT_MAX) {
		setAttackOrderToIdle(combats[regionID].friendly);
		setAttackOrderToIdle(combats[regionID].enemy);

		if (lateDelete) {
			regionsToDelete.push_back(regionID);
		} else {
			combats.erase(regionID);
			regionsInCombat.erase(regionID);
		}
		return;
	}

	int combatEnd = combatLenght + _time;
	if (combatEnd == INT_MAX || combatEnd < 0) {
		LOG("Wrong combat time: " << combatEnd << " while updating combats at region " << regionID);
		LOG(toString());
	}

	// update end combat time for each unit attacking in this region
	for (auto unitGroup : combats[regionID].friendly) {
		if (unitGroup->orderId == abstractOrder::Attack) unitGroup->endFrame = combatEnd;
	}
	for (auto unitGroup : combats[regionID].enemy) {
		if (unitGroup->orderId == abstractOrder::Attack) unitGroup->endFrame = combatEnd;
	}
}

void GameState::setAttackOrderToIdle(unitGroupVector &army)
{
	for (auto unitGroup : army) {
		if (unitGroup->orderId == abstractOrder::Attack) {
			unitGroup->orderId = abstractOrder::Idle;
			unitGroup->endFrame = 0;
		}
	}
}

bool GameState::canExecuteAnyAction(bool player) 
{
	unitGroupVector* armySide;
	if (player) armySide = &army.friendly;
	else armySide = &army.enemy;

	for (unitGroupVector::size_type i = 0; i < armySide->size(); ++i) {
		if ((*armySide)[i]->endFrame <= _time) {
			return true;
		}
	}

    return false;
}

// Move forward the game state until the first unit finish its action (can delete unitGroup_t*)
void GameState::moveForward(int forwardTime)
{
// 	bool unlimitedTime = false;
	unitGroupVector* armySide;

    if (forwardTime == 0) {
// 		unlimitedTime = true;
	    forwardTime = INT_MAX;
	    // search the first action to finish
		for (int i = 0; i < 2; i++) {
			if (i == 0) armySide = &army.friendly;
			else armySide = &army.enemy;
			for (unitGroupVector::size_type j = 0; j < armySide->size(); ++j) {
				forwardTime = (std::min)(forwardTime, armySide->at(j)->endFrame);
			}
		}
    }

	// forward the time of the game state to the first action to finish
	int timeStep = forwardTime - _time;
	_time = forwardTime;

	// update finished actions
	std::set<int> simulateCombatIn; // set because we don't want duplicates
	std::set<int> simulateCombatStepIn; // set because we don't want duplicates
	for (int i=0; i<2; i++) {
		if (i == 0) armySide = &army.friendly;
		else armySide = &army.enemy;
		for (unitGroupVector::size_type j = 0; j < armySide->size(); ++j) {
			if ((*armySide)[j]->endFrame <= _time) {
				if ((*armySide)[j]->orderId == abstractOrder::Move) {
					// if we are moving to/from a region in combat we need to forward the combat time
					if (regionsInCombat.find((int)(*armySide)[j]->regionId) != regionsInCombat.end() ||
						regionsInCombat.find((int)(*armySide)[j]->targetRegionId) != regionsInCombat.end()) {
						simulateCombatStepIn.insert((*armySide)[j]->regionId);
					}
					// we are only updating the position (no "merge group" operation)
					(*armySide)[j]->regionId = (*armySide)[j]->targetRegionId;
					(*armySide)[j]->orderId = abstractOrder::Idle;
				} else if ((*armySide)[j]->orderId == abstractOrder::Attack) {
					// keep track of the regions with units in attack state
					simulateCombatIn.insert((*armySide)[j]->regionId);
					(*armySide)[j]->orderId = abstractOrder::Idle;
				}
			}
		}
	}

	// simulate FINISHED combat in regions
    if (!simulateCombatIn.empty()) {
		for (auto regionId : simulateCombatIn) {
			cs->simulateCombat(&combats[regionId], &army);
			combats.erase(regionId); // remove region from combatList
			regionsInCombat.erase(regionId); // remove region from regionsInCombat
		}

		// sometimes a group was killed by a "moving" group, therefore we need to
		// move forward again if none of the players can move
		if (!canExecuteAnyAction(true) && !canExecuteAnyAction(false)) {
			moveForward();
		}
	}

	// simulate NOT FINISHED combat in regions (because one unit left/come to the region)
	if (!simulateCombatStepIn.empty()) {
		updateCombatList();
		for (auto regionId : simulateCombatStepIn) {
			cs->simulateCombat(&combats[regionId], &army, timeStep);
			
			updateCombatList();
			updateCombatLengthAtRegion(regionId);
		}
	}
}


void GameState::mergeGroups() {
	for (unitGroupVector::iterator it = army.friendly.begin(); it != army.friendly.end(); ++it) {
		for (unitGroupVector::iterator it2 = army.friendly.begin(); it2 != army.friendly.end();) {
            if ( it != it2 &&
				(*it)->unitTypeId == (*it2)->unitTypeId &&
				(*it)->regionId == (*it2)->regionId) {
				(*it)->numUnits += (*it2)->numUnits;
				delete *it2;
				it2 = army.friendly.erase(it2);
            } else {
                ++it2;
            }
        }
    }
}

int GameState::winner() {
	if (army.friendly.empty()) return 0;
	if (army.enemy.empty()) return 1;
	return -1;
}

bool GameState::gameover() {
	if (army.friendly.empty() || army.enemy.empty()) {
        return true;
    }
    // only buildings is also "gameover" (nothing to do)
// 	if (hasOnlyBuildings(army.friendly) && hasOnlyBuildings(army.enemy)) {
//         return true;
// 	}

    return false;
}

bool GameState::hasOnlyBuildings(unitGroupVector armySide) {
	for (unitGroupVector::size_type i = 0; i < armySide.size(); i++) {
		BWAPI::UnitType unitType(armySide[i]->unitTypeId);
        if (!unitType.isBuilding()) return false;
	}
    return true;
}

GameState GameState::cloneIssue(playerActions_t playerActions, bool player) {
	GameState nextGameState = *this;
	nextGameState.execute(playerActions, player);
	nextGameState.moveForward();
	return nextGameState;
}

void GameState::resetFriendlyActions()
{
	for (unitGroupVector::size_type i = 0; i < army.friendly.size(); i++) {
        // skip buildings
		BWAPI::UnitType unitType(army.friendly[i]->unitTypeId);
        if (unitType.isBuilding()) continue;

		army.friendly[i]->endFrame = _time;
	}
}

void GameState::compareAllUnits(GameState gs, int &misplacedUnits, int &totalUnits)
{
	compareUnits(army.friendly, gs.army.friendly, misplacedUnits, totalUnits);
	compareUnits(army.enemy, gs.army.enemy, misplacedUnits, totalUnits);
}

void GameState::compareFriendlyUnits(GameState gs, int &misplacedUnits, int &totalUnits)
{
	compareUnits(army.friendly, gs.army.friendly, misplacedUnits, totalUnits);
}

void GameState::compareEnemyUnits(GameState gs, int &misplacedUnits, int &totalUnits)
{
	compareUnits(army.enemy, gs.army.enemy, misplacedUnits, totalUnits);
}

// TODO we are not considering units of the same type and region but with different orders!!
void GameState::compareUnits(unitGroupVector units1, unitGroupVector units2, int &misplacedUnits, int &totalUnits)
{
	bool match;
	for (auto groupUnit1 : units1) {
		match = false;
		for (auto groupUnit2 : units2) {
			if (groupUnit1->unitTypeId == groupUnit2->unitTypeId && groupUnit1->regionId == groupUnit2->regionId) {
				misplacedUnits += abs(groupUnit1->numUnits - groupUnit2->numUnits);
				totalUnits += std::max(groupUnit1->numUnits, groupUnit2->numUnits);
				match = true;
				break; // we find it, stop looking for a match
			}
		}
		if (!match) {
			misplacedUnits += groupUnit1->numUnits;
			totalUnits += groupUnit1->numUnits;
		}
	}

	for (auto groupUnit2 : units2) {
		match = false;
		for (auto groupUnit1 : units1) {
			if (groupUnit1->unitTypeId == groupUnit2->unitTypeId && groupUnit1->regionId == groupUnit2->regionId) {
				match = true;
				break;
			}
		}
		if (!match) {
			misplacedUnits += groupUnit2->numUnits;
			totalUnits += groupUnit2->numUnits;
		}
	}
}

// TODO we are not considering units of the same type and region but with different orders!!
void GameState::compareUnits2(unitGroupVector units1, unitGroupVector units2, int &intersectionUnits, int &unionUnits)
{
	bool match;
	for (auto groupUnit1 : units1) {
		match = false;
		for (auto groupUnit2 : units2) {
			if (groupUnit1->unitTypeId == groupUnit2->unitTypeId && groupUnit1->regionId == groupUnit2->regionId) {
				intersectionUnits += std::min(groupUnit1->numUnits, groupUnit2->numUnits);
				unionUnits += std::max(groupUnit1->numUnits, groupUnit2->numUnits);
				match = true;
				break; // we find it, stop looking for a match
			}
		}
		if (!match) {
			unionUnits += groupUnit1->numUnits;
		}
	}

	for (auto groupUnit2 : units2) {
		match = false;
		for (auto groupUnit1 : units1) {
			if (groupUnit1->unitTypeId == groupUnit2->unitTypeId && groupUnit1->regionId == groupUnit2->regionId) {
				match = true;
				break;
			}
		}
		if (!match) {
			unionUnits += groupUnit2->numUnits;
		}
	}
}

// returns the Jaccard index comparing all the units from the current game state against other game state
float GameState::getJaccard(GameState gs) {
	int intersectionUnits = 0;
	int unionUnits = 0;
	compareUnits2(army.friendly, gs.army.friendly, intersectionUnits, unionUnits);
	compareUnits2(army.enemy, gs.army.enemy, intersectionUnits, unionUnits);
	return (float(intersectionUnits) / float(unionUnits));
}

// returns the Jaccard index using a quality measure
// given a initialState we compare the "quality" of the current state and the other finalState
float GameState::getJaccard2(GameState initialState, GameState finalState, bool useKillScore)
{
	const double K = 1.0f;

	double intersectionSum = 0;
	double unionSum = 0;

	compareUnitsWithWeight(initialState.army.friendly, K, useKillScore, army.friendly, finalState.army.friendly, intersectionSum, unionSum);
	compareUnitsWithWeight(initialState.army.enemy, K, useKillScore, army.enemy, finalState.army.enemy, intersectionSum, unionSum);

	return float(intersectionSum / unionSum);
}

void GameState::compareUnitsWithWeight(unitGroupVector unitsWeight, const double K, bool useKillScore,
	unitGroupVector units1, unitGroupVector units2, double &intersectionUnits, double &unionUnits)
{
	// get weights from the initialState
	// weight = killscore * (1 / (n + k)) 

	std::map<uint8_t, double> weights;
	double score = 0.0;
	for (auto groupUnit : unitsWeight) {
		score = 1 / ((double)groupUnit->numUnits + K);
		weights.insert(std::pair<uint8_t, double>(groupUnit->unitTypeId, score));
		if (useKillScore) {
			BWAPI::UnitType unitType(groupUnit->unitTypeId);
			weights[groupUnit->unitTypeId] *= unitType.destroyScore();
		}
	}

	bool match;
	for (auto groupUnit1 : units1) {
		match = false;
		for (auto groupUnit2 : units2) {
			if (groupUnit1->unitTypeId == groupUnit2->unitTypeId && groupUnit1->regionId == groupUnit2->regionId) {
				intersectionUnits += (double)std::min(groupUnit1->numUnits, groupUnit2->numUnits) * weights[groupUnit1->unitTypeId];
				unionUnits += (double)std::max(groupUnit1->numUnits, groupUnit2->numUnits) * weights[groupUnit1->unitTypeId];
				match = true;
				break; // we find it, stop looking for a match
			}
		}
		if (!match) {
			unionUnits += groupUnit1->numUnits;
		}
	}

	for (auto groupUnit2 : units2) {
		match = false;
		for (auto groupUnit1 : units1) {
			if (groupUnit1->unitTypeId == groupUnit2->unitTypeId && groupUnit1->regionId == groupUnit2->regionId) {
				match = true;
				break;
			}
		}
		if (!match) {
			unionUnits += (double)groupUnit2->numUnits  * weights[groupUnit2->unitTypeId];
		}
	}
}