#include "CombatSimulatorDPS.h"
#include "InformationManager.h"

CombatSimulatorDPS::CombatSimulatorDPS()
	:_unitTypeDPF(nullptr)
{

}

CombatSimulatorDPS::CombatSimulatorDPS(dpfStats* unitTypeDPF)
	:_unitTypeDPF(unitTypeDPF)
{

}

void CombatSimulatorDPS::simulateCombat(GameState::army_t* armyInCombat, GameState::army_t* army, int frames)
{
	if (!canSimulate(armyInCombat, army)) return;

	int regionInCombat = (*armyInCombat->friendly.begin())->regionId;

	// the order to kill units is sequential from the set (i.e. arbitrary)
	UnitStateVector::iterator friendToKill = armyInCombat->friendly.begin();
	BWAPI::UnitType friendType((*friendToKill)->unitTypeId);
	int friendHP = friendType.maxShields() + friendType.maxHitPoints();

	UnitStateVector::iterator enemyToKill = armyInCombat->enemy.begin();
	BWAPI::UnitType enemyType((*enemyToKill)->unitTypeId);
	int enemyHP = enemyType.maxShields() + enemyType.maxHitPoints();

	double enemyDPF, friendDPF;
	bool armyDestroyed = false;
	double timeToKillFriend = DBL_MAX;
	double timeToKillEnemy = DBL_MAX;

	while (!armyDestroyed) { // we end the loop when one army is empty or we cannot attack each other
		// calculate time to kill one unit of the target group
		timeToKillEnemy = getTimeToKillUnit(armyInCombat->friendly, enemyType, enemyHP, friendDPF);
		// check if we can attack the enemy
		while (timeToKillEnemy == DBL_MAX && enemyToKill != armyInCombat->enemy.end()) {
			++enemyToKill;
			if (enemyToKill == armyInCombat->enemy.end()) break; // if it was the last unit, end the loop
			enemyType = BWAPI::UnitType((*enemyToKill)->unitTypeId);
			enemyHP = enemyType.maxShields() + enemyType.maxHitPoints();
			timeToKillEnemy = getTimeToKillUnit(armyInCombat->friendly, enemyType, enemyHP, friendDPF);
		}

		timeToKillFriend = getTimeToKillUnit(armyInCombat->enemy, friendType, friendHP, enemyDPF);
		while (timeToKillFriend == DBL_MAX && friendToKill != armyInCombat->friendly.end()) {
			++friendToKill;
			if (friendToKill == armyInCombat->friendly.end()) break; // if it was the last unit, end the loop
			friendType = BWAPI::UnitType((*friendToKill)->unitTypeId);
			friendHP = friendType.maxShields() + friendType.maxHitPoints();
			timeToKillFriend = getTimeToKillUnit(armyInCombat->enemy, friendType, friendHP, enemyDPF);
		}

		if (timeToKillEnemy == DBL_MAX && timeToKillFriend == DBL_MAX) {
			LOG("Stopping combat because we cannot kill each other");
			break;
		} else if (timeToKillEnemy < timeToKillFriend) { // friend won
			armyDestroyed = killUnit(enemyToKill, armyInCombat->enemy, &army->enemy, friendHP, enemyHP, timeToKillEnemy, enemyDPF);
		} else { // enemy won (enemy won too when the time to kill of both is exactly the same)
			armyDestroyed = killUnit(friendToKill, armyInCombat->friendly, &army->friendly, enemyHP, friendHP, timeToKillFriend, friendDPF);
		}
	}
}

// return true if we need to end the combat (army empty)
bool CombatSimulatorDPS::killUnit(UnitStateVector::iterator &unitToKill, UnitStateVector &unitsInCombat, UnitStateVector* unitsList, int &HPsurvivor, int &HPkilled, double timeToKill, double DPF)
{
	bool isArmyDestroyed = false;
	// remove unit killed
	(*unitToKill)->numUnits = (*unitToKill)->numUnits - 1;
	if ((*unitToKill)->numUnits == 0) { // if last unit in group
		// delete from general units list
		removeUnit(*unitToKill, unitsList);
		// delete and set new target (sequential from army vector)
		unitToKill = unitsInCombat.erase(unitToKill);
		isArmyDestroyed = (unitToKill == unitsInCombat.end());
	}
	if (!isArmyDestroyed) {
		// reset HP of killed
		BWAPI::UnitType killedType((*unitToKill)->unitTypeId);
		HPkilled = killedType.maxShields() + killedType.maxHitPoints();
	}
	// reduce HP of survivor
	HPsurvivor -= (int)(timeToKill * DPF);
// 	if (HPsurvivor <= 0) DEBUG("Problem!!! both units are killed");
	return isArmyDestroyed;
}

double CombatSimulatorDPS::getTimeToKillUnit(const UnitStateVector &unitsInCombat, BWAPI::UnitType enemyType, int enemyHP, double &DPF)
{
	DPF = 0.0;
// 	int minTime = DBL_MAX; // the minimum time to kill a unit is at least the minimum cooldown
	
	if (_unitTypeDPF == nullptr) {
		// TODO spell casters have special DPS!!!!
		double finalDamage = 0.0; // damage considering (type of damage vs size of target)
		for (auto unitGroup : unitsInCombat) {
			BWAPI::UnitType unitType(unitGroup->unitTypeId);
			if (!enemyType.isFlyer() && unitType.groundWeapon().damageAmount() > 0) {
// 				minTime = std::min(minTime, unitType.groundWeapon().damageCooldown());
				finalDamage = (double)unitType.groundWeapon().damageAmount();
				if (unitType.groundWeapon().damageType() == BWAPI::DamageTypes::Concussive) {
					if (enemyType.size() == BWAPI::UnitSizeTypes::Large) finalDamage *= 0.25;
					else if (enemyType.size() == BWAPI::UnitSizeTypes::Medium) finalDamage *= 0.5;
				} else if (unitType.groundWeapon().damageType() == BWAPI::DamageTypes::Explosive) {
					if (enemyType.size() == BWAPI::UnitSizeTypes::Small) finalDamage *= 0.25;
					else if (enemyType.size() == BWAPI::UnitSizeTypes::Medium) finalDamage *= 0.5;
				}
				// In the case of Firebats and Zealots, the damage returned by BWAPI is not right, since they have two weapons:
				if (unitType == BWAPI::UnitTypes::Terran_Firebat || unitType == BWAPI::UnitTypes::Protoss_Zealot)
					finalDamage *= 2;
				DPF += unitGroup->numUnits * (finalDamage / unitType.groundWeapon().damageCooldown());
			} else if (enemyType.isFlyer() && unitType.airWeapon().damageAmount() > 0) {
// 				minTime = std::min(minTime, unitType.airWeapon().damageCooldown());
				finalDamage = (double)unitType.airWeapon().damageAmount();
				if (unitType.airWeapon().damageType() == BWAPI::DamageTypes::Concussive) {
					if (enemyType.size() == BWAPI::UnitSizeTypes::Large) finalDamage *= 0.25;
					else if (enemyType.size() == BWAPI::UnitSizeTypes::Medium) finalDamage *= 0.5;
				} else if (unitType.airWeapon().damageType() == BWAPI::DamageTypes::Explosive) {
					if (enemyType.size() == BWAPI::UnitSizeTypes::Small) finalDamage *= 0.25;
					else if (enemyType.size() == BWAPI::UnitSizeTypes::Medium) finalDamage *= 0.5;
				}
				DPF += unitGroup->numUnits * (finalDamage / unitType.airWeapon().damageCooldown());
			}
		}
	} else {
		for (auto unitGroup : unitsInCombat) {
			BWAPI::UnitType unitType(unitGroup->unitTypeId);
			if ((*_unitTypeDPF)[unitType][enemyType].DPF > 0) {
				DPF += unitGroup->numUnits * (*_unitTypeDPF)[unitType][enemyType].DPF;
// 				if (!enemyType.isFlyer()) minTime = std::min(minTime, unitType.groundWeapon().damageCooldown());
// 				else minTime = std::min(minTime, unitType.airWeapon().damageCooldown());
			}
		}
	}

	double timeToKillUnit = (DPF == 0) ? DBL_MAX : (double)enemyHP / DPF;
	return round(timeToKillUnit); // we consider the time as a natural number
// 	return std::max((int)timeToKillUnit, minTime);
}

void CombatSimulatorDPS::removeUnit(unitGroup_t* unitToRemove, UnitStateVector* unitsList)
{
	for (UnitStateVector::iterator it = unitsList->begin(); it != unitsList->end(); ++it) {
		if (unitToRemove == *it) {
			delete *it;
			it = unitsList->erase(it);
			break;
		}
	}
}


void CombatSimulatorDPS::updateCombatStats(combatStats_t & combatStats, BWAPI::UnitType unitType, uint8_t numUnits)
{
	if (unitType.airWeapon().damageAmount() > 0)
		combatStats.airDPF += numUnits * ((double)unitType.airWeapon().damageAmount() / unitType.airWeapon().damageCooldown());
	if (unitType.groundWeapon().damageAmount() > 0)
		combatStats.groundDPF += numUnits * ((double)unitType.groundWeapon().damageAmount() / unitType.groundWeapon().damageCooldown());
	// In the case of Firebats and Zealots, the damage returned by BWAPI is not right, since they have two weapons:
	if (unitType == BWAPI::UnitTypes::Terran_Firebat || unitType == BWAPI::UnitTypes::Protoss_Zealot)
		combatStats.groundDPF += numUnits * ((double)unitType.groundWeapon().damageAmount() / unitType.groundWeapon().damageCooldown());
	if (unitType.isFlyer())
		combatStats.airHP += numUnits * ((double)(unitType.maxShields() + unitType.maxHitPoints()));
	else
		combatStats.groundHP += numUnits * ((double)(unitType.maxShields() + unitType.maxHitPoints()));
}


// TODO combat time is from CombatSimulatorBASIC
int CombatSimulatorDPS::getCombatLength(combatStats_t friendStats, combatStats_t enemyStats, int &timeToKillEnemy)
{
	double timeToKillEnemyAir = (enemyStats.airHP > 0) ? (friendStats.airDPF == 0) ? 99999 : enemyStats.airHP / friendStats.airDPF : 0;
	double timeToKillEnemyGround = (enemyStats.groundHP > 0) ? (friendStats.groundDPF == 0) ? 99999 : enemyStats.groundHP / friendStats.groundDPF : 0;
	double timeToKillFriendAir = (friendStats.airHP > 0) ? (enemyStats.airDPF == 0) ? 99999 : friendStats.airHP / enemyStats.airDPF : 0;
	double timeToKillFriendGround = (friendStats.groundHP > 0) ? (enemyStats.groundDPF == 0) ? 99999 : friendStats.groundHP / enemyStats.groundDPF : 0;

	timeToKillEnemy = (int)(std::max)(timeToKillEnemyAir, timeToKillEnemyGround);
	int timeToKillFriend = (int)(std::max)(timeToKillFriendAir, timeToKillFriendGround);
	return (std::min)(timeToKillEnemy, timeToKillFriend);
}

CombatSimulatorDPS::combatStats_t CombatSimulatorDPS::getCombatStats(const UnitStateVector &army)
{
	combatStats_t stats;
	for (auto unitGroup : army) {
		BWAPI::UnitType unitType(unitGroup->unitTypeId);
		updateCombatStats(stats, unitType, unitGroup->numUnits);
	}
	return stats;
}

int CombatSimulatorDPS::getCombatLength(GameState::army_t* army)
{
	combatStats_t army1Stats = getCombatStats(army->friendly);
	combatStats_t army2Stats = getCombatStats(army->enemy);

	int timeToKillarmy2;
	return getCombatLength(army1Stats, army2Stats, timeToKillarmy2);
}