#include "ActionGenerator.h"
#include "InformationManager.h"
#include <cstdlib>

using namespace BWAPI;

ActionGenerator::ActionGenerator()
	:_moreActions(false)
{}

ActionGenerator::ActionGenerator(GameState* gs, bool player)
{
	_gs = gs;
	_size = 1;
	_player = player;

	// set myUnitsList and enemyUnitList depends on the player
	if (_player) {
		_myUnitsList = &_gs->army.friendly;
		_enemyUnitsList = &_gs->army.enemy;
	} else {
		_myUnitsList = &_gs->army.enemy;
		_enemyUnitsList = &_gs->army.friendly;
	}
	
	//cleanActions(); // TODO temporal for debuging

	// Generate all possible choices
	_choices.clear();
	for (unitGroupVector::size_type i = 0; i < _myUnitsList->size(); i++) {
		if ((*_myUnitsList)[i]->endFrame <= _gs->_time) {
			unitGroup_t* unit = (*_myUnitsList)[i];
			std::vector<action_t> &actions = getUnitActions(unit);
#ifdef DEBUG_ORDERS
			_choices.push_back(choice_t(i, actions, unit->unitTypeId, unit->regionId, player));
#else
			_choices.push_back(choice_t(i, actions));
#endif
			_size *= actions.size();
		}
	}
	if (!_choices.empty()) {
		_moreActions = true;
		_choiceSizes.resize(_choices.size());
		_currentChoice.resize(_choices.size(), 0);
		for (unsigned int i = 0; i < _choices.size(); i++) {
			_choiceSizes[i] = _choices[i].actions.size();
		}
	} else {
		_moreActions = false;
	}
}

// Warning!! This will clean the actions of the original game state!
void ActionGenerator::cleanActions()
{
	for (unitGroupVector::size_type i = 0; i < _myUnitsList->size(); i++) {
		(*_myUnitsList)[i]->endFrame = -1;
		(*_myUnitsList)[i]->orderId = abstractOrder::Unknown;;
	}
}

std::vector<action_t> ActionGenerator::getUnitActions(unitGroup_t* unit)
{
	std::vector<action_t> possibleActions;
	
	BWAPI::UnitType unitType(unit->unitTypeId);

    if (unitType.isBuilding()) {
		possibleActions.push_back(action_t(abstractOrder::Idle, unit->regionId));
        return possibleActions;
    }

	// Move
	// --------------------------------
	// get neighbors from a region or chokepoint
    if (unitType.canMove()) {
		BWTA::Region* region = informationManager->_regionFromID[unit->regionId];
	    if (region != NULL) {
		    const std::set<BWTA::Chokepoint*> chokePoints = region->getChokepoints();
            if (!informationManager->_onlyRegions) {
		        for(std::set<BWTA::Chokepoint*>::const_iterator c=chokePoints.begin();c!=chokePoints.end();c++) {
					possibleActions.push_back(action_t(abstractOrder::Move, informationManager->_chokePointID[*c]));
		        }
            } else {
				std::vector<int> duplicates; // sorted vector to lookup duplicate regionID
				std::vector<int>::iterator i;
				int regionIDtoAdd;
                for(std::set<BWTA::Chokepoint*>::const_iterator c=chokePoints.begin();c!=chokePoints.end();c++) {
                    const std::pair<BWTA::Region*,BWTA::Region*> regions = (*c)->getRegions();
					regionIDtoAdd = informationManager->_regionID[regions.first];
					if (informationManager->_regionID[regions.first] == unit->regionId) {
						regionIDtoAdd = informationManager->_regionID[regions.second];
                    }
					i = std::lower_bound(duplicates.begin(), duplicates.end(), regionIDtoAdd);
					if (i == duplicates.end() || regionIDtoAdd < *i) {
						// insert if no duplicates
						duplicates.insert(i, regionIDtoAdd);
						possibleActions.push_back(action_t(abstractOrder::Move, regionIDtoAdd));
					}
		        }
            }
	    } else {
			BWTA::Chokepoint* cp = informationManager->_chokePointFromID[unit->regionId];
		    if (cp != NULL) {
			    const std::pair<BWTA::Region*,BWTA::Region*> regions = cp->getRegions();
				possibleActions.push_back(action_t(abstractOrder::Move, informationManager->_regionID[regions.first]));
				possibleActions.push_back(action_t(abstractOrder::Move, informationManager->_regionID[regions.second]));
		    }
	    }
    }

	// Attack
	// --------------------------------
	// only if there are enemies in the region that we can attack
	// TODO we cannot simulate spellCasters!!!
    if (unitType.canAttack()) {
		for (unitGroupVector::size_type i = 0; i < _enemyUnitsList->size(); i++) {
			if ((*_enemyUnitsList)[i]->regionId == unit->regionId && canAttackType(unitType, (*_enemyUnitsList)[i]->unitTypeId)) {
				possibleActions.push_back(action_t(abstractOrder::Attack, unit->regionId));
			    break;
		    }
	    }
    }

    // Nothing
    // --------------------------------
	possibleActions.push_back(action_t(abstractOrder::Idle, unit->regionId));

	return possibleActions;
}

void ActionGenerator::incrementCurrentChoice(unsigned int startPosition) {
    for(unsigned int i = 0;i<startPosition;i++) _currentChoice[i] = 0;
    _currentChoice[startPosition]++;
    if (_currentChoice[startPosition] >= _choiceSizes[startPosition]) {
        if (startPosition < _currentChoice.size()-1) {
            incrementCurrentChoice(startPosition+1);
        } else {
            _moreActions = false;
        }
    }
}

playerActions_t ActionGenerator::getNextAction() {
	playerActions_t playerActions;
	if(_moreActions) {
        int i = _choices.size();
        playerActions.reserve(i);
#ifdef DEBUG_ORDERS
        if (i == 0) DEBUG("This should not have happened...");
#endif
        while(i>0) {
            i--;
			// add unit and action to perform
			choice_t unitChoices = _choices[i];
			int choice = _currentChoice[i];
			playerActions.push_back(playerAction_t(unitChoices, choice));
        }
        incrementCurrentChoice(i);
    }
	if (!playerActions.empty()) {
		_lastAction = playerActions;
	}
	return playerActions;
}

playerActions_t ActionGenerator::getRandomAction() {
    int i = _choices.size();
    playerActions_t playerActions;
    while(i>0) {
        i--;
		// add unit and action to perform
		choice_t unitChoices = _choices[i];
        int choice = std::rand() % (unitChoices.actions.size()-1); // get random choice
		playerActions.push_back(playerAction_t(unitChoices, choice));
    }
	if (!playerActions.empty()) {
		_lastAction = playerActions;
	}
	return playerActions;
}

std::string ActionGenerator::toString(playerActions_t playerActions)
{
	std::string tmp = "";
	for (const auto& playerAction : playerActions) {
#ifdef DEBUG_ORDERS
		tmp += BWAPI::UnitType(playerAction.unitTypeId).getName() + " (" + intToString(playerAction.regionId) + "): ";
#endif
		tmp += abstractOrder::name[playerAction.action.orderID] + " (" + intToString(playerAction.action.targetRegion) + ")\n";
	}
	return tmp;
}

std::string ActionGenerator::toString()
{
    std::ostringstream tmp;
	tmp << "Possible Actions:\n";
	for (unsigned int i=0; i < _choices.size(); i++) {
#ifdef DEBUG_ORDERS
		tmp << BWAPI::UnitType(_choices[i].unitTypeId).getName() << " (" << intToString(_choices[i].regionId) << ")\n";
#endif
		for (const auto& action : _choices[i].actions) {
			tmp << "  - " << _gs->getAbstractOrderName(action.orderID) << " (" << intToString(action.targetRegion) << ")\n";
		}
	}
	tmp << "Number of children: " << _size << "\n";

	return tmp.str();
}

double ActionGenerator::getHighLevelFriendlyActions()
{
    return getHighLevelActions(_gs->army.friendly);
}

double ActionGenerator::getHighLevelEnemyActions()
{
    return getHighLevelActions(_gs->army.enemy);
}

double ActionGenerator::getHighLevelActions(unitGroupVector unitList)
{
    double size = 1;
    for (unsigned int i=0; i < unitList.size(); i++) {
	    unitGroup_t* unit = unitList[i];
	    std::vector<action_t> actions = getUnitActions(unit);
	    size *= actions.size();
    }
    return size;
}

double ActionGenerator::getLowLevelFriendlyActions()
{
    return getLowLevelActions(Broodwar->self()->getUnits());
}

double ActionGenerator::getLowLevelEnemyActions()
{
    return getLowLevelActions(Broodwar->enemy()->getUnits());
}

double ActionGenerator::getLowLevelActions(BWAPI::Unitset units)
{
    double size = 1;
	for (BWAPI::Unitset::iterator it = units.begin(); it != units.end(); it++) {
        int unitActions = 1;
        Unit unit = *it;
		
        // building case
        if (unit->getType().isBuilding()) {
            // train
            if (unit->getType().canProduce() && !unit->isTraining()) {
                unitActions++;
            } 

            if (unit->isTraining())
                unitActions++; // cancel order

        // worker case
        } else if (unit->getType().isWorker()) {
            BWAPI::Unitset unitsInSeekRange = Broodwar->getUnitsInRadius(unit->getPosition(), unit->getType().seekRange());
            // move
            unitActions += 8; // move directions
            unitActions++; // cancel order
            unitActions++; // halt

            // attack on units on range
            for (BWAPI::Unitset::iterator it2 = unitsInSeekRange.begin(); it2 != unitsInSeekRange.end(); it2++) {
                if (unit->isInWeaponRange(*it2)) unitActions++;
            }

            // harvest
            for (BWAPI::Unitset::iterator it2 = unitsInSeekRange.begin(); it2 != unitsInSeekRange.end(); it2++) {
                if ((*it2)->getType().isResourceContainer()) unitActions++;
            }

            // repair
            for (BWAPI::Unitset::iterator it2 = unitsInSeekRange.begin(); it2 != unitsInSeekRange.end(); it2++) {
                if ((*it2)->getType().isMechanical()) unitActions++;
            }

            // build
            unitActions += 5; // considering only the 5 basic Terran buildings

            unitActions++; // cancel order
        // combat unit case
        } else {
            BWAPI::Unitset unitsInSeekRange = Broodwar->getUnitsInRadius(unit->getPosition(), unit->getType().seekRange());
            if (unit->getType().canMove()) {
                unitActions += 8; // move directions
                unitActions++; // cancel order
                unitActions++; // halt
                unitActions += 8; // patrol

            }
            // abilities on units on range
            if (!unit->getType().abilities().empty()) {
				for (auto &ability : unit->getType().abilities()) {
					// some abilities affect target units
					if (ability.targetsUnit() || ability.targetsPosition()) {
						unitActions += (unit->getType().abilities().size()*unitsInSeekRange.size());
					} else { // other abilities affect only yourself
						unitActions++;
					}
				}
            }

            if (unit->getType().canAttack()) {
                // attack on units on range
                for (BWAPI::Unitset::iterator it2 = unitsInSeekRange.begin(); it2 != unitsInSeekRange.end(); it2++) {
                    if (unit->isInWeaponRange(*it2)) unitActions++;
                }
            }
        }

        // update total actions
        size *= unitActions;
    }
    return size;
}

double ActionGenerator::getSparcraftFriendlyActions()
{
    return getSparcraftActions(Broodwar->self()->getUnits());
}
double ActionGenerator::getSparcraftEnemyActions()
{
    return getSparcraftActions(Broodwar->enemy()->getUnits());
}

double ActionGenerator::getSparcraftActions(BWAPI::Unitset units)
{
    double size = 1;
	for (BWAPI::Unitset::iterator it = units.begin(); it != units.end(); it++) {
        int unitActions = 1; // skip
        Unit unit = *it;

        // skip non combat units
        if (unit->getType().canAttack()) {
            BWAPI::Unitset unitsInSeekRange = Broodwar->getUnitsInRadius(unit->getPosition(), unit->getType().seekRange());
            // attack on units on range
            for (BWAPI::Unitset::iterator it2 = unitsInSeekRange.begin(); it2 != unitsInSeekRange.end(); it2++) {
                if (unit->isInWeaponRange(*it2)) unitActions++;
            }
        }

        if (unit->getType().canMove()) {
            unitActions += 4; // move directions
        }

        if (unit->getType() == UnitTypes::Terran_Medic) {
            BWAPI::Unitset unitsInSeekRange = Broodwar->getUnitsInRadius(unit->getPosition(), unit->getType().seekRange());
            // attack on units on range
            for (BWAPI::Unitset::iterator it2 = unitsInSeekRange.begin(); it2 != unitsInSeekRange.end(); it2++) {
                if ((*it2)->getType().isOrganic()) unitActions++;
            }
        }

        size *= unitActions;
    }
    return size;

}