#include "AbstractLayer.h"
#include "ABCD.h"
#include "MCTSCD.h"
#include "EvaluationFunctionBasic.h"

AbstractLayer::AbstractLayer()
{
    // clear lists
	informationManager->gameState.cleanArmyData();

    // add only enemy units
    informationManager->gameState.addAllEnemyUnits();

    // add our buildings
    informationManager->gameState.addSelfBuildings();

}

// Add squad to game state and store reference ID
void AbstractLayer::addSquadToGameState(SquadAgent* squad)
{
	std::map<unsigned short, unsigned int> groupIdFrequency;
	
	unsigned short abstractGroupID;
    // for each unit on the squad, add it to the game state and save id reference
	for (const auto& squadUnit : squad->_squadUnits) {
		if (squadUnit->_unit->getType().isWorker()) continue; // ignore workers
		abstractGroupID = informationManager->gameState.addFriendlyUnit(squadUnit->_unit);
        groupIdFrequency[abstractGroupID]++;
    }

    unsigned int maxFrequency = 0;
	unsigned short bestGroup;
    // assign to the squad the most common group ID
	for (const auto& frequency : groupIdFrequency) {
		if (frequency.second > maxFrequency) {
			bestGroup = frequency.first;
			maxFrequency = frequency.second;
        }
    }

    // one idGroup can have many squads!!
    _idToSquad[bestGroup].insert(squad);
    //LOG("Best group for squad (" << squad << "): " << bestGroup);

}

// Execute search and return best targetPosition for each squad
std::map<SquadAgent*, BWAPI::Position> AbstractLayer::searchBestOrders()
{
    // compare game states
    if (!informationManager->lastGameState.gameover()) {
        int misplacedUnits = 0;
        int totalUnits = 0;
		informationManager->gameState.compareFriendlyUnits(informationManager->lastGameState, misplacedUnits, totalUnits);
        LOG("Correct Units: " << totalUnits-misplacedUnits << " of " << totalUnits << " jaccard: " << float(totalUnits-misplacedUnits)/float(totalUnits));
    }

    // now that we have all the units in the game state, compute expected end frame
    informationManager->gameState.calculateExpectedEndFrameForAllGroups();
    // and forward until next point decision
    // we can move Forward or set our orders to 0 to find the best inmidate action
    //informationManager->gameState.moveForward();
    //LOG(informationManager->gameState.toString());
    informationManager->gameState.resetFriendlyActions();
    //LOG(informationManager->gameState.toString());

    // Search algorithm
    playerActions_t bestActions;
    EvaluationFunctionBasic ef;
    if (SEARCH_ALGORITHM == "ABCD") {
        int depth = LoadConfigInt("ABCD", "depth", 1);
        ABCD searchAlg = ABCD(depth, &ef);
        bestActions = searchAlg.start(true, informationManager->gameState);
    } else if (SEARCH_ALGORITHM == "MCTSCD") {
        int depth = LoadConfigInt("MCTSCD", "depth", 1);
        int iterations = LoadConfigInt("MCTSCD", "iterations");
        int maxSimTime = LoadConfigInt("MCTSCD", "max_simulation_time");
        MCTSCD searchAlg = MCTSCD(depth, &ef, iterations, maxSimTime);
        bestActions = searchAlg.start(true, informationManager->gameState);
    } else {
        // get random actions
        ActionGenerator moveGenerator = ActionGenerator(&informationManager->gameState, true);
        bestActions = moveGenerator.getRandomAction();
    }

    // update last gameState
    informationManager->lastGameState = informationManager->gameState;
    informationManager->lastGameState.execute(bestActions, true);
    informationManager->lastGameState.moveForward(HIGH_LEVEL_REFRESH);
    informationManager->lastGameState.mergeGroups();

    //LOG("Best actions: ");
    std::map<SquadAgent*, BWAPI::Position> bestOrders;
    BWAPI::Position targetPosition;
	for (const auto& groupAction : bestActions) {
// 		abstractOrder::unpackAction(*i, groupID, orderId, targetRegionId);
		std::set<SquadAgent*> squadSet = _idToSquad[groupAction.pos];

        for(std::set<SquadAgent*>::const_iterator squad = squadSet.begin(); squad!=squadSet.end(); ++squad) {
			targetPosition = informationManager->gameState.getCenterRegionId((int)groupAction.action.targetRegion);
            bestOrders[*squad] = targetPosition;
//             LOG("  - Group ID: " << groupID << ", action: " << informationManager->gameState.getAbstractOrderName((int)orderId) << " region: " << (int)targetRegionId 
//                 << "(" << targetPosition.x << "," << targetPosition.y << ") squad (" << *squad << ")");
        }
    }

    return bestOrders;
}

bool AbstractLayer::hasFriendlyUnits()
{
    return !(informationManager->gameState.army.friendly.empty() || informationManager->gameState.hasOnlyBuildings(informationManager->gameState.army.friendly));
}

float AbstractLayer::getEvaluation()
{
    EvaluationFunctionBasic ef;
    return ef.evaluate(informationManager->gameState, true);
}

void AbstractLayer::printBranchingStats()
{
   // branching factor start for root node
    ActionGenerator actions = ActionGenerator(&informationManager->gameState, true);
    double highLevelFriendly = actions.getHighLevelFriendlyActions();
    double highLevelEnemy = actions.getHighLevelEnemyActions();
    double highLevelTotal = highLevelFriendly * highLevelEnemy;
    double lowLevelFriendly = actions.getLowLevelFriendlyActions();
    double lowLevelEnemy = actions.getLowLevelEnemyActions();
    double lowLevelTotal = lowLevelFriendly * lowLevelEnemy;
    double sparcraftFriendly = actions.getSparcraftFriendlyActions();
    double sparcraftEnemy = actions.getSparcraftEnemyActions();
    double sparcraftTotal = sparcraftFriendly * sparcraftEnemy;
    LOG("LL friend: " << lowLevelFriendly << " LL enemy: " << lowLevelEnemy << " LL total: " << lowLevelTotal <<
        " Spar friend: " << sparcraftFriendly << " Spar enemy: " << sparcraftEnemy << " Spar total: " << sparcraftTotal <<
        " HL friend: " << highLevelFriendly << " HL enemy: " << highLevelEnemy << " HL total: " << highLevelTotal);
}