#include "QtWindow.h"
#include "InformationManager.h"
#include "Search/ABCD.h"
#include "Search/EvaluationFunctionBasic.h"
#include "Search/DpsLearner.h"
#include "Search/CombatSimulatorBasic.h"
#include "Search/CombatSimulatorDPS.h"
#include "Search/TargetSorting.h"
#include "Utils/Timer.h"

myQtApp::myQtApp(QWidget *parent)
	:regionIdSelected(-1)
{
	spacePartition = LoadConfigString("high_level_search", "space_partition", "REGIONS_AND_CHOKEPOINTS");

	// this sets up GUI
	setupUi(this); 

	// -------------------- INITIALIZING VARIABLES --------------------
	// Tab Options
	chkMapAnalysisBWTA->setChecked(false);
	chkBuildMap->setChecked(PRINT_BUILD_MAP);
	chkGroundDPS->setChecked(PRINT_GROUND_DPS);
	chkAirDPS->setChecked(PRINT_AIR_DPS);
	chkBuildOrder->setChecked(PRINT_BUILD_ORDER);
	// Tab Game State
	mapScene = new QGraphicsScene();
	chkRegionIdMap->setChecked(PRINT_REGION_ID_MAP);

	gameStateLabels << tr("Unit type") << tr("Number") << tr("Region") << tr("Order") << tr("Target") << tr("End frame");
	unitsToLocationTable->setColumnCount(6);
	unitsToLocationTable->setHorizontalHeaderLabels(gameStateLabels);
	unitsToLocationTable->setColumnWidth(0,150);

	unitsToLocationTable_2->setColumnCount(6);
	unitsToLocationTable_2->setHorizontalHeaderLabels(gameStateLabels);
	unitsToLocationTable_2->setColumnWidth(0,150);

	unitMapLabels << tr("Unit type") << tr("Number") << tr("Order") << tr("End frame");
	unitsToLocationTable_3->setColumnCount(4);
	unitsToLocationTable_3->setHorizontalHeaderLabels(unitMapLabels);
	unitsToLocationTable_3->setColumnWidth(0,150);

	unitsToLocationTable_4->setColumnCount(4);
	unitsToLocationTable_4->setHorizontalHeaderLabels(unitMapLabels);
	unitsToLocationTable_4->setColumnWidth(0,150);

	// Default data for Effectiveness tab
	QStringList unitTypesLabels;
	unitTypesLabels << tr("SVC") << tr("Marine") << tr("Medic") << tr("Firebat") << tr("Ghost")
					<< tr("Vulture") << tr("Tank") << tr("Goliath") << tr("Wraith") << tr("Science Vessel")
					<< tr("Battlecruiser") << tr("Valkyrie");
	unitEffecTable->setColumnCount(12);
	unitEffecTable->setRowCount(12);
	unitEffecTable->setHorizontalHeaderLabels(unitTypesLabels);
	unitEffecTable->setVerticalHeaderLabels(unitTypesLabels);

	// Tab Combat Simulator
	for (auto unitType : BWAPI::UnitTypes::allUnitTypes()) {
		if (unitType.getID() > 163) break; // no interesting units farther this point
		if ((unitType.canAttack() || unitType.isSpellcaster()) && !unitType.isHero()) {
			comboDPF1->addItem(unitType.c_str(), unitType.getID());
			comboDPF2->addItem(unitType.c_str(), unitType.getID());
		}
	}
	comboDPF1->model()->sort(0);
	comboDPF2->model()->sort(0);
	
	// Initialize with tab selected
	gameStateTabChanged(gameStateTab->currentIndex());


	// -------------------- SIGNLAS/SLOTS ------------------------
	// Tab Options
	connect(speedSlider, &QSlider::valueChanged, this, &myQtApp::changeSpeed);
	connect(pauseButton, &QPushButton::clicked, this, &myQtApp::pauseGame);
	connect(resumeButton, &QPushButton::clicked, this, &myQtApp::resumeGame);
	connect(chkBuildMap, &QCheckBox::stateChanged, this, &myQtApp::changeBuildMap);
	connect(chkGroundDPS, &QCheckBox::stateChanged, this, &myQtApp::changeGroundDPS);
	connect(chkAirDPS, &QCheckBox::stateChanged, this, &myQtApp::changeAirDPS);
	connect(chkBuildOrder, &QCheckBox::stateChanged, this, &myQtApp::changeBuildOrder);
	// Tab Game State
	connect(chkRegionIdMap, &QCheckBox::stateChanged, this, &myQtApp::changeRegionId);
	connect(informationManager->_GUIsignal, &Qsignal::mapInfoChanged, this, &myQtApp::changeBWTAdata);
	connect(informationManager->_GUIsignal, &Qsignal::gameStateChanged, this, &myQtApp::updateGameStateTable);
	connect(gameStateTab, &QTabWidget::currentChanged, this, &myQtApp::gameStateTabChanged);
	// -- SubTab Game Search
	connect(importStateButton, &QPushButton::clicked, this, &myQtApp::importState);
	connect(loadButton, &QPushButton::clicked, this, &myQtApp::loadFile);
	connect(abcdSearchButton, &QPushButton::clicked, this, &myQtApp::abcdSearch);

	connect(generateActionsButton, &QPushButton::clicked, this, &myQtApp::generateActions);
	connect(nextActionButton, &QPushButton::clicked, this, &myQtApp::expandNode);
	connect(randomActionButton, &QPushButton::clicked, this, &myQtApp::expandRandomNode);
	connect(executeActionButton, &QPushButton::clicked, this, &myQtApp::executeAction);
	connect(rolloutButton, &QPushButton::clicked, this, &myQtApp::rolloutGame);
	// -- SubTab Combat Simulator
	connect(combatParseButton, &QPushButton::clicked, this, &myQtApp::parseCombats);
	connect(combatClearButton, &QPushButton::clicked, this, &myQtApp::clearCombatsParsed);
	connect(crossValidationButton, &QPushButton::clicked, this, &myQtApp::crossValidation);
	connect(learnButton, &QPushButton::clicked, this, &myQtApp::learnCombat);
	connect(getDPFbutton, &QPushButton::clicked, this, &myQtApp::getDPF);
	connect(combatSimulatorButton, &QPushButton::clicked, this, &myQtApp::combatSimulator);
}

myQtApp::~myQtApp()
{
	delete mapScene;
}

void myQtApp::changeSpeed(int value)
{
	BWAPI::Broodwar->setLocalSpeed(value);
	speedLabel->setText( "Game speed: " + QString::number(value) );
}

void myQtApp::pauseGame()
{
	BWAPI::Broodwar->pauseGame();
}

void myQtApp::resumeGame()
{
	BWAPI::Broodwar->resumeGame();
}

void myQtApp::changeBWTAdata()
{
// 	textEdit->append("myQtApp::changeBWTAdata() called");
	mapScene->clear();
	// draw regions
	drawPolygons(&BWTA::getUnwalkablePolygons(),mapScene);

	double x0, y0, x1, y1;
	QPen linePen(QColor(0, 0, 0));
	linePen.setWidth(2);

	// draw chokepoints borders
	for (auto chokepoint : BWTA::getChokepoints()) {
		const auto positions = chokepoint->getSides();
		x0 = (double)positions.first.x / 8;
		y0 = (double)positions.first.y / 8;
		x1 = (double)positions.second.x / 8;
		y1 = (double)positions.second.y / 8;
		mapScene->addLine(QLineF(x0, y0, x1, y1));
	}

	// draw center of regions
	QColor color = Qt::blue;
	for (auto region : BWTA::getRegions()) {
		x0 = (double)(region->getCenter().x) / 8;
		y0 = (double)(region->getCenter().y) / 8;
		mapScene->addEllipse(QRectF(x0 - 6, y0 - 6, 12, 12), QPen(color), QBrush(color));
		QGraphicsTextItem* text = mapScene->addText(QString::number(informationManager->_regionID[region]));
		text->setPos(x0, y0 + 6);
		text->setDefaultTextColor(color);
	}

	if (spacePartition == "REGIONS_AND_CHOKEPOINTS") {
		color = Qt::red;
		// draw center of chokepoints and lines to regions
		for (auto chokepoint : BWTA::getChokepoints()) {
			x0 = (double)(chokepoint->getCenter().x) / 8;
			y0 = (double)(chokepoint->getCenter().y) / 8;

			// lines
			const auto regions = chokepoint->getRegions();
			x1 = (double)regions.first->getCenter().x / 8;
			y1 = (double)regions.first->getCenter().y / 8;
			mapScene->addLine(QLineF(x0, y0, x1, y1), linePen);
			x1 = (double)regions.second->getCenter().x / 8;
			y1 = (double)regions.second->getCenter().y / 8;
			mapScene->addLine(QLineF(x0, y0, x1, y1), linePen);

			// chokepoint center
			mapScene->addEllipse(QRectF(x0 - 3, y0 - 3, 6, 6), QPen(color), QBrush(color));
			QGraphicsTextItem* text = mapScene->addText(QString::number(informationManager->_chokePointID[chokepoint]));
			text->setPos(x0, y0 + 6);
			text->setDefaultTextColor(color);
		}
	} else {
		// draw lines between regions
		for (auto chokepoint : BWTA::getChokepoints()) {
			const auto regions = chokepoint->getRegions();
			x0 = (double)regions.first->getCenter().x / 8;
			y0 = (double)regions.first->getCenter().y / 8;
			x1 = (double)regions.second->getCenter().x / 8;
			y1 = (double)regions.second->getCenter().y / 8;
			mapScene->addLine(QLineF(x0, y0, x1, y1), linePen);
		}
	}

	//textEdit->append( "Scene size " + QString::number(mapScene->width()) + "," + QString::number(mapScene->height()) );

	// Render map
	mapView->setScene(mapScene);
	mapView->setRenderHint(QPainter::Antialiasing);
	mapView->show();
}

void myQtApp::updateGameStateTable()
{
	if (gameStateTab->currentIndex() == 1) {
		updateUnitMap();
	} else {
		gameStateMutex.lock();
		unitGroupVector friendlyUnits = informationManager->gameState.army.friendly;
		int rows = friendlyUnits.size();
		unitsToLocationTable->setRowCount(rows);
		for( int i = 0; i < rows; ++i ) {
			BWAPI::UnitType unitType = BWAPI::UnitType(friendlyUnits[i]->unitTypeId);
			//BWAPI::Order unitCommand = BWAPI::Order(friendlyUnits[i].orderId);
			//std::string unitCommand = BWAPI::Order(friendlyUnits[i].orderId).getName();
			std::string unitCommand = informationManager->gameState.getAbstractOrderName(friendlyUnits[i]->orderId);
			QTableWidgetItem *item0 = new QTableWidgetItem(tr("%1").arg(unitType.getName().c_str()));
			QTableWidgetItem *item1 = new QTableWidgetItem(tr("%1").arg(friendlyUnits[i]->numUnits));
			QTableWidgetItem *item2 = new QTableWidgetItem(tr("%1").arg(friendlyUnits[i]->regionId));
			QTableWidgetItem *item3 = new QTableWidgetItem(tr("%1").arg(unitCommand.c_str()));
			//QTableWidgetItem *item4 = new QTableWidgetItem( tr("%1").arg(informationManager->gameState.getAbstractOrderName(unitCommand).c_str()) );
			QTableWidgetItem *item5 = new QTableWidgetItem(tr("%1").arg(friendlyUnits[i]->targetRegionId));
			int endFrame = friendlyUnits[i]->endFrame;
			if (endFrame > 0) endFrame -= BWAPI::Broodwar->getFrameCount();
			QTableWidgetItem *item6 = new QTableWidgetItem( tr("%1").arg(endFrame) );
			unitsToLocationTable->setItem( i, 0, item0 );
			unitsToLocationTable->setItem( i, 1, item1 );
			unitsToLocationTable->setItem( i, 2, item2 );
			unitsToLocationTable->setItem( i, 3, item3 );
			unitsToLocationTable->setItem( i, 4, item5 );
			unitsToLocationTable->setItem( i, 5, item6 );
	// 		unitsToLocationTable->setItem( i, 4, item4 );
	// 		unitsToLocationTable->setItem( i, 5, item5 );
		}
		unitsToLocationTable->sortByColumn(0, Qt::AscendingOrder);

		//enemy
		unitGroupVector enemyUnits = informationManager->gameState.army.enemy;
		rows = enemyUnits.size();
		unitsToLocationTable_2->setRowCount(rows);
		for( int i = 0; i < rows; ++i ) {
			BWAPI::UnitType unitType = BWAPI::UnitType(enemyUnits[i]->unitTypeId);
			//BWAPI::Order unitCommand = BWAPI::Order(enemyUnits[i].orderId);
			//std::string unitCommand = BWAPI::Order(enemyUnits[i].orderId).getName();
			std::string unitCommand = informationManager->gameState.getAbstractOrderName(enemyUnits[i]->orderId);
			QTableWidgetItem *item0 = new QTableWidgetItem(tr("%1").arg(unitType.getName().c_str()));
			QTableWidgetItem *item1 = new QTableWidgetItem(tr("%1").arg(enemyUnits[i]->numUnits));
			QTableWidgetItem *item2 = new QTableWidgetItem(tr("%1").arg(enemyUnits[i]->regionId));
			QTableWidgetItem *item3 = new QTableWidgetItem(tr("%1").arg(unitCommand.c_str()));
			//QTableWidgetItem *item4 = new QTableWidgetItem( tr("%1").arg(informationManager->gameState.getAbstractOrderName(unitCommand).c_str()) );
			QTableWidgetItem *item5 = new QTableWidgetItem(tr("%1").arg(enemyUnits[i]->targetRegionId));
			int endFrame = enemyUnits[i]->endFrame;
			if (endFrame > 0) endFrame -= BWAPI::Broodwar->getFrameCount();
			QTableWidgetItem *item6 = new QTableWidgetItem( tr("%1").arg(endFrame) );
			unitsToLocationTable_2->setItem( i, 0, item0 );
			unitsToLocationTable_2->setItem( i, 1, item1 );
			unitsToLocationTable_2->setItem( i, 2, item2 );
			unitsToLocationTable_2->setItem( i, 3, item3 );
			unitsToLocationTable_2->setItem( i, 4, item5 );
			unitsToLocationTable_2->setItem( i, 5, item6 );
	// 		unitsToLocationTable_2->setItem( i, 4, item4 );
	// 		unitsToLocationTable_2->setItem( i, 5, item5 );
		}
		unitsToLocationTable_2->sortByColumn(0, Qt::AscendingOrder);
		gameStateMutex.unlock();
	}
}

void myQtApp::drawPolygons(const std::set<BWTA::Polygon*>* polygons, QGraphicsScene* scene)
{
	for(std::set<BWTA::Polygon*>::const_iterator i=polygons->begin();i!=polygons->end();i++) {
		BWTA::Polygon boundary = *(*i);
		drawPolygon(boundary,QColor(180,180,180),scene);
		std::vector<BWTA::Polygon> pHoles = boundary.holes;
		for(std::vector<BWTA::Polygon>::iterator h=pHoles.begin();h!=pHoles.end();h++) {
			drawPolygon(*h,QColor(255,100,255),scene);
		}
	}
}

void myQtApp::drawPolygon(BWTA::Polygon& p, QColor qc, QGraphicsScene* scene, double scale)
{
	QVector<QPointF> qp;
	for(int i=0;i<(int)p.size();i++) {
		int j=(i+1)%p.size();
		qp.push_back(QPointF(p[i].x*scale,p[i].y*scale));
	}
	scene->addPolygon(QPolygonF(qp),QPen(QColor(0,0,0)),QBrush(qc));  
}

void myQtApp::drawCustomPolygon(BWTA::Polygon& p, QGraphicsScene* scene, int regionId)
{
	double scale = 0.125;
	QVector<QPointF> qp;
	for (int i = 0; i < (int)p.size(); i++) {
		int j = (i + 1) % p.size();
		qp.push_back(QPointF(p[i].x*scale, p[i].y*scale));
	}

	CustomPolygonItem* item = new CustomPolygonItem;
	item->setPolygon(QPolygonF(qp));
// 	item->setBrush(QBrush(color));
	item->setPen(QPen(Qt::white));
	item->regionId = regionId;

	scene->addItem(item);
}

void myQtApp::changeDisplayBWTA(int state)
{
	//textEdit->append( "Display BWTA: " + QString::number(state) );
}

void myQtApp::changeBuildMap(int state)
{
	PRINT_BUILD_MAP = !PRINT_BUILD_MAP;
}

void myQtApp::changeGroundDPS(int state)
{
	PRINT_GROUND_DPS = !PRINT_GROUND_DPS;
}

void myQtApp::changeAirDPS(int state)
{
	PRINT_AIR_DPS = !PRINT_AIR_DPS;
}

void myQtApp::changeBuildOrder(int state)
{
	PRINT_BUILD_ORDER = !PRINT_BUILD_ORDER;
}

void myQtApp::changeRegionId(int state)
{
	PRINT_REGION_ID_MAP = !PRINT_REGION_ID_MAP;
}

void myQtApp::gameStateTabChanged(int tabIndex)
{
	switch (tabIndex) {
		case 1: // Units
			updateUnitMap();
			blueInfo->setText( "Friendly units" );
			redInfo->setText( "Enemy units" );
			break;
		case 2: // Effectiveness
			updateEffectivenessMap();
			blueInfo->setText( "Friendly region effectiveness" );
			redInfo->setText( "Enemy region effectiveness" );
			break;
		case 0: // Region ID
		default:
			changeBWTAdata();
			blueInfo->setText( "Center of the region" );
			redInfo->setText( "Center of the chokepoint" );
			break;
	}
}

void myQtApp::updateUnitMap()
{
// 	textEdit->append("myQtApp::updateUnitMap()");
	mapScene->clear();
	// draw regions
	drawPolygons(&BWTA::getUnwalkablePolygons(),mapScene);
	// draw chokepoints borders
	double x0, y0, x1, y1;
	for (auto chokepoint : BWTA::getChokepoints()) {
		const auto positions = chokepoint->getSides();
		x0 = (double)positions.first.x / 8;
		y0 = (double)positions.first.y / 8;
		x1 = (double)positions.second.x / 8;
		y1 = (double)positions.second.y / 8;
		mapScene->addLine(QLineF(x0, y0, x1, y1));
	}

	// draw selected region
	for (auto region : BWTA::getRegions()) {
		BWTA::Polygon regionPolygon = region->getPolygon();
		drawCustomPolygon(regionPolygon, mapScene, informationManager->_regionID[region]);
	}

	// Print circles on center regions
	// TODO i don't need mutex anymore
	gameStateMutex.lock();
	unitGroupVector friendlyUnits = informationManager->gameState.army.friendly;
	for( unsigned int i = 0; i < friendlyUnits.size(); ++i ) {
		BWAPI::Position groupPosition = getCenterRegionId(friendlyUnits[i]->regionId);
		if (groupPosition != BWAPI::Positions::None) {
			drawUnitEllipse((int)groupPosition.x / 8, (int)groupPosition.y / 8, 5, Qt::blue, friendlyUnits[i]->regionId, mapScene);
		}
	}
	unitGroupVector enemyUnits = informationManager->gameState.army.enemy;
	for( unsigned int i = 0; i < enemyUnits.size(); ++i ) {
		BWAPI::Position groupPosition = getCenterRegionId(enemyUnits[i]->regionId);
		if (groupPosition != BWAPI::Positions::None) {
			drawUnitEllipse((int)groupPosition.x / 8, (int)groupPosition.y / 8, -5, Qt::red, enemyUnits[i]->regionId, mapScene);
		}
	}
	gameStateMutex.unlock();


	// Render map
	mapView->setScene(mapScene);
	mapView->setRenderHint(QPainter::Antialiasing);
	mapView->show();

	updateUnitTable();
}

void myQtApp::updateUnitTable()
{
	int rows = 0;
	int printRow = 0;
	gameStateMutex.lock();
	unitGroupVector friendlyUnits = informationManager->gameState.army.friendly;
	for (unsigned int i = 0; i < friendlyUnits.size(); ++i) {
		if (regionIdSelected == friendlyUnits[i]->regionId) {
			rows++;
		}
	}
	unitsToLocationTable_3->setRowCount(rows);
	if (rows > 0) {
		for (unsigned int i = 0; i < friendlyUnits.size(); ++i) {
			if (regionIdSelected == friendlyUnits[i]->regionId) {
				BWAPI::UnitType unitType = BWAPI::UnitType(friendlyUnits[i]->unitTypeId);
				std::string unitCommand = informationManager->gameState.getAbstractOrderName(friendlyUnits[i]->orderId);
				QTableWidgetItem *item0 = new QTableWidgetItem(tr("%1").arg(unitType.getName().c_str()));
				QTableWidgetItem *item1 = new QTableWidgetItem(tr("%1").arg(friendlyUnits[i]->numUnits));
				QTableWidgetItem *item3 = new QTableWidgetItem(tr("%1").arg(unitCommand.c_str()));
				int endFrame = friendlyUnits[i]->endFrame;
				if (endFrame > 0) endFrame -= BWAPI::Broodwar->getFrameCount();
				QTableWidgetItem *item6 = new QTableWidgetItem(tr("%1").arg(endFrame));
				unitsToLocationTable_3->setItem(printRow, 0, item0);
				unitsToLocationTable_3->setItem(printRow, 1, item1);
				unitsToLocationTable_3->setItem(printRow, 2, item3);
				unitsToLocationTable_3->setItem(printRow, 3, item6);
				printRow++;
			}
		}
		unitsToLocationTable_3->sortByColumn(0, Qt::AscendingOrder);
	}

	rows = 0;
	printRow = 0;
	unitGroupVector enemyUnits = informationManager->gameState.army.enemy;
	for (unsigned int i = 0; i < enemyUnits.size(); ++i) {
		if (regionIdSelected == enemyUnits[i]->regionId) {
			rows++;
		}
	}
	unitsToLocationTable_4->setRowCount(rows);
	if (rows > 0) {
		for (unsigned int i = 0; i < enemyUnits.size(); ++i) {
			if (regionIdSelected == enemyUnits[i]->regionId) {
				BWAPI::UnitType unitType = BWAPI::UnitType(enemyUnits[i]->unitTypeId);
				std::string unitCommand = informationManager->gameState.getAbstractOrderName(enemyUnits[i]->orderId);
				QTableWidgetItem *item0 = new QTableWidgetItem(tr("%1").arg(unitType.getName().c_str()));
				QTableWidgetItem *item1 = new QTableWidgetItem(tr("%1").arg(enemyUnits[i]->numUnits));
				QTableWidgetItem *item3 = new QTableWidgetItem(tr("%1").arg(unitCommand.c_str()));
				int endFrame = enemyUnits[i]->endFrame;
				if (endFrame > 0) endFrame -= BWAPI::Broodwar->getFrameCount();
				QTableWidgetItem *item6 = new QTableWidgetItem(tr("%1").arg(endFrame));
				unitsToLocationTable_4->setItem(printRow, 0, item0);
				unitsToLocationTable_4->setItem(printRow, 1, item1);
				unitsToLocationTable_4->setItem(printRow, 2, item3);
				unitsToLocationTable_4->setItem(printRow, 3, item6);
				printRow++;
			}
		}
		unitsToLocationTable_4->sortByColumn(0, Qt::AscendingOrder);
	}
	gameStateMutex.unlock();
}

void myQtApp::drawUnitEllipse(int x, int y, int xOffset, QColor color, int regionId, QGraphicsScene* scene)
{
	CustomEllipseItem* item = new CustomEllipseItem;
	item->setRect(x+xOffset, y, 10, 10);
	item->setBrush(QBrush(color));
	item->setPen(QPen(color));
	item->regionId = regionId;

	scene->addItem(item);
}

void myQtApp::drawCircleText(int x, int y, int xOffset, QColor color, int score, QGraphicsScene* scene)
{
	CustomEllipseItem* item = new CustomEllipseItem;
	item->setRect(x+xOffset, y, 20, 20);
	item->setBrush(QBrush(Qt::white));
	item->setPen(QPen(color));
	item->regionId = score;
	scene->addItem(item);

	QGraphicsTextItem * io = new QGraphicsTextItem;
	io->setDefaultTextColor(QColor(0,0,255));
	io->setPos(x+xOffset-1,y-1);
	io->setPlainText(QString::number(score));
	scene->addItem(io);
}

// TODO: duplicated from GameState!!!!
BWAPI::Position myQtApp::getCenterRegionId(int regionId)
{
	BWTA::Region* region = informationManager->_regionFromID[regionId];
	if (region != NULL) {
		return region->getCenter();
	} else {
		BWTA::Chokepoint* cp = informationManager->_chokePointFromID[regionId];
		if (cp != NULL) {
			return cp->getCenter();
		} else {
			return BWAPI::Positions::None;
		}
	}
}

void myQtApp::updateEffectivenessMap()
{
// 	textEdit->append("myQtApp::updateEffectivenessMap()");
	mapScene->clear();
	// draw regions
	drawPolygons(&BWTA::getUnwalkablePolygons(),mapScene);
	// draw center of choke points/regions
	double x0, y0, x1, y1;
	QPen qp(QColor(0,0,0));
	qp.setWidth(2);
	const std::set<BWTA::Chokepoint*> chokePoints = BWTA::getChokepoints();
	for(std::set<BWTA::Chokepoint*>::const_iterator c=chokePoints.begin();c!=chokePoints.end();c++) {
		// draw choke point
		x0 = (double)((*c)->getCenter().x)/8;
		y0 = (double)((*c)->getCenter().y)/8;
		drawCircleText(x0, y0, 10, Qt::blue, 12, mapScene);
		drawCircleText(x0, y0, -10, Qt::red, 12, mapScene);

		const std::pair<BWTA::Region*,BWTA::Region*> regions = (*c)->getRegions();
		// draw region 1
		x1 = (double)regions.first->getCenter().x/8;
		y1 = (double)regions.first->getCenter().y/8;
		drawCircleText(x1, y1, 10, Qt::blue, 13, mapScene);
		drawCircleText(x1, y1, -10, Qt::red, 10, mapScene);

		// draw region 2
		x1 = (double)regions.second->getCenter().x/8;
		y1 = (double)regions.second->getCenter().y/8;
		drawCircleText(x1, y1, 10, Qt::blue, 10, mapScene);
		drawCircleText(x1, y1, -10, Qt::red, 15, mapScene);
	}

	// Render map
	mapView->setScene(mapScene);
	mapView->setRenderHint(QPainter::Antialiasing);
	mapView->show();
}

// Game Search
// *********************

void myQtApp::importState()
{
	// TODO import state from StarCraft
	testActions = ActionGenerator(&informationManager->gameState);
	//actionGeneration.cleanActions();
	searchLog->append(QString::fromStdString(testActions._gs->toString()));
	searchLog->append("By default generating actions for friendly");
	searchLog->append(QString::fromStdString(testActions.toString()));
}

void myQtApp::generateActions()
{
	if (informationManager->gameState.canExecuteAnyAction(true)) {
		searchLog->append("Generating actions for friendly");
		testActions = ActionGenerator(&informationManager->gameState, true);
	} else {
		searchLog->append("Generating actions for enemy");
		testActions = ActionGenerator(&informationManager->gameState, false);
	}
	searchLog->append(QString::fromStdString(testActions.toString()));
}

void myQtApp::expandNode()
{
	playerActions_t unitsAction = testActions.getNextAction();
	if (!unitsAction.empty()) {
		searchLog->append(QString::fromStdString(testActions.toString(unitsAction)));
	} else {
		searchLog->append("No more actions");
	}
}

void myQtApp::expandRandomNode()
{
	playerActions_t unitsAction = testActions.getRandomAction();
	if (!unitsAction.empty()) {
		searchLog->append(QString::fromStdString(testActions.toString(unitsAction)));
	} else {
		searchLog->append("No more actions");
	}
}


void myQtApp::executeAction()
{
	searchLog->append("Execute actions:");
	informationManager->gameState.execute(testActions._lastAction, testActions._player);
	searchLog->append(QString::fromStdString(informationManager->gameState.toString()));

	searchLog->append("Forward time:");
	informationManager->gameState.moveForward();
	searchLog->append(QString::fromStdString(informationManager->gameState.toString()));
}

void myQtApp::loadFile()
{
    std::string gameStateLog = "bwapi-data\\logs\\gameState.txt";
    searchLog->append( QString("Loading file: %1").arg(gameStateLog.c_str()) );
    LOG( "Loading file: " << gameStateLog );

    // clear lists
	informationManager->gameState.cleanArmyData();
	informationManager->gameState._time = 0;

    std::ifstream infile(gameStateLog.c_str());
    std::string line;
    int listID = 1; // 1=friendList 2=enemyList
    while (std::getline(infile, line)) {
        //searchLog->append(line.c_str());
        //LOG(line.c_str());
        std::istringstream iss(line);
        int unitTypeID, numUnits, regionID, targetRegionID, endFrame;
        std::string orderString, extraInfo;
		if (!(iss >> unitTypeID >> numUnits >> regionID >> orderString >> targetRegionID >> endFrame >> extraInfo)) {
            // Error processing line
			if (line.compare(0, 1, "(") == 0)  listID = 2;
        } else {
// 			searchLog->append(QString("Extracted: %1 %2 %3 %4 %5 %6").arg(unitTypeID).arg(numUnits).arg(regionID).arg(orderString.c_str()).arg(targetRegionID).arg(endFrame));
			// get abstract order
			abstractOrder::abstractOrder order = abstractOrder::getOrder(orderString);
			informationManager->gameState.addGroup(unitTypeID, numUnits, regionID, listID, order, targetRegionID, endFrame);
        }
    }
	infile.close();

	LOG("=====Game state imported:");
	searchLog->append("Game state imported:");
	informationManager->gameState.calculateExpectedEndFrameForAllGroups();
	searchLog->append(QString::fromStdString(informationManager->gameState.toString()));

	EvaluationFunctionBasic ef;
	double reward = ef.evaluate(informationManager->gameState, true);
	searchLog->append(QString("Reward %1").arg(reward));
}

void myQtApp::rolloutGame()
{
	// get ready for the action
	informationManager->gameState.moveForward();

	__int8 nextPlayerInSimultaneousNode = 1;
	int depth = 0;
	ActionGenerator moveGenerator;
	int nextPlayer = -1;
	while (!informationManager->gameState.gameover() && informationManager->gameState._time < 2880 && depth < 1000) {
		// look next player to move
		nextPlayer = -1;
		if (informationManager->gameState.canExecuteAnyAction(true)) {
			if (informationManager->gameState.canExecuteAnyAction(false)) {
				// if both can move: alternate
				nextPlayer = (int)nextPlayerInSimultaneousNode;
				nextPlayerInSimultaneousNode = 1 - nextPlayerInSimultaneousNode;
			} else {
				nextPlayer = 1;
			}
		} else {
			if (informationManager->gameState.canExecuteAnyAction(false)) nextPlayer = 0;
		}
		if (nextPlayer != -1) {
			moveGenerator = ActionGenerator(&informationManager->gameState, nextPlayer == 1);
		} else {
			printError(searchLog, QString("Both players can't move!!!"));
			break;
		}

		// chose random action
		playerActions_t unitsAction = moveGenerator.getRandomAction();
		searchLog->append(QString::fromStdString(moveGenerator.toString(unitsAction)));

		// execute action
		informationManager->gameState.execute(unitsAction, moveGenerator._player);
		searchLog->append("After execute");
		searchLog->append(QString::fromStdString(informationManager->gameState.toString()));
		informationManager->gameState.moveForward();
		searchLog->append("After move forward");
		searchLog->append(QString::fromStdString(informationManager->gameState.toString()));
		depth++;
	}

	searchLog->append(QString("Game state simulated after 2880 frames and depth %1:").arg(depth));
	informationManager->gameState.calculateExpectedEndFrameForAllGroups();
	searchLog->append(QString::fromStdString(informationManager->gameState.toString()));

	EvaluationFunctionBasic ef;
	double reward = ef.evaluate(informationManager->gameState, true);
	searchLog->append(QString("Reward %1").arg(reward));
}

void myQtApp::abcdSearch()
{
	/*
	// now that we have all the units in the game state, compute expected end frame
	informationManager->gameState.expectedEndFrame();
	// and forward until next point decision
	informationManager->gameState.resetFriendlyActions();

	// Search algorithm
	EvaluationFunctionBasic ef;
	ABCD minimax = ABCD(3, &ef);
	LOG("ABCD depth 3");
	//LOG( informationManager->gameState.toString() );
	playerActions_t bestActions = minimax.start(true, informationManager->gameState);

	LOG("Best actions: ");
	//std::map<SquadAgent*, BWAPI::Position> bestOrders;
	int groupID;
	uint8_t orderId;
	uint8_t targetRegionId;
	//BWAPI::Position targetPosition;
	for(playerActions_t::const_iterator i = bestActions.begin(); i!=bestActions.end(); ++i) {
	groupID = (*i).first;
	orderId = (*i).second.first;
	targetRegionId = (*i).second.second;
	LOG("  - Group ID: " << groupID << ", action: " << informationManager->gameState.getAbstractOrderName((int)orderId) << " region: " << (int)targetRegionId);
	//targetPosition = informationManager->gameState.getCenterRegionId((int)targetRegionId);
	//bestOrders[_idToSquad[groupID]] = targetPosition;
	}
	*/
}

void myQtApp::combatSimulator()
{
	// Experiment options
	const bool SKIP_DROPSHIPS = checkSkipDropships->isChecked();
	const bool ONLY_SPARCRAFT_SUPPORTED = checkSkipSparcraft->isChecked();
	const bool SIMULATE_SPARCAFT = checkRunSparcraft->isChecked();
	const int SPARCRAFT_MOVE_LIMIT = moveLimitEdit->text().toInt();
	int index1 = compareSim1->value();
	int index2 = compareSim2->value();

// 	combatLog->append(QString("Dropships %1 SparCraft compatible %2 SparCraft %3 move limit %4").arg(SKIP_DROPSHIPS).arg(ONLY_SPARCRAFT_SUPPORTED).arg(SIMULATE_SPARCAFT).arg(SPARCRAFT_MOVE_LIMIT));
// 	return;

	const int SIMULATIONS_EXPERIMENTS = 11;
#ifndef _DEBUG  // too many experiments on debug mode produce a stack overflow
	const int SPARCRAFT_EXPERIMENTS = 6;
#else
	const int SPARCRAFT_EXPERIMENTS = 1;
#endif
	const int NUMBER_OF_TESTS = SIMULATIONS_EXPERIMENTS + SPARCRAFT_EXPERIMENTS;

	// constructors needed for SparCraft
	SparCraft::init();
	SparCraft::PlayerPtr closest1(new SparCraft::Player_AttackClosest(SparCraft::Players::Player_One));
	SparCraft::PlayerPtr closest2(new SparCraft::Player_AttackClosest(SparCraft::Players::Player_Two));
#ifndef _DEBUG
	SparCraft::PlayerPtr dps1(new SparCraft::Player_AttackDPS(SparCraft::Players::Player_One));
	SparCraft::PlayerPtr dps2(new SparCraft::Player_AttackDPS(SparCraft::Players::Player_Two));
	SparCraft::PlayerPtr weakest1(new SparCraft::Player_AttackWeakest(SparCraft::Players::Player_One));
	SparCraft::PlayerPtr weakest2(new SparCraft::Player_AttackWeakest(SparCraft::Players::Player_Two));
	SparCraft::PlayerPtr nokdps1(new SparCraft::Player_NOKDPS(SparCraft::Players::Player_One));
	SparCraft::PlayerPtr nokdps2(new SparCraft::Player_NOKDPS(SparCraft::Players::Player_Two));
	SparCraft::PlayerPtr kiter1(new SparCraft::Player_Kiter(SparCraft::Players::Player_One));
	SparCraft::PlayerPtr kiter2(new SparCraft::Player_Kiter(SparCraft::Players::Player_Two));
	SparCraft::PlayerPtr kiterdps1(new SparCraft::Player_KiterDPS(SparCraft::Players::Player_One));
	SparCraft::PlayerPtr kiterdps2(new SparCraft::Player_KiterDPS(SparCraft::Players::Player_Two));
#endif // !_DEBUG

	// creating timers
	Timer timers[NUMBER_OF_TESTS];
	double totalTimes[NUMBER_OF_TESTS] = { 0 };
	Timer timerSort, timerSortBorda;
	double totalTimeSort = 0;
	double totalTimeSortBorda = 0;

	float jaccardIndex[NUMBER_OF_TESTS] = { 0 };
	double jaccardIndex2[NUMBER_OF_TESTS] = { 0 };
	double jaccardIndex3[NUMBER_OF_TESTS] = { 0 };
	float jaccardCount[NUMBER_OF_TESTS] = { 0 };
	double jaccardCount2[NUMBER_OF_TESTS] = { 0 };
	double jaccardCount3[NUMBER_OF_TESTS] = { 0 };

	// counters
	int combatSkippedMine = 0;
	int combatSkippedDropship = 0;
	int combatSkippedSparCraft = 0;
	int playerNotKilled = 0;
	int wasBetter = 0;
	int wasWorst = 0;
	
	bool hasMines = false;
	bool hasDropships = false;
	bool skipSparCraft = false;

	combatSimBar->setMaximum(learner.allCombats.size());
	combatSimBar->setValue(0);

	DpsLearner::combatInfo combat;
	for (std::size_t k = 0; k < learner.allCombats.size(); ++k) {
		combat = learner.allCombats[k];

		combatSimBar->setValue(combatSimBar->value() + 1);
		GameState simulatorCache;
		GameState simulators[SIMULATIONS_EXPERIMENTS];
		std::vector<SparCraft::Game> sparCraftGames;
		GameState simulatorsSparCraft[SPARCRAFT_EXPERIMENTS];

		// **************************************************************
		// import initState
		// **************************************************************
		GameState initalState(new CombatSimulatorBasic(&learner.unitDPFbwapi, &learner.unitHPbwapi));
		SparCraft::GameState sparCraftInitialState;

		// Add units from data replay to game state
		addUnitsFromReplaytoGameState(combat.armyUnits1, initalState, sparCraftInitialState, 1,
			hasMines, hasDropships, skipSparCraft, SKIP_DROPSHIPS, ONLY_SPARCRAFT_SUPPORTED);

		addUnitsFromReplaytoGameState(combat.armyUnits2, initalState, sparCraftInitialState, 2,
			hasMines, hasDropships, skipSparCraft, SKIP_DROPSHIPS, ONLY_SPARCRAFT_SUPPORTED);
		
		// skip some combats
		if (hasMines) {
			hasMines = false;
			combatSkippedMine++;
			continue;
		}
		if (SKIP_DROPSHIPS && hasDropships) {
			hasDropships = false;
			combatSkippedDropship++;
			continue;
		}
		if (ONLY_SPARCRAFT_SUPPORTED && skipSparCraft) {
			skipSparCraft = false;
			combatSkippedSparCraft++;
			continue;
		}

		initalState.calculateExpectedEndFrameForAllGroups();

		// **************************************************************
		// import finalState
		// **************************************************************
		GameState finalState;
		for (auto army : combat.armyUnitsEnd1) {
			finalState.addUnit(army.second.typeID, 8, 1, abstractOrder::Attack);
		}
		for (auto army : combat.armyUnitsEnd2) {
			finalState.addUnit(army.second.typeID, 8, 2, abstractOrder::Attack);
		}

		// **************************************************************
		// create a game state for each simulator
		// **************************************************************
		simulatorCache = initalState;
		simulators[0] = initalState;
		simulators[0].changeCombatSimulator(new CombatSimulatorBasic(&learner.unitDPFbwapi, &learner.unitHPbwapi));
		simulators[1] = initalState;
		simulators[1].changeCombatSimulator(new CombatSimulatorDPS(&learner.unitTypeDPFbwapi));
		simulators[2] = initalState;
		simulators[2].changeCombatSimulator(new CombatSimulatorDPS(&learner.unitTypeDPF));
		simulators[9] = initalState;
		simulators[9].changeCombatSimulator(new CombatSimulatorBasic(&learner.unitDPF, &learner.unitHPbwapi));

		// Sorted by Score
		GameState initalStateSorted = initalState;
		timerSort.start();
		std::sort(initalStateSorted.army.friendly.begin(), initalStateSorted.army.friendly.end(), sortByScore);
		std::sort(initalStateSorted.army.enemy.begin(), initalStateSorted.army.enemy.end(), sortByScore);
		totalTimeSort += timerSort.stopAndGetTime();

		simulators[3] = initalStateSorted;
		simulators[3].changeCombatSimulator(new CombatSimulatorBasic(&learner.unitDPFbwapi, &learner.unitHPbwapi));
		simulators[4] = initalStateSorted;
		simulators[4].changeCombatSimulator(new CombatSimulatorDPS(&learner.unitTypeDPFbwapi));
		simulators[5] = initalStateSorted;
		simulators[5].changeCombatSimulator(new CombatSimulatorDPS(&learner.unitTypeDPF));

		// Sorted by type priority learned
		sortByType.typePriority = &learner.typePriority;
		GameState initalStateSorted2 = initalState;
		timerSortBorda.start();
		std::sort(initalStateSorted2.army.friendly.begin(), initalStateSorted2.army.friendly.end(), sortByType);
		std::sort(initalStateSorted2.army.enemy.begin(), initalStateSorted2.army.enemy.end(), sortByType);
		totalTimeSortBorda += timerSortBorda.stopAndGetTime();

		simulators[6] = initalStateSorted2;
		simulators[6].changeCombatSimulator(new CombatSimulatorBasic(&learner.unitDPFbwapi, &learner.unitHPbwapi));
		simulators[7] = initalStateSorted2;
		simulators[7].changeCombatSimulator(new CombatSimulatorDPS(&learner.unitTypeDPFbwapi));
		simulators[8] = initalStateSorted2;
		simulators[8].changeCombatSimulator(new CombatSimulatorDPS(&learner.unitTypeDPF));
		simulators[10] = initalStateSorted2;
		simulators[10].changeCombatSimulator(new CombatSimulatorBasic(&learner.unitDPF, &learner.unitHPbwapi));

		// **************************************************************
		// simulate all states with different simulators
		// **************************************************************

		simulatorCache.moveForward(); // to avoid cache issues with timers
		for (int i = 0; i < SIMULATIONS_EXPERIMENTS; ++i) {
			timers[i].start();
			simulators[i].moveForward();
			totalTimes[i] += timers[i].stopAndGetTime();
		}

		if (SIMULATE_SPARCAFT) {
			sparCraftGames.push_back(SparCraft::Game(sparCraftInitialState, closest1, closest2, SPARCRAFT_MOVE_LIMIT));
#ifndef _DEBUG
			sparCraftGames.push_back(SparCraft::Game(sparCraftInitialState, dps1, dps2, SPARCRAFT_MOVE_LIMIT));
			sparCraftGames.push_back(SparCraft::Game(sparCraftInitialState, weakest1, weakest2, SPARCRAFT_MOVE_LIMIT));
			sparCraftGames.push_back(SparCraft::Game(sparCraftInitialState, nokdps1, nokdps2, SPARCRAFT_MOVE_LIMIT));
			sparCraftGames.push_back(SparCraft::Game(sparCraftInitialState, kiter1, kiter2, SPARCRAFT_MOVE_LIMIT));
			sparCraftGames.push_back(SparCraft::Game(sparCraftInitialState, kiterdps1, kiterdps2, SPARCRAFT_MOVE_LIMIT));
#endif

			if (!skipSparCraft) {
				for (int i = 0; i < SPARCRAFT_EXPERIMENTS; ++i) {
					timers[SIMULATIONS_EXPERIMENTS + i].start();
					sparCraftGames[i].play();
					totalTimes[SIMULATIONS_EXPERIMENTS + i] += timers[SIMULATIONS_EXPERIMENTS + i].stopAndGetTime();

					if (sparCraftGames[i].getState().numUnits(SparCraft::Players::Player_One) > 0 &&
						sparCraftGames[i].getState().numUnits(SparCraft::Players::Player_Two) > 0) {
						playerNotKilled++; // both players still alive
					}
				}
			}
		}

		// **************************************************************
		// compare states against finalState
		// **************************************************************
		for (int i = 0; i < SIMULATIONS_EXPERIMENTS; ++i) {
			jaccardIndex[i] += finalState.getJaccard(simulators[i]);
			jaccardCount[i]++;
			jaccardIndex2[i] += finalState.getJaccard2(initalState, simulators[i], false);
			jaccardCount2[i]++;
			jaccardIndex3[i] += finalState.getJaccard2(initalState, simulators[i], true);
			jaccardCount3[i]++;
		}
		// logs to calculate the p-value
// 		LOG(finalState.getJaccard2(initalState, simulators[0], false) << "," <<
// 			finalState.getJaccard2(initalState, simulators[1], false) << "," <<
// 			finalState.getJaccard2(initalState, simulators[2], false) << "," <<
// 			finalState.getJaccard2(initalState, simulators[3], false) << "," <<
// 			finalState.getJaccard2(initalState, simulators[4], false) << "," <<
// 			finalState.getJaccard2(initalState, simulators[5], false) << "," <<
// 			finalState.getJaccard2(initalState, simulators[6], false) << "," <<
// 			finalState.getJaccard2(initalState, simulators[7], false) << "," <<
// 			finalState.getJaccard2(initalState, simulators[8], false));

		if (SIMULATE_SPARCAFT) {
			if (!skipSparCraft) {
				for (int i = 0; i < SPARCRAFT_EXPERIMENTS; ++i) {
					simulatorsSparCraft[i] = getGameState(sparCraftGames[i].getState());
					jaccardIndex[SIMULATIONS_EXPERIMENTS + i] += finalState.getJaccard(simulatorsSparCraft[i]);
					jaccardCount[SIMULATIONS_EXPERIMENTS + i]++;
					jaccardIndex2[SIMULATIONS_EXPERIMENTS + i] += finalState.getJaccard2(initalState, simulatorsSparCraft[i], false);
					jaccardCount2[SIMULATIONS_EXPERIMENTS + i]++;
					jaccardIndex3[SIMULATIONS_EXPERIMENTS + i] += finalState.getJaccard2(initalState, simulatorsSparCraft[i], true);
					jaccardCount3[SIMULATIONS_EXPERIMENTS + i]++;
				}
			} else {
				skipSparCraft = false;
				combatSkippedSparCraft++;
			}
		}

		// **************************************************************
		// compare 2 simulators
		// **************************************************************
		float jaccard1 = finalState.getJaccard(simulators[index1]);
		float jaccard2 = finalState.getJaccard(simulators[index2]);
// 		if (jaccard1 != jaccard2) break;
		if (jaccard1 > jaccard2) {
			++wasBetter;
			combatLog2->append(QString("Combat index: %1").arg(k));
			combatLog2->append("Initial State");
			combatLog2->append(QString::fromStdString(initalState.toString()));
			combatLog2->append("Final State");
			combatLog2->append(QString::fromStdString((finalState.toString())));
			combatLog2->append(QString("Simulation %1 jaccard (%2)").arg(index1).arg(jaccard1));
			combatLog2->append(QString::fromStdString((simulators[index1].toString())));
			combatLog2->append(QString("Simulation %1 jaccard (%2)").arg(index2).arg(jaccard2));
			combatLog2->append(QString::fromStdString((simulators[index2].toString())));
			combatLog2->append("\n\n");
		} else if (jaccard1 < jaccard2) {
			++wasWorst;
		}

	}

	int allCombatsSize = learner.allCombats.size();
	int analyzedCombats = allCombatsSize - combatSkippedMine - combatSkippedDropship;
	if (ONLY_SPARCRAFT_SUPPORTED) analyzedCombats -= combatSkippedSparCraft;
	combatLog->append(QString("Analyzed %1 of %2 combats").arg(analyzedCombats).arg(allCombatsSize));
	combatLog->append(QString("Reasons to skip:"));
	combatLog->append(QString("%1 combats with Vulture mines").arg(combatSkippedMine));
	if (SKIP_DROPSHIPS) {
		combatLog->append(QString("%1 combats with Dropships").arg(combatSkippedDropship));
	}
	if (ONLY_SPARCRAFT_SUPPORTED) {
		combatLog->append(QString("%1 combats not Compatible with SparCraft").arg(combatSkippedSparCraft));
	}
	if (SIMULATE_SPARCAFT) {
		if (!ONLY_SPARCRAFT_SUPPORTED) {
			combatLog->append(QString("%1 SparCraft simulations").arg(analyzedCombats - combatSkippedSparCraft));
		}
		if (playerNotKilled > 0) {
			printError(combatLog, QString("%1 SparCraft games where both players are still alive").arg(playerNotKilled));
		}
	}
	combatLog->append("\n");

	// index1 was better than index2
	combatLog->append(QString("%1 times simulator %2 was better than %3").arg(wasBetter).arg(index1).arg(index2));
	combatLog->append(QString("%1 times simulator %2 was worst than %3").arg(wasWorst).arg(index1).arg(index2));
	combatLog->append("\n\n");

	// Calculate Avg Jaccard distance
	double jaccardDis[NUMBER_OF_TESTS] = { 0 };
	double jaccardDis2[NUMBER_OF_TESTS] = { 0 };
	double jaccardDis3[NUMBER_OF_TESTS] = { 0 };
	for (int i = 0; i < NUMBER_OF_TESTS; ++i) {
		jaccardDis[i] = jaccardIndex[i] / jaccardCount[i];
		jaccardDis2[i] = jaccardIndex2[i] / jaccardCount2[i];
		jaccardDis3[i] = jaccardIndex3[i] / jaccardCount3[i];
	}

	QString htmlString = "<table><tr><td>Simulator</td><td>Jaccard vanilla</td><td>Jaccard weight</td><td>Jaccard killScore</td><td>Total time</td></tr>";
	htmlString += QString("<tr><td>[0]Basic</td><td>%1</td><td>%2</td><td>%3</td><td>%4</td></tr>").arg(jaccardDis[0]).arg(jaccardDis2[0]).arg(jaccardDis3[0]).arg(totalTimes[0]);
	htmlString += QString("<tr><td>[1]DPF</td><td>%1</td><td>%2</td><td>%3</td><td>%4</td></tr>").arg(jaccardDis[1]).arg(jaccardDis2[1]).arg(jaccardDis3[1]).arg(totalTimes[1]);
	htmlString += QString("<tr><td>[2]DPF learn</td><td>%1</td><td>%2</td><td>%3</td><td>%4</td></tr>").arg(jaccardDis[2]).arg(jaccardDis2[2]).arg(jaccardDis3[2]).arg(totalTimes[2]);
	htmlString += QString("<tr><td>[9]Basic learn</td><td>%1</td><td>%2</td><td>%3</td><td>%4</td></tr>").arg(jaccardDis[9]).arg(jaccardDis2[9]).arg(jaccardDis3[9]).arg(totalTimes[9]);
	htmlString += QString("<tr><td colspan=\"5\"><b>Sorted by score (%1)</b></td></tr>").arg(totalTimeSort);
	htmlString += QString("<tr><td>[3]Basic</td><td>%1</td><td>%2</td><td>%3</td><td>%4</td></tr>").arg(jaccardDis[3]).arg(jaccardDis2[3]).arg(jaccardDis3[3]).arg(totalTimes[3]);
	htmlString += QString("<tr><td>[4]DPF</td><td>%1</td><td>%2</td><td>%3</td><td>%4</td></tr>").arg(jaccardDis[4]).arg(jaccardDis2[4]).arg(jaccardDis3[4]).arg(totalTimes[4]);
	htmlString += QString("<tr><td>[5]DPF learn</td><td>%1</td><td>%2</td><td>%3</td><td>%4</td></tr>").arg(jaccardDis[5]).arg(jaccardDis2[5]).arg(jaccardDis3[5]).arg(totalTimes[5]);
	htmlString += QString("<tr><td colspan=\"5\"><b>Sorted by Borda Count (%1)</b></td></tr>").arg(totalTimeSortBorda);
	htmlString += QString("<tr><td>[6]Basic</td><td>%1</td><td>%2</td><td>%3</td><td>%4</td></tr>").arg(jaccardDis[6]).arg(jaccardDis2[6]).arg(jaccardDis3[6]).arg(totalTimes[6]);
	htmlString += QString("<tr><td>[7]DPF</td><td>%1</td><td>%2</td><td>%3</td><td>%4</td></tr>").arg(jaccardDis[7]).arg(jaccardDis2[7]).arg(jaccardDis3[7]).arg(totalTimes[7]);
	htmlString += QString("<tr><td>[8]DPF learn</td><td>%1</td><td>%2</td><td>%3</td><td>%4</td></tr>").arg(jaccardDis[8]).arg(jaccardDis2[8]).arg(jaccardDis3[8]).arg(totalTimes[8]);
	htmlString += QString("<tr><td>[10]Basic learn</td><td>%1</td><td>%2</td><td>%3</td><td>%4</td></tr>").arg(jaccardDis[10]).arg(jaccardDis2[10]).arg(jaccardDis3[10]).arg(totalTimes[10]);

	if (SIMULATE_SPARCAFT) {
		int offset = SIMULATIONS_EXPERIMENTS;
		htmlString += "<tr><td colspan=\"5\"><b>SparCraft</b></td></tr>";
		htmlString += QString("<tr><td>closest</td><td>%1</td><td>%2</td><td>%3</td><td>%4</td></tr>").arg(jaccardDis[offset]).arg(jaccardDis2[offset]).arg(jaccardDis3[offset]).arg(totalTimes[offset]);
#ifndef _DEBUG
		htmlString += QString("<tr><td>weakest</td><td>%1</td><td>%2</td><td>%3</td><td>%4</td></tr>").arg(jaccardDis[offset + 2]).arg(jaccardDis2[offset + 2]).arg(jaccardDis3[offset + 2]).arg(totalTimes[offset + 2]);
		htmlString += QString("<tr><td>DPS</td><td>%1</td><td>%2</td><td>%3</td><td>%4</td></tr>").arg(jaccardDis[offset + 1]).arg(jaccardDis2[offset + 1]).arg(jaccardDis3[offset + 1]).arg(totalTimes[offset + 1]);
		htmlString += QString("<tr><td>NOKDPS</td><td>%1</td><td>%2</td><td>%3</td><td>%4</td></tr>").arg(jaccardDis[offset + 3]).arg(jaccardDis2[offset + 3]).arg(jaccardDis3[offset + 3]).arg(totalTimes[offset + 3]);
		htmlString += QString("<tr><td>kiter</td><td>%1</td><td>%2</td><td>%3</td><td>%4</td></tr>").arg(jaccardDis[offset + 4]).arg(jaccardDis2[offset + 4]).arg(jaccardDis3[offset + 4]).arg(totalTimes[offset + 4]);
		htmlString += QString("<tr><td>kiter DPS</td><td>%1</td><td>%2</td><td>%3</td><td>%4</td></tr>").arg(jaccardDis[offset + 5]).arg(jaccardDis2[offset + 5]).arg(jaccardDis3[offset + 5]).arg(totalTimes[offset + 5]);
#endif
	}

	htmlString += "</table><br>";
	combatLog->insertHtml(htmlString);
}

void myQtApp::addUnitsFromReplaytoGameState(std::map<int, DpsLearner::unitInfo> armyUnits, GameState &myGameState, SparCraft::GameState &sparCraftState, 
	int playerID, bool &hasMines, bool &hasDropships, bool &skipSparCraft,
	bool SKIP_DROPSHIPS, bool ONLY_SPARCRAFT_SUPPORTED)
{
	SparCraft::IDType sparCraftPlayerID = SparCraft::Players::Player_One;
	if (playerID == 2) sparCraftPlayerID = SparCraft::Players::Player_Two;

	for (auto army : armyUnits) {
		if (army.second.typeID == BWAPI::UnitTypes::Enum::Terran_Vulture_Spider_Mine) {
			hasMines = true;
			break;
		}
		if (SKIP_DROPSHIPS) {
			if (army.second.typeID == BWAPI::UnitTypes::Enum::Terran_Dropship) {
				hasDropships = true;
				break;
			}
		}
		// the region (8) is arbitrary, but should be the same
		myGameState.addUnit(army.second.typeID, 8, playerID, abstractOrder::Attack);
		//SparCraft doesn't support all units
		if (skipSparCraft || !SparCraft::System::isSupportedUnitType(army.second.typeID)
			|| army.second.typeID == BWAPI::UnitTypes::Enum::Terran_Dropship
			|| army.second.typeID == BWAPI::UnitTypes::Enum::Terran_Missile_Turret) {
			skipSparCraft = true;
			if (ONLY_SPARCRAFT_SUPPORTED) break;
		} else {
			sparCraftState.addUnit(army.second.typeID, sparCraftPlayerID, SparCraft::Position(army.second.x, army.second.y));
		}
	}
}

// translate SparCraft gameState to ours in order to compare states
GameState myQtApp::getGameState(SparCraft::GameState sparCraftGameState)
{
	GameState myGameState;
	for (SparCraft::UnitCountType u(0); u < sparCraftGameState.numUnits(SparCraft::Players::Player_One); ++u) {
		const SparCraft::Unit & unit(sparCraftGameState.getUnit(SparCraft::Players::Player_One, u));
		myGameState.addUnit(unit.typeID(), 8, 1, abstractOrder::Attack);
	}
	for (SparCraft::UnitCountType u(0); u < sparCraftGameState.numUnits(SparCraft::Players::Player_Two); ++u) {
		const SparCraft::Unit & unit(sparCraftGameState.getUnit(SparCraft::Players::Player_Two, u));
		myGameState.addUnit(unit.typeID(), 8, 2, abstractOrder::Attack);
	}
	return myGameState;
}

void myQtApp::printError(QTextEdit * textEdit, const QString & text)
{
	// save    
	int fw = textEdit->fontWeight();
	QColor tc = textEdit->textColor();
	// append
	textEdit->setFontWeight(QFont::DemiBold);
	textEdit->setTextColor(QColor("red"));
	textEdit->append(text);
	// restore
	textEdit->setFontWeight(fw);
	textEdit->setTextColor(tc);
}

void myQtApp::parseCombats()
{
	QString datasetPath = combatDatasetLine->text();
	combatLog->append(QString("Analyzing file %1").arg(datasetPath));
	combatLog->repaint();
	learner.parseReplayData(datasetPath.toStdString());

	combatLog->append(QString("%1 combats ended by army destroyed").arg(learner.armyDestroyed));
	combatLog->append(QString("%1 combats ended by reinforcement").arg(learner.armyReinforcement));
	combatLog->append(QString("%1 combats ended by peace").arg(learner.armyPeace));
	combatLog->append(QString("%1 combats ended by game end").arg(learner.armyGameEnd));
	combatLog->append(QString("%1 combats parsed").arg(learner.allCombats.size()));
	if (learner.armyKilledWithoutKills > 0)
		printError(combatLog, QString("%1 combats ended by army destroyed without kills").arg(learner.armyKilledWithoutKills));
	if (learner.moreThanTwoArmies > 0)
		printError(combatLog, QString("%1 combats with more than 2 armies").arg(learner.moreThanTwoArmies));

	combatsParsedLabel->setText(QString("%1").arg(learner.allCombats.size()));
	learnButton->setEnabled(true);
	combatClearButton->setEnabled(true);
	crossValidationButton->setEnabled(true);
}

void myQtApp::learnCombat()
{
	// be sure the learn stuff is clear
	learner.clear();

	// Learn DPF from combat replays data
	learner.calculateDPS(DPF_noTransport->isChecked(), DPF_1type->isChecked());
	if (checkBWAPIupperBound->isChecked()) {
		learner.damageUpperBound();
	} else {
		// cleaning stats
		learner.DPFbounded.clear();
		for (int matchupInt = DpsLearner::Matchups::TVT; matchupInt != DpsLearner::Matchups::NONE; matchupInt++) {
			learner.DPFbounded.insert(std::pair<DpsLearner::Matchups, int>(static_cast<DpsLearner::Matchups>(matchupInt), 0));
		}
	}
	combatLog->append(QString("DPS calculated from %1 combats").arg(learner.combatsProcessed));
	
	for (int matchupInt = DpsLearner::Matchups::TVT; matchupInt != DpsLearner::Matchups::NONE; matchupInt++) {
		DpsLearner::Matchups matchId = static_cast<DpsLearner::Matchups>(matchupInt);
		if (learner.DPFcases[matchId].noCases > 0) {
			printError(combatLog, QString("[%1] Detected %2 DPF cases (%3 missing, %4 bounded, %5 learned)")
				.arg(learner.getMatchupName(matchId).c_str())
				.arg(learner.DPFcases[matchId].size - learner.DPFcases[matchId].noCases)
				.arg(learner.DPFcases[matchId].noCases)
				.arg(learner.DPFbounded[matchId])
				.arg(learner.DPFcases[matchId].size - learner.DPFcases[matchId].noCases - learner.DPFbounded[matchId])
				);
		}
	}
	
	
	// Learn target selection from combat replays data
	if (firstGroupKillButton->isChecked()) {
		learner.calculatePriorityFirstBorda();
	} else {
		learner.calculatePriorityLastBorda();
	}

	getDPFbutton->setEnabled(true);
	combatSimulatorButton->setEnabled(true);
}

void myQtApp::clearCombatsParsed()
{
	learner.allCombats.clear();
	learner.clear();
	learnButton->setEnabled(false);
	combatClearButton->setEnabled(false);

	getDPFbutton->setEnabled(false);
	combatSimulatorButton->setEnabled(false);
	crossValidationButton->setEnabled(false);

	combatsParsedLabel->setText("0");
}

void myQtApp::getDPF()
{
	int unitType1 = (comboDPF1->itemData(comboDPF1->currentIndex())).toInt();
	int unitType2 = (comboDPF2->itemData(comboDPF2->currentIndex())).toInt();

	combatLog->append(QString("Getting DPF of %1(%2) vs %3(%4)").arg(comboDPF1->currentText()).arg(unitType1).arg(comboDPF2->currentText()).arg(unitType2));
	
	DPF_learned->setText(QString("%1").arg(learner.unitTypeDPF[unitType1][unitType2].DPF));
	DPF_BWAPI->setText(QString("%1").arg(learner.unitTypeDPFbwapi[unitType1][unitType2].DPF));

	DPF_learned_2->setText(QString("%1").arg(learner.unitTypeDPF[unitType2][unitType1].DPF));
	DPF_BWAPI_2->setText(QString("%1").arg(learner.unitTypeDPFbwapi[unitType2][unitType1].DPF));

	BWAPI::UnitType bwapiType(unitType1);
	HP_label1->setText(QString("%1 (%2)").arg(bwapiType.maxHitPoints()).arg(bwapiType.maxShields()));
	BWAPI::UnitType bwapiType2(unitType2);
	HP_label2->setText(QString("%1 (%2)").arg(bwapiType2.maxHitPoints()).arg(bwapiType2.maxShields()));
}

// after parsing the combats we can perform a 10-fold cross validation
void myQtApp::crossValidation()
{
	// filter combats
	// **************************************************************
	std::vector<DpsLearner::combatInfo> combatsFiltered;
	bool skipCombat;
	for (auto combat : learner.allCombats) {
		skipCombat = false;
		for (auto army : combat.armyUnits1) {
			if (army.second.typeID == BWAPI::UnitTypes::Enum::Terran_Vulture_Spider_Mine) {
				skipCombat = true;
				break;
			}
			if (army.second.typeID == BWAPI::UnitTypes::Enum::Terran_Dropship) {
				skipCombat = true;
				break;
			}
		}
		if (!skipCombat) combatsFiltered.push_back(combat);
	}

	combatLog->append(QString("%1 combats dataset").arg(combatsFiltered.size()));

	// calculate on what fold each index belongs
	// **************************************************************
	const int k = 10; // number of folds
	const int size = combatsFiltered.size();

	int* indices = new int[size];

	for (int i = 0; i < size; i++)
		indices[i] = i%k;

	// shuffle Array using http://en.wikipedia.org/wiki/Fisher%E2%80%93Yates_shuffle
	int n = size;
	while (n > 1) {
		int k = rand() % n;
		n--; // n is now the last pertinent index
		std::swap(indices[n], indices[k]);
	}

	const int NUMBER_OF_TESTS = 6;
	double jaccardAverage[NUMBER_OF_TESTS] = { 0 };
	double jaccardCount[NUMBER_OF_TESTS] = { 0 };

	// perform the cross validation
	// **************************************************************
	for (int testingIndex = 0; testingIndex < k; ++testingIndex) {

		// train all combats except testing Index
		// **************************************************************

		// Learn DPF from combat replays data
		learner.calculateDPS(combatsFiltered, indices, testingIndex);
		combatLog->append(QString("DPS learned from %1 combats for fold %2").arg(learner.combatsProcessed).arg(testingIndex));

// 		for (int matchupInt = DpsLearner::Matchups::TVT; matchupInt != DpsLearner::Matchups::NONE; matchupInt++) {
// 			DpsLearner::Matchups matchId = static_cast<DpsLearner::Matchups>(matchupInt);
// 			if (learner.DPFcases[matchId].noCases > 0) {
// 				printError(combatLog, QString("[%1] Detected %2 DPF cases (%3 missing)")
// 					.arg(learner.getMatchupName(matchId).c_str())
// 					.arg(learner.DPFcases[matchId].size - learner.DPFcases[matchId].noCases)
// 					.arg(learner.DPFcases[matchId].noCases)
// 					);
// 			}
// 		}

		// Learn target selection from combat replays data
		learner.calculatePriorityFirstBorda();






		// simulate only testing Index combats 
		// **************************************************************
		double jaccardIndex2[NUMBER_OF_TESTS] = { 0 };
		double jaccardCount2[NUMBER_OF_TESTS] = { 0 };

		// counters
		int playerNotKilled = 0;

		DpsLearner::combatInfo combat;
		for (std::size_t n = 0; n < combatsFiltered.size(); ++n) {
			if (indices[n] != testingIndex) continue;
			combat = combatsFiltered[n];
			GameState simulators[NUMBER_OF_TESTS];

			// **************************************************************
			// import initState
			// **************************************************************
			GameState initalState(new CombatSimulatorBasic(&learner.unitDPFbwapi, &learner.unitHPbwapi));

			// Add units from data replay to game state
			for (auto army : combat.armyUnits1) {
				initalState.addUnit(army.second.typeID, 8, 1, abstractOrder::Attack);
			}
			for (auto army : combat.armyUnits2) {
				initalState.addUnit(army.second.typeID, 8, 2, abstractOrder::Attack);
			}
			initalState.calculateExpectedEndFrameForAllGroups();


			// **************************************************************
			// import finalState
			// **************************************************************
			GameState finalState;
			for (auto army : combat.armyUnitsEnd1) {
				finalState.addUnit(army.second.typeID, 8, 1, abstractOrder::Attack);
			}
			for (auto army : combat.armyUnitsEnd2) {
				finalState.addUnit(army.second.typeID, 8, 2, abstractOrder::Attack);
			}

			// **************************************************************
			// create a game state for each simulator
			// **************************************************************
			simulators[0] = initalState;
			simulators[0].changeCombatSimulator(new CombatSimulatorBasic(&learner.unitDPFbwapi, &learner.unitHPbwapi));
			simulators[1] = initalState;
			simulators[1].changeCombatSimulator(new CombatSimulatorDPS(&learner.unitTypeDPFbwapi));

			// Sorted by Score
			GameState initalStateSorted = initalState;
			std::sort(initalStateSorted.army.friendly.begin(), initalStateSorted.army.friendly.end(), sortByScore);
			std::sort(initalStateSorted.army.enemy.begin(), initalStateSorted.army.enemy.end(), sortByScore);

			simulators[2] = initalStateSorted;
			simulators[2].changeCombatSimulator(new CombatSimulatorBasic(&learner.unitDPFbwapi, &learner.unitHPbwapi));
			simulators[3] = initalStateSorted;
			simulators[3].changeCombatSimulator(new CombatSimulatorDPS(&learner.unitTypeDPFbwapi));

			// Sorted by type priority learned
			sortByType.typePriority = &learner.typePriority;
			GameState initalStateSorted2 = initalState;
			std::sort(initalStateSorted2.army.friendly.begin(), initalStateSorted2.army.friendly.end(), sortByType);
			std::sort(initalStateSorted2.army.enemy.begin(), initalStateSorted2.army.enemy.end(), sortByType);

			simulators[4].army.friendly.clear();
			simulators[4].army.enemy.clear();
			simulators[4] = initalStateSorted2;
			simulators[4].changeCombatSimulator(new CombatSimulatorBasic(&learner.unitDPF, &learner.unitHPbwapi));
			simulators[5] = initalStateSorted2;
			simulators[5].changeCombatSimulator(new CombatSimulatorDPS(&learner.unitTypeDPF));
			

			// **************************************************************
			// simulate all states with different simulators
			// **************************************************************

			for (int i = 0; i < NUMBER_OF_TESTS; ++i) {
				simulators[i].moveForward();
			}

			// **************************************************************
			// compare states against finalState
			// **************************************************************
			for (int i = 0; i < NUMBER_OF_TESTS; ++i) {
				jaccardIndex2[i] += finalState.getJaccard2(initalState, simulators[i], false);
				jaccardCount2[i]++;
			}
		}


		// Calculate Avg Jaccard distance
		double jaccardDis2[NUMBER_OF_TESTS] = { 0 };
		for (int i = 0; i < NUMBER_OF_TESTS; ++i) {
			jaccardDis2[i] = jaccardIndex2[i] / jaccardCount2[i];
		}

// 		QString htmlString = "<table><tr><td>Simulator</td><td>DPF,Random</td><td>DPF, KS</td><td>DPF learn, TS learn</td></tr>";
// 		htmlString += QString("<tr><td>simDPFsus</td><td>%1</td><td>%2</td><td>%3</td></tr>").arg(jaccardDis2[0]).arg(jaccardDis2[2]).arg(jaccardDis2[4]);
// 		htmlString += QString("<tr><td>simDPFdec</td><td>%1</td><td>%2</td><td>%3</td></tr>").arg(jaccardDis2[1]).arg(jaccardDis2[3]).arg(jaccardDis2[5]);
// 		htmlString += "</table><br>";
// 		combatLog->insertHtml(htmlString);

		for (int i = 0; i < NUMBER_OF_TESTS; ++i) {
			++jaccardCount[i];
			jaccardAverage[i] += (jaccardDis2[i] - jaccardAverage[i]) / jaccardCount[i];
			
		}
	}

	combatLog->append("Final Result:");
	QString htmlString = "<table><tr><td>Simulator</td><td>DPF,Random</td><td>DPF, KS</td><td>DPF learn, TS learn</td></tr>";
	htmlString += QString("<tr><td>simDPFsus</td><td>%1</td><td>%2</td><td>%3</td></tr>").arg(jaccardAverage[0]).arg(jaccardAverage[2]).arg(jaccardAverage[4]);
	htmlString += QString("<tr><td>simDPFdec</td><td>%1</td><td>%2</td><td>%3</td></tr>").arg(jaccardAverage[1]).arg(jaccardAverage[3]).arg(jaccardAverage[5]);
	htmlString += "</table><br>";
	combatLog->insertHtml(htmlString);

	delete indices;
}