/*	-----------------------------------------------------------------------------
	M A A S C R A F T

	StarCraft: Brood War - Bot

	Author: Dennis Soemers
	Maastricht University
	-----------------------------------------------------------------------------
*/

#include "CommonIncludes.h"

#include "Base.h"
#include "BaseLocation.h"
#include "BaseManager.h"
#include "Distances.h"
#include "MathConstants.h"
#include "OpponentTracker.h"
#include "StrategyManager.h"
#include "Tasks/GatherResourceTask.hpp"
#include "Tasks/WorkerDefendBase.hpp"
#include "UnitOwner.h"
#include "UnitTracker.h"
#include "WorkerManager.h"

/*
	Implementation of the WorkerManager
*/

using namespace BWAPI;
using namespace std;

void WorkerManager::onFrame()
{
	// loop through defending workers and remove them if they have died
	Unitset copyDefenders = Unitset(currentDefenders);

	for(auto it = copyDefenders.begin(); it != copyDefenders.end(); ++it)
	{
		Unit u = *it;

		if(u->getHitPoints() <= 0)
			currentDefenders.erase(u);
	}

	const Unitset& workers = getOwnedUnits();

	for(auto it = workers.begin(); it != workers.end(); ++it)
	{
		Unit u = *it;
		if(unitTasksMap.count(u) == 0)
			continue;

		TaskSet& tasks = unitTasksMap[u];

		tasks.update();
		Task* currentTask = tasks.getCurrentTask();

		Unitset inRange = UnitUtils::getUnitsInWeaponRange(u, u->getType().groundWeapon(), Filter::IsEnemy && Filter::IsDetected);

		if(!inRange.empty())
		{
			Behaviours::combatBehaviourFight(u);
		}
		else if(currentTask)
		{
			currentTask->execute();
		}
		else
		{
			Task* newTask = requestNewWorkerTask(u);

			if(newTask)
				tasks.addTask(newTask);
			else
			{
				newTask = StrategyManager::Instance()->requestNewWorkerTask(u);

				if(newTask)
					tasks.addTask(newTask);
			}

			if(newTask)		// try once more to immediately execute
			{
				tasks.update();
				currentTask = tasks.getCurrentTask();

				if(currentTask)
					currentTask->execute();
			}
		}
	}
}

void WorkerManager::baseUnderAttack(const Base& base)
{
	Unitset opponents = OpponentTracker::Instance()->getOpponentsInRegion(base.getLocation()->getRegion());

	defendingProbesNeeded = 1;

	for(auto it = opponents.begin(); it != opponents.end(); ++it)
	{
		Unit unit = *it;
		UnitType type = unit->getType();

		if(type.isFlyer())
			continue;

		if(type.isBuilding())
			defendingProbesNeeded += 10;
		else if(!unit->isAttacking())
			continue;

		defendingProbesNeeded++;
	}

	const Unitset& workers = getOwnedUnits();

	for(auto it = workers.begin(); it != workers.end(); ++it)
	{
		Unit u = *it;

		TaskSet& tasks = unitTasksMap[u];

		if(currentDefenders.size() < defendingProbesNeeded)
		{
			tasks.addTask(new WorkerDefendBase(Priorities::PRIORITY_BASE_DEFENSE, base, u));
			currentDefenders.push_back(u);
		}
		else
		{
			break;
		}
	}
}

void WorkerManager::noDefenseNeeded()
{
	defendingProbesNeeded = 0;
	currentDefenders.clear();
}

const size_t WorkerManager::numDefendersNeeded()
{
	return defendingProbesNeeded;
}

void WorkerManager::receiveOwnership(const Unit unit)
{
	if(!unit)
		return;

#ifdef MAASCRAFT_DEBUG
	if(!unit->getType().isWorker())
	{
		LOG_WARNING(StringBuilder() << "WorkerManager received ownership of a non-Worker unit! Type = " << unit->getType())
		return;
	}
#endif

	UnitOwner::receiveOwnership(unit);
	unitTasksMap[unit];
	lastWorkerReceived = unit;
}

void WorkerManager::removeOwnership(const Unit unit)
{
#ifdef MAASCRAFT_DEBUG
	if(!unit->getType().isWorker())
	{
		LOG_WARNING(StringBuilder() << "WorkerManager lost ownership of a non-Worker unit! Type = " << unit->getType())
		return;
	}
#endif

	UnitOwner::removeOwnership(unit);
	if(unitTasksMap.count(unit) != 0)
	{
		Task* currentTask = unitTasksMap[unit].getCurrentTask();
		if(currentTask)
			currentTask->onEnd();

		unitTasksMap.erase(unit);
	}

	if(unit == lastWorkerReceived)
		lastWorkerReceived = nullptr;
}

Task* WorkerManager::requestNewWorkerTask(Unit worker)
{
	// best candidate resource is a resource node which is the closest node
	// to one of our bases' depots which does not have the max number of workers
	// assigned yet
	Unit bestCandidateResource = nullptr;
	int minDistSq = MathConstants::MAX_INT;
	const vector<Base>& bases = BaseManager::Instance()->getSelfBases();

	for(auto base_it = bases.begin(); base_it != bases.end(); ++base_it)
	{
		const BaseLocation* loc = (*base_it).getLocation();
		const Unitset& resources = loc->getResources();

		const Position depotCenter = (*base_it).getDepotCenter();

		for(auto resource_it = resources.begin(); resource_it != resources.end(); ++resource_it)
		{
			Unit resource = *resource_it;
			UnitType type = resource->getType();

			if(type.isMineralField())
			{
				if(UnitTracker::Instance()->getNumAssignedWorkers(resource) < std::ceil(Options::ResourceGathering::WORKERS_PER_MINERAL_PATCH))
				{
					const int distSq = Distances::getSquaredDistance(depotCenter, resource);

					if(distSq < minDistSq)
					{
						minDistSq = distSq;
						bestCandidateResource = resource;
					}
				}
			}
			else if(type == UnitTypes::Protoss_Assimilator && resource->getPlayer() == Broodwar->self())
			{
				if(UnitTracker::Instance()->getNumAssignedWorkers(resource) < Options::ResourceGathering::WORKERS_PER_VESPENE_GAS_GEYSER)
				{
					const int distSq = Distances::getSquaredDistance(depotCenter, resource);

					if(distSq < minDistSq)
					{
						minDistSq = distSq;
						bestCandidateResource = resource;
					}
				}
			}
		}
	}

	if(bestCandidateResource)		// not nullptr, so found a resource node to gather
		return new GatherResourceTask(worker, bestCandidateResource);
	else
	{
		MapRegion* workerRegion = MapAnalyser::Instance()->getRegion(worker->getPosition());
		Unitset opponents = OpponentTracker::Instance()->getOpponentsInRegion(workerRegion);

		if(!opponents.empty())
			worker->attack(opponents.front());
		else
		{
			Unit mineral = Broodwar->getClosestUnit(worker->getPosition(), Filter::IsMineralField);
			worker->gather(mineral);
		}
	}

	return nullptr;
}

Unit WorkerManager::requestBuilder(const BWAPI::TilePosition buildPos)
{
	// first try the worker we received control of last, it is likely he was previously a builder and still isn't doing anything useful
	if(lastWorkerReceived)
	{
		if(!(lastWorkerReceived->isCarryingGas() || lastWorkerReceived->isCarryingMinerals()) &&
				currentDefenders.find(lastWorkerReceived) == currentDefenders.end())
		{
			TaskSet& taskSet = unitTasksMap[lastWorkerReceived];

			if(!taskSet.getCurrentTask() || taskSet.getCurrentTask()->allowInterruption())		// make sure we're allowed to interrupt any current tasks
			{
				// this guy is valid, so return him
				return lastWorkerReceived;
			}
		}
	}

	// find Worker who is not currently carrying resources closest to build position
	const Unitset& workers = UnitTracker::Instance()->getSelfWorkerUnits();
	Unit bestWorker = nullptr;
	int minDistSq = MathConstants::MAX_INT;

	for(auto it = workers.begin(); it != workers.end(); ++it)
	{
		Unit u = *it;
		if(unitTasksMap.count(u) == 0)		// this worker not under control of WorkerManager
			continue;

		if(u->isCarryingGas() || u->isCarryingMinerals())
			continue;

		if(currentDefenders.find(u) != currentDefenders.end())	// this guy is busy defending
			continue;

		if(u->isIdle())		// found a worker which was for some reason idle. No longer care about distances, just take this one
			return u;

		if(u->isMoving())	// dont want workers which are standing still, for example because they're harvesting already
		{
			TaskSet& taskSet = unitTasksMap[u];

			if(taskSet.getCurrentTask() && !taskSet.getCurrentTask()->allowInterruption())		// make sure we're allowed to interrupt any current tasks
				continue;

			const int distSq = Distances::getSquaredDistance(PositionUtils::toPosition(buildPos), u);
			if(distSq < minDistSq)
			{
				minDistSq = distSq;
				bestWorker = u;
			}
		}
	}

	return bestWorker;
}

Unit WorkerManager::requestScout()
{
	// find Worker who is not currently carrying resources
	const Unitset& workers = UnitTracker::Instance()->getSelfWorkerUnits();
	Unit scout = nullptr;

	for(auto it = workers.begin(); it != workers.end(); ++it)
	{
		Unit u = *it;
		if(unitTasksMap.count(u) == 0)		// this worker not under control of WorkerManager
			continue;

		if(u->isCarryingGas() || u->isCarryingMinerals())
			continue;

		if(currentDefenders.find(u) != currentDefenders.end())	// this guy is busy defending
			continue;

		if(u->isIdle() || u->isMoving())
		{
			TaskSet& taskSet = unitTasksMap[u];

			if(taskSet.getCurrentTask() && taskSet.getCurrentTask()->allowInterruption())
			{
				scout = u;
				break;
			}
		}
	}

	return scout;
}

const Unitset WorkerManager::getPotentialTargets(Unit defender) const
{
	Unitset potentialTargets = UnitUtils::getUnitsInRadius(defender, 640, Filter::IsEnemy && Filter::IsDetected);

	return potentialTargets;
}