/*	-----------------------------------------------------------------------------
	M A A S C R A F T

	StarCraft: Brood War - Bot

	Author: Dennis Soemers
	Maastricht University
	-----------------------------------------------------------------------------
*/

/*
	Implementation of UserCommandProcessor.h

	If MAASCRAFT_DEBUG is not defined, the entire module will not be compiled.
*/

#include "CommonIncludes.h"

#ifdef MAASCRAFT_DEBUG

#include "ArmyManager.h"
#include "ArrayUtils.h"
#include "BuildingManager.h"
#include "BWAPI/Color.h"
#include "DebugDrawing.h"
#include "MapAnalyser.h"
#include "MapGraph.h"
#include "MapGraphEdge.h"
#include "MapGraphNode.h"
#include "PathFinder.h"
#include "PositionState.h"
#include "PositionUtils.h"
#include "OpponentTracker.h"
#include "StrategyManager.h"
#include "Timer.hpp"
#include "UserCommandProcessor.h"

using namespace BWAPI;
using namespace std;

void UserCommandProcessor::onFrame()
{
	static Timer frameTimer;
	frameTimer.start();

	const Unitset& selectedUnits = Broodwar->getSelectedUnits();

	for(auto it = selectedUnits.begin(); it != selectedUnits.end(); ++it)
	{
		Unit unit = *it;

		if(DRAW_PATHS)
		{
			forward_list<PositionState> path = PathFinder::findWalkablePath(PositionUtils::toWalkPosition(unit->getPosition()), 
																				PositionUtils::toWalkPosition(MARKED_POSITION)		);

			DebugDrawing::drawPath(path, BWAPI::Colors::Green);
		}
	}

	if(DRAW_BASE_LOCATIONS)
	{
		const vector<BaseLocation*>& baseLocations = MapAnalyser::Instance()->getBaseLocations();

		for(auto it = baseLocations.begin(); it != baseLocations.end(); ++it)
		{
			TilePosition optimalDepotLocation = (*it)->getOptimalDepotLocation();
			if((*it)->isStartLocation())
				DebugDrawing::colorTilePosition(optimalDepotLocation, Colors::Red);
			else
				DebugDrawing::colorTilePosition(optimalDepotLocation, Colors::Yellow);

			if(Broodwar->isVisible(optimalDepotLocation))
				DebugDrawing::linkUnitsToPos(PositionUtils::getCenterPosition(optimalDepotLocation), (*it)->getResources(), Colors::Yellow);
		}
	}

	if(DRAW_CHOKE_POINTS)
	{
		const vector<MapChokePoint*>& chokepoints = MapAnalyser::Instance()->getChokepoints();

		for(auto it = chokepoints.begin(); it != chokepoints.end(); ++it)
		{
			MapChokePoint* choke = (*it);

			Position pos1 = PositionUtils::toPosition(choke->getCenter());
			Position pos2 = PositionUtils::toPosition(choke->getTiles().at(0));
			Position pos3 = PositionUtils::toPosition(choke->getTiles().at(choke->getTiles().size() - 1));

			Broodwar->drawLineMap(pos1, pos2, Colors::White);
			Broodwar->drawLineMap(pos1, pos3, Colors::White);
		}
	}

	if(DRAW_MAP_GRAPH)
	{
		const MapGraph* graph = MapAnalyser::Instance()->getGraph();
		const vector<MapGraphEdge*>& edges = graph->getEdges();
		const vector<MapGraphNode*>& nodes = graph->getNodes();

		for(auto it = nodes.begin(); it != nodes.end(); ++it)
		{
			MapGraphNode* node = *it;
			Position pos = node->getPosition();

			Broodwar->drawCircleMap(pos, 5, Colors::Red, true);
		}

		for(auto it = edges.begin(); it != edges.end(); ++it)
		{
			MapGraphEdge* edge = *it;
			pair<MapGraphNode*, MapGraphNode*> nodes = edge->getNodes();

			Position pos1 = nodes.first->getPosition();
			Position pos2 = nodes.second->getPosition();

			Broodwar->drawLineMap(pos1, pos2, Colors::Red);
		}
	}

	if(DRAW_OPPONENT_CLUSTERS)
	{
		const vector<Unitset>& clusters = OpponentTracker::Instance()->getOpponentClusters();

		DebugDrawing::ColorIterator colorIterator;
		for(auto it = clusters.begin(); it != clusters.end(); ++it)
		{
			Unitset cluster = *it;
			BWAPI::Color color = colorIterator.nextColor();

			for(size_t j = 0; j < cluster.size(); ++j)
			{
				DebugDrawing::drawBoxesAroundUnits(cluster, color);
			}
		}
	}

	if(DRAW_RESERVED)
	{
		int startX = PositionUtils::toTilePosition(Broodwar->getScreenPosition());
		int startY = PositionUtils::toTilePosition(Broodwar->getScreenPosition());

		const int height = Broodwar->mapHeight();
		const int width = Broodwar->mapWidth();
		for(int y = startY; y < height; ++y)
		{
			for(int x = startX; x < width; ++x)
			{
				TilePosition pos = TilePosition(x, y);

				if(BuildingManager::Instance()->isTileReserved(pos))
					DebugDrawing::colorTilePosition(pos, Colors::Red);
				else if(BuildingManager::Instance()->isTileInResourceBox(pos))
					DebugDrawing::colorTilePosition(pos, Colors::Yellow);
			}
		}
	}

	if(DRAW_THREAT_MAP)
	{
		const int height = Broodwar->mapHeight();
		const int width = Broodwar->mapWidth();
		OpponentTracker* opponentTracker = OpponentTracker::Instance();

		for(int y = 0; y < height; ++y)
		{
			for(int x = 0; x < width; ++x)
			{
				if(opponentTracker->getAirThreat(x, y) > 0.0f || opponentTracker->getGroundThreat(x, y) > 0.0f)
					Broodwar->drawTextMap(x*32, y*32, "\x02 %0.2f \x08 \n %0.2f", opponentTracker->getAirThreat(x, y), 
																				opponentTracker->getGroundThreat(x, y));
			}
		}
	}

	if(DRAW_WALKABILITY)
	{
		MapAnalyser* mapAnalyser = MapAnalyser::Instance();
		int startX = PositionUtils::toWalkPosition(Broodwar->getScreenPosition());
		int startY = PositionUtils::toWalkPosition(Broodwar->getScreenPosition());

		const int height = Broodwar->mapHeight() * 4;
		const int width = Broodwar->mapWidth() * 4;
		for(int y = startY; y < height; ++y)
		{
			for(int x = startX; x < width; ++x)
			{
				if(!mapAnalyser->isWalkable(x, y))
				{
					Broodwar->drawBoxMap(x*8, y*8, (x+1)*8, (y+1)*8, Colors::Black, true);
				}
			}
		}
	}

	if(PAUSED)
	{
		while(frameTimer.getElapsedTimeInMilliSec() < 200)
		{/*Run at 5 FPS during ''pauses''*/}
	}
}

void UserCommandProcessor::processCommand(string& command)
{
	if((*command.begin()) != '/')
		return;

	if(command == "/clear")
	{
		DRAW_BASE_LOCATIONS = false;
		DRAW_CHOKE_POINTS = false;
		DRAW_MAP_GRAPH = false;
		DRAW_OPPONENT_CLUSTERS = false;
		DRAW_PATHS = false;
		DRAW_RESERVED = false;
		DRAW_THREAT_MAP = false;
		DRAW_WALKABILITY = false;
	}
	else if(command == "/dumpUnitInfo")
	{
		const Unitset& selected = Broodwar->getSelectedUnits();

		if(selected.empty())
			return;

		Unit u = selected.front();

		LOG_MESSAGE("Dumping Unit info")
		LOG_MESSAGE(StringBuilder() << "canAttack(): " << u->canAttack())
		LOG_MESSAGE(StringBuilder() << "canCommand(): " << u->canCommand())
		LOG_MESSAGE(StringBuilder() << "canMove(): " << u->canMove())
		LOG_MESSAGE(StringBuilder() << "canRightClick(): " << u->canRightClick())
		LOG_MESSAGE(StringBuilder() << "getLastCommand().getType(): " << u->getLastCommand().getType())
		LOG_MESSAGE(StringBuilder() << "getTarget(): " << u->getTarget())
		LOG_MESSAGE(StringBuilder() << "getTargetPosition(): " << u->getTargetPosition())
		LOG_MESSAGE(StringBuilder() << "getVelocityX(): " << u->getVelocityX())
		LOG_MESSAGE(StringBuilder() << "getVelocityY(): " << u->getVelocityY())
		LOG_MESSAGE(StringBuilder() << "isAccelerating(): " << u->isAccelerating())
		LOG_MESSAGE(StringBuilder() << "isAttackFrame(): " << u->isAttackFrame())
		LOG_MESSAGE(StringBuilder() << "isAttacking(): " << u->isAttacking())
		LOG_MESSAGE(StringBuilder() << "isBraking(): " << u->isBraking())
		LOG_MESSAGE(StringBuilder() << "isHoldingPosition(): " << u->isHoldingPosition())
		LOG_MESSAGE(StringBuilder() << "isIdle(): " << u->isIdle())
		LOG_MESSAGE(StringBuilder() << "isMoving(): " << u->isMoving())
		LOG_MESSAGE(StringBuilder() << "isStuck(): " << u->isStuck())
		LOG_MESSAGE("")
	}
	else if(command == "/drawBaseLocations")
	{
		Broodwar << "Starting locations will be drawn in red, other Base locations will be drawn as yellow rectangles." << endl;
		DRAW_BASE_LOCATIONS = true;
	}
	else if(command == "/drawChokePoints")
	{
		Broodwar << "Chokepoints will be drawn as white lines" << endl;
		DRAW_CHOKE_POINTS = true;
	}
	else if(command == "/drawMapGraph")
	{
		DRAW_MAP_GRAPH = true;
	}
	else if(command == "/drawOpponentClusters")
	{
		Broodwar << "Visualizing opponent clusters" << endl;
		DRAW_OPPONENT_CLUSTERS = true;
	}
	else if(command == "/drawPaths")
	{
		Broodwar << "Paths will be drawn in green." << endl;
		DRAW_PATHS = true;
	}
	else if(command == "/drawReserved")
	{
		Broodwar << "Tiles in resource-paths will be indicated in yellow, reserved tiles in red." << endl;
		DRAW_RESERVED = true;
	}
	else if(command == "/drawThreatMap")
	{
		DRAW_THREAT_MAP = true;
	}
	else if(command == "/drawWalkability")
	{
		DRAW_WALKABILITY = true;
	}
	else if(command == "/logRandomTraversal")
	{
		ArmyManager::Instance()->getSearchEngine().logRandomTraversal();
	}
	else if(command == "/pause")
	{
		if(PAUSED)
			PAUSED = false;
		else
			PAUSED = true;
	}
	else if(command == "/setMarker")
	{
		MARKED_POSITION = Broodwar->getMousePosition() + Broodwar->getScreenPosition();
		Broodwar << "Marked Position (" << MARKED_POSITION.x << ", " << MARKED_POSITION.y << ")" << endl;
	}
	else if(command == "/verboseUCT")
	{
		ArmyManager::Instance()->getSearchEngine().setVerbose();
	}
	else if(command.find("/help") == 0)
	{
		string text = "";

		if(command.size() > 6)
			text = command.substr(6);	// all text after ''/help ''

		for(size_t i = 0; i < ArrayUtils::getArraySize(KNOWN_COMMANDS); ++i)
		{
			if(KNOWN_COMMANDS[i].find(text) != string::npos)
				Broodwar << " - " << KNOWN_COMMANDS[i] << endl;
		}
	}
	else
		Broodwar << "Command not known. Try \"/help <text>\" to find all commands containing <text>" << endl;
}

#endif	// MAASCRAFT_DEBUG