/*	-----------------------------------------------------------------------------
	M A A S C R A F T

	StarCraft: Brood War - Bot

	Author: Dennis Soemers
	Maastricht University
	-----------------------------------------------------------------------------
*/

#include "../CommonIncludes.h"

#include "../Distances.h"
#include "../MathConstants.h"
#include "../OpponentTracker.h"
#include "../Squad.h"
#include "../UnitTracker.h"
#include "../UnitUtils.h"
#include "TargetSelection.h"

#pragma warning( push )
#pragma warning( disable:4244 )

using namespace BWAPI;

// Aggro = (target's DPS to us) / (target HP / (our DPS to target)) = (target's DPS to us) * (our DPS to target) / (target HP)
const float TargetSelection::computeAggro(const Unit attacker, const Unit target)
{
	float selfDPS = UnitUtils::dpsToTarget(attacker, target);

	if(selfDPS < 0.0f)
		return (-10.0f * target->getHitPoints()) / target->getType().maxHitPoints();

	float opponentDPS = UnitUtils::dpsToTarget(target, attacker);

	if(opponentDPS <= 0.0f)
		opponentDPS = 0.0f;
	else if(target->getType() == UnitTypes::Terran_Siege_Tank_Siege_Mode || target->getType() == UnitTypes::Terran_Siege_Tank_Tank_Mode)
		opponentDPS *= 2.333f;
	
	if(target->isRepairing() && Distances::getDistance(attacker, target) <= Broodwar->self()->weaponMaxRange(attacker->getType().groundWeapon()))
	{
		opponentDPS += 5.0f;
	}
	
	return ((opponentDPS * selfDPS) / UnitUtils::hp(target));
}

// Tactical threat numbers based on Nova's AIIDE 2012 source code
const float TargetSelection::computeTacticalThreat(const Unit attacker, const Unit target)
{
	const UnitType attackerType = attacker->getType();
	const UnitType targetType = target->getType();
	const bool attackerIsFlyer = attackerType.isFlyer();

	// give high priority to targets we can kill in a single attack
	if(targetType.isFlyer() || target->isLifted())
	{
		WeaponType weapon = attackerType.airWeapon();

		if(UnitUtils::damageToTarget(attacker, target) >= (target->getHitPoints() + target->getShields()) &&
			Distances::getDistance(attacker, target) <= Broodwar->self()->weaponMaxRange(weapon)			)
		{
			return 30000.0f;
		}
	}
	else
	{
		WeaponType weapon = attackerType.groundWeapon();

		if(UnitUtils::damageToTarget(attacker, target) >= (target->getHitPoints() + target->getShields()) &&
			Distances::getDistance(attacker, target) <= Broodwar->self()->weaponMaxRange(weapon)			)
		{
			return 30000.0f;
		}
	}

	if(targetType.isBuilding())			// BUILDINGS
	{
		if(!attackerIsFlyer && targetType == UnitTypes::Zerg_Sunken_Colony)
			return 30000.0f;

		if(targetType == UnitTypes::Terran_Bunker && attackerType == UnitTypes::Protoss_Zealot)
		{
			return 100.0f;
		}
		else if(targetType.canProduce())
		{
			if(targetType == UnitTypes::Zerg_Hatchery	||
				targetType == UnitTypes::Zerg_Lair		||
				targetType == UnitTypes::Zerg_Hive			)
				return 50.0f;

			return 10.0f;
		}

		if(targetType.isRefinery() || targetType == UnitTypes::Protoss_Pylon)
			return 10.0f;

		return 5.0f;
	}
	else								// UNITS
	{
		if(targetType == UnitTypes::Terran_Medic)
			return 500.0f;

		if(targetType.isWorker())
		{
			if(targetType == UnitTypes::Terran_SCV									&&
					(target->isRepairing()									|| 
					target->getHitPoints() < targetType.maxHitPoints()		|| 
					target->isConstructing()								|| 
					target->isAttacking()										)	&&
				Distances::getDistance(attacker, target) <= Broodwar->self()->weaponMaxRange(attackerType.groundWeapon()))
			{
				UnitOwner* owner = UnitTracker::Instance()->getUnitOwner(attacker);
				Squad* squad = dynamic_cast<Squad*>(owner);

				if(squad && squad->getSize() >= 4)	// don't prioritize SCVs if we have a large squad
				{
					return 250.0f;
				}
				else
				{
					return 5000.0f;
				}
			}
			else
			{
				return 250.0f;
			}
		}

		if(targetType == UnitTypes::Terran_Siege_Tank_Siege_Mode || targetType == UnitTypes::Terran_Siege_Tank_Tank_Mode)
			return 150.0f;
		
		if(targetType == UnitTypes::Protoss_Observer)
			return 500.0f;

		if(targetType == UnitTypes::Terran_Science_Vessel	||
			targetType == UnitTypes::Terran_Dropship		||
			targetType == UnitTypes::Protoss_Shuttle		||
			targetType == UnitTypes::Zerg_Overlord			||
			targetType == UnitTypes::Zerg_Queen					)
			return 10.0f;

		if(targetType == UnitTypes::Zerg_Egg || targetType == UnitTypes::Zerg_Larva
											|| targetType == UnitTypes::Zerg_Lurker_Egg)
		{
			return -100.0f;
		}
	}

	return 0.0f;
}

// Target selection algorithm based on target selection used by Nova
// as described in Master thesis: http://nova.wolfwork.com/papers/Multi-Reactive_Planning_for_Real-Time_Strategy_Games.pdf
Unit TargetSelection::selectTarget(const Unit attacker, const Unitset& potentialTargets)
{
	static const float AGGRO_WEIGHT = 1000.0f;
	static const float TACTICAL_WEIGHT = 1.0f;
	static const float DISTANCE_WEIGHT = 0.5f;

	if(potentialTargets.empty())
		return nullptr;

	Unit bestTarget = potentialTargets.front();
	float bestScore = MathConstants::MIN_FLOAT;

	OpponentTracker* opponentTracker = OpponentTracker::Instance();

	for(auto it = potentialTargets.begin(); it != potentialTargets.end(); ++it) 
	{
		BWAPI::Unit target = *it;

		if(!target->isDetected() && target->isVisible())
			continue;

		const bool melee = (!(target->isFlying() || target->isLifted()) && attacker->getType().groundWeapon().maxRange() <= 48);

		UnitType targetType = opponentTracker->getLastType(target);

		if(melee && targetType == UnitTypes::Terran_Vulture_Spider_Mine)
			continue;

		float aggro;
		if(targetType.isWorker() && !target->isRepairing() && !target->isAttacking() && !target->isConstructing())
			aggro = 0.0f;
		else
			aggro = computeAggro(attacker, target);

		if(aggro < 0.0f)	// means we can't attack this target
			continue;

		if(aggro == 0.0f)
			aggro = -1.0f;

		float tacticalThreat = computeTacticalThreat(attacker, target);
		float score = aggro * AGGRO_WEIGHT + tacticalThreat * TACTICAL_WEIGHT;

		float meleeFactor = melee ? 200.0f : 1.0f;

		score -= Distances::getDistance(attacker, target) * DISTANCE_WEIGHT * meleeFactor;

		if(meleeFactor > 1.0f)
		{
			score -= 10.0f * targetType.topSpeed() * DISTANCE_WEIGHT * meleeFactor;
		}

		if(aggro <= 0.0f)		// pretend target is a bit further away if he can't hurt us
			score -= 320 * DISTANCE_WEIGHT * meleeFactor;

		if(score > bestScore)
		{
			bestScore = score;
			bestTarget = target;
		}
	}

	return bestTarget;
}

#pragma warning( pop )