/*	-----------------------------------------------------------------------------
	M A A S C R A F T

	StarCraft: Brood War - Bot

	Author: Dennis Soemers
	Maastricht University
	-----------------------------------------------------------------------------
*/

/*
	A concrete Task to Scout a Base Location
*/

#pragma once

#include "../CommonIncludes.h"

#include "../BaseLocation.h"
#include "../BaseManager.h"
#include "../MapAnalyser.h"
#include "../OpponentTracker.h"
#include "../PositionUtils.h"
#include "../Predicates.h"
#include "../Priorities.h"
#include "../ScoutManager.h"
#include "../Task.h"
#include "../WorkerManager.h"

template <typename SchedulePred = Predicates::Tautology, typename IsValidPred = Predicates::Tautology>
class ScoutBaseLocation : public Task
{
private:
	BWAPI::Unit scout;
	BaseLocation* target;
	int lastObserved;

public:
	ScoutBaseLocation(BWAPI::Unit scout, BaseLocation* target) : 
	  Task(Priorities::PRIORITY_SCOUTING), scout(scout), target(target),
	  lastObserved(target->getLastObserved()) {}

	void onStart()
	{
		target->setCurrentScoutTarget(true);
	}

	void onEnd()
	{
		if(scout)
		{
			scout->stop();
		}

		target->setCurrentScoutTarget(false);
	}

	inline const bool rescheduleIfInterrupted() const
	{
		return true;
	}

	inline const bool allowInterruption() const
	{
		return true;
	}

	inline const bool canExecute() const
	{
		return true;
	}

	void execute()
	{
		if(scout)
		{
			Position targetPos = Positions::Invalid;

			if(OpponentTracker::Instance()->getGroundThreat(scout->getPosition()) > 0.0f && BaseManager::Instance()->getOpponentBases().size() > 0)
			{
				targetPos = (PositionUtils::toPosition(BaseManager::Instance()->getMainBaseLocation()->getOptimalDepotLocation()));
			}
			else if(!scout->isMoving() || !MapAnalyser::Instance()->positionsInSameRegion(scout->getPosition(), PositionUtils::toPosition(target->getOptimalDepotLocation())))
			{
				targetPos = (PositionUtils::toPosition(target->getOptimalDepotLocation()));
			}

			if(targetPos != Positions::Invalid)
			{
				if(!scout->isMoving() || scout->getLastCommand().getTargetPosition() != targetPos)
					scout->move(targetPos);
			}
		}
	}

	// Task is finished either if we've move all the way far enough to observe the Depot Location,
	// or we have moved far enough to at least observe some building and far enough to get in a threatening position
	inline const bool isFinished() const
	{
		return (target->getLastObserved() > lastObserved		&& 
				(OpponentTracker::Instance()->getGroundThreat(scout->getPosition()) > 0.0f	||
					Broodwar->isVisible(target->getOptimalDepotLocation())						));
	}

	const bool isValid() const
	{
		return (scout->getHitPoints() > 0	&& 
				IsValidPred()()				&&
				MapAnalyser::Instance()->canReach(scout, PositionUtils::toWalkPosition(target->getOptimalDepotLocation())));
	}

	inline const bool scheduleConditions() const
	{
		return SchedulePred()();
	}

#ifdef MAASCRAFT_DEBUG
	std::string getDescription() const
	{
		return "Scouting a Base Location";
	}
#endif
};