/*	-----------------------------------------------------------------------------
	M A A S C R A F T

	StarCraft: Brood War - Bot

	Author: Dennis Soemers
	Maastricht University
	-----------------------------------------------------------------------------
*/

/*
	A concrete Task to retreat all the way to a base we own
*/

#pragma once

#include "../CommonIncludes.h"

#include "../BaseLocation.h"
#include "../BaseManager.h"
#include "../Distances.h"
#include "../MapAnalyser.h"
#include "../MathConstants.h"
#include "../PositionUtils.h"
#include "../Squad.h"
#include "../Task.h"
#include "../UnitBehaviours/CombatBehaviourRetreat.h"
#include "../UnitUtils.h"

#ifdef ARMY_MANAGER_SCRIPTED

class Retreat : public Task
{
private:
	BWAPI::Position destination;
	Squad* squad;

public:
	Retreat(const int priority, Squad* squad) : 
	  Task(priority), squad(squad) 
	{
		BWAPI::Position currentSquadPos = squad->getOwnedUnits().getPosition();

		int smallestDistSq = MathConstants::MAX_INT;
		BWAPI::Position bestDestination = BWAPI::Positions::Invalid;

		const std::vector<Base>& bases = BaseManager::Instance()->getSelfBases();
		for(auto it = bases.begin(); it != bases.end(); ++it)
		{
			Base base = *it;
			BaseLocation* loc = base.getLocation();
			BWAPI::Position basePos = PositionUtils::toPosition(loc->getOptimalDepotLocation());

			const int distSq = Distances::getSquaredDistance(currentSquadPos, basePos);
			if(distSq < smallestDistSq)
			{
				smallestDistSq = distSq;
				bestDestination = basePos;
			}
		}

		destination = bestDestination;
	}

	void onStart()
	{
		squad->setRegion(MapAnalyser::Instance()->getRegion(destination));
	}

	inline const bool allowInterruption() const
	{
		return false;
	}

	inline const bool canExecute() const
	{
		return true;
	}

	void execute()
	{
		ArmyManager* armyManager = ArmyManager::Instance();
		MapAnalyser* mapAnalyser = MapAnalyser::Instance();
		BWAPI::Unitset units = squad->getOwnedUnits();

		for(auto it = units.begin(); it != units.end(); ++it)
		{
			const BWAPI::Unit unit = *it;

			if(mapAnalyser->getRegion(unit->getPosition()) == squad->getRegion())
			{
				armyManager->transferUnitToDefensiveSquad(squad, unit);
			}
			else if(UnitUtils::isInCombat(unit))
			{
				Behaviours::combatBehaviourRetreat(unit, destination);
			}
			else
			{
				unit->move(destination);
			}
		}
	}

	inline const bool isFinished() const
	{
		return (squad->getOwnedUnits().size() == 0);
	}

	const bool isValid() const
	{
		return true;
	}

	inline const bool scheduleConditions() const
	{
		return true;
	}

#ifdef MAASCRAFT_DEBUG
	std::string getDescription() const
	{
		return "Retreat";
	}
#endif
};

#endif // ARMY_MANAGER_SCRIPTED