/*	-----------------------------------------------------------------------------
	M A A S C R A F T

	StarCraft: Brood War - Bot

	Author: Dennis Soemers
	Maastricht University
	-----------------------------------------------------------------------------
*/

/*
	A concrete Task to move a Squad to a location
*/

#pragma once

#include "../CommonIncludes.h"

#include "../DebugDrawing.h"
#include "../Distances.h"
#include "../OpponentTracker.h"
#include "../Squad.h"
#include "../Task.h"
#include "../UnitBehaviours/AvoidGroundThreats.h"
#include "../UnitBehaviours/CombatBehaviourFight.h"
#include "../UnitUtils.h"

#ifdef ARMY_MANAGER_UCT

class MoveSquad : public Task
{
private:
	MapGraphNode* node;
	Squad* squad;
	bool wait;

public:
	MoveSquad(const int priority, Squad* squad, MapGraphNode* node, const bool wait) : 
	  Task(priority), node(node), squad(squad), wait(wait) {}

	inline const bool allowInterruption() const
	{
		return true;
	}

	inline const bool canExecute() const
	{
		return true;
	}

	BWAPI::Unitset getPotentialTargets(BWAPI::Unit squadMember) const
	{
		BWAPI::Position selfPos = squadMember->getPosition();
		BWAPI::Position moveTarget = node->getPosition();

		const int distSqSelfToMoveTarget = Distances::getSquaredDistance(selfPos, moveTarget);

		BWAPI::Unitset potentialTargets = UnitUtils::getUnitsInRadius(squadMember, 640, BWAPI::Filter::IsEnemy		&& 
																						BWAPI::Filter::IsDetected	&&
																						[&](BWAPI::Unit u)
		{
			return (distSqSelfToMoveTarget < Distances::getSquaredDistance(OpponentTracker::Instance()->getLastPosition(u), moveTarget));
		});

		return potentialTargets;
	}

	void execute()
	{
		OpponentTracker* opponentTracker = OpponentTracker::Instance();
		const BWAPI::Unitset& units = squad->getOwnedUnits();

		for(auto it = units.begin(); it != units.end(); ++it)
		{
			const BWAPI::Unit unit = *it;

			if(squad->isInCombat())
			{
				BWAPI::Unitset enemies = BWAPI::Broodwar->getUnitsInRadius(unit->getPosition(), 7 * 32, BWAPI::Filter::IsEnemy && BWAPI::Filter::IsDetected);

				bool fight = false;
				for(auto it = enemies.begin(); it != enemies.end(); ++it)
				{
					BWAPI::Unit enemy = *it;

					int range;
					if(unit->getType().isFlyer())
						range = UnitUtils::rangeAir(enemy);
					else
						range = UnitUtils::rangeGround(enemy);

					range = std::max(range, 150);

					if(Distances::getSquaredDistance(unit, enemy) < range * range)
					{
						fight = true;
						break;
					}
				}

				if(fight && Behaviours::combatBehaviourFight(unit))
				{
					continue;
				}
			}

			BWAPI::UnitType unitType = unit->getType();

			if(unitType.groundWeapon() != BWAPI::WeaponTypes::None	&& 
				!(UnitUtils::getUnitsInWeaponRange(unit, unitType.groundWeapon(), BWAPI::Filter::IsEnemy && BWAPI::Filter::IsDetected)).empty())
			{
				if(!Behaviours::combatBehaviourFight(unit))
					LOG_WARNING(StringBuilder() << "Frame " << BWAPI::Broodwar->getFrameCount() << ": MoveSquad: Enemy in range but combatBehaviourFight() returns false!")
			}
			else if(unitType.airWeapon() != BWAPI::WeaponTypes::None	&& 
				!(UnitUtils::getUnitsInWeaponRange(unit, unitType.airWeapon(), BWAPI::Filter::IsEnemy && BWAPI::Filter::IsDetected)).empty())
			{
				if(!Behaviours::combatBehaviourFight(unit))
					LOG_WARNING(StringBuilder() << "Frame " << BWAPI::Broodwar->getFrameCount() << ": MoveSquad: Enemy in range but combatBehaviourFight() returns false!")
			}
			else
			{
				if(UnitUtils::canMove(unit) && (unit->isIdle() || unit->getLastCommand().getTargetPosition() != node->getPosition()))
				{
					if(wait)
					{
						BWAPI::Position squadPos = units.getPosition();

						if(Distances::getSquaredDistance(squadPos, unit->getPosition()) > 1024)
							unit->move(squadPos);
						else
							unit->move(node->getPosition());
					}
					else
					{
						unit->move(node->getPosition());
					}
				}
			}
		}
	}

	inline const bool isFinished() const
	{
		return true;
	}

	const bool isValid() const
	{
		return true;
	}

	inline const bool scheduleConditions() const
	{
		return true;
	}

#ifdef MAASCRAFT_DEBUG
	std::string getDescription() const
	{
		return "Move Squad";
	}
#endif
};

#endif // ARMY_MANAGER_UCT