/*	-----------------------------------------------------------------------------
	M A A S C R A F T

	StarCraft: Brood War - Bot

	Author: Dennis Soemers
	Maastricht University
	-----------------------------------------------------------------------------
*/

/*
	A concrete Task to create a new Expansion
*/

#pragma once

#include "../CommonIncludes.h"

#include "../BaseLocation.h"
#include "../BaseManager.h"
#include "../BuildingManager.h"
#include "../Predicates.h"
#include "../ProductionManager.h"
#include "../Task.h"
#include "../WorkerManager.h"
#include "ConstructPylonForPower.hpp"

template <typename SchedulePred = Predicates::Tautology, typename IsValidPred = Predicates::Tautology>
class Expand : public ProductionTask
{
private:
	mutable BWAPI::TilePosition buildPos;
	mutable BWAPI::Unit builder;
	BWAPI::TilePosition hintPos;
	mutable bool reserved;

public:
	Expand(const int priority, const BWAPI::TilePosition& hintPos) : 
		ProductionTask(priority, BWAPI::UnitTypes::Protoss_Nexus), buildPos(TilePositions::Invalid), builder(nullptr), 
		hintPos(hintPos), reserved(false) {}

	inline const bool allowInterruption() const
	{
		return (!builder || builder->getLastCommand().getType() != BWAPI::UnitCommandTypes::Build);
	}

	void onEnd()
	{
		ProductionTask::onEnd();
		BuildingManager::Instance()->transferOwnershipTo(builder, WorkerManager::Instance());
	}

	const bool canExecute() const
	{
		if(ProductionManager::Instance()->canConstructBuilding(getProductionType()))
		{
			if(buildPos == TilePositions::Invalid)
				buildPos = BuildingManager::Instance()->getBuildLocationNear(hintPos, getProductionType(), MapAnalyser::Instance()->getRegion(hintPos));

			return true;
		}
		else
		{
			return false;
		}
	}

	void execute()
	{
		if(!builder || builder->getHitPoints() == 0)
		{
			builder = WorkerManager::Instance()->requestBuilder(buildPos);

			if(!builder)	// can't execute yet, waiting for a builder
			{
				return;
			}
			else
			{
				WorkerManager::Instance()->transferOwnershipTo(builder, BuildingManager::Instance());
			}
		}

		if(builder->isConstructing())		// probably already executing this command
		{
			return;
		}

		if(!reserved)
		{
			BuildingManager::Instance()->reserveSpaceForBuilding(getProductionType(), buildPos);
			reserved = true;
		}

		if(!BWAPI::Broodwar->isVisible(buildPos))
			builder->move(PositionUtils::toPosition(buildPos));
		else
			builder->build(getProductionType(), buildPos);
	}

	const bool haltsLowerPriorities() const
	{
		return true;
	}

	inline const bool isFinished() const
	{
		BWAPI::Unitset unitsOnTile = BWAPI::Broodwar->getUnitsOnTile(buildPos);

		for(auto it = unitsOnTile.begin(); it != unitsOnTile.end(); ++it)
		{
			if((*it)->getType() == getProductionType())
			{
				return true;
			}
		}

		return false;
	}

	const bool isValid() const
	{
		return true;
	}

	inline const bool scheduleConditions() const
	{
		return SchedulePred()();
	}

	inline const BWAPI::TilePosition getBuildPos() const
	{
		return buildPos;
	}

	inline const bool rescheduleIfInterrupted() const
	{
		return true;
	}

	inline Task* reset()
	{
		if(builder)
			builder = nullptr;

		buildPos = BWAPI::TilePositions::Invalid;

		return this;
	}

#ifdef MAASCRAFT_DEBUG
	std::string getDescription() const
	{
		return (StringBuilder() << "Construct building: " << getProductionType()).getString();
	}
#endif
};