/*	-----------------------------------------------------------------------------
	M A A S C R A F T

	StarCraft: Brood War - Bot

	Author: Dennis Soemers
	Maastricht University
	-----------------------------------------------------------------------------
*/

/*
	A concrete Task to defend a chokepoint
*/

#pragma once

#include "../CommonIncludes.h"

#include "../Distances.h"
#include "../MapAnalyser.h"
#include "../MapChokePoint.h"
#include "../MapRegion.h"
#include "../PositionUtils.h"
#include "../Squad.h"
#include "../Task.h"
#include "../UnitUtils.h"

#include "../UnitBehaviours/CombatBehaviourFight.h"

#ifdef ARMY_MANAGER_SCRIPTED

class DefendChokePoint : public Task
{
private:
	Squad* squad;
	MapRegion* region;
	MapChokePoint* chokePoint;

public:
	DefendChokePoint(const int priority, Squad* squad, MapRegion* region, MapChokePoint* chokePoint) : 
	  Task(priority), squad(squad), region(region), chokePoint(chokePoint) {}

	void onStart()
	{
		squad->setRegion(region);
	}

	inline const bool allowInterruption() const
	{
		return true;
	}

	inline const bool canExecute() const
	{
		return true;
	}

	void execute()
	{
		const BWAPI::Unitset& units = squad->getOwnedUnits();
		bool firstTile = true;	// alternate between true and false when looping through units, sending half to each side

		const std::vector<BWAPI::WalkPosition>& chokeTiles = chokePoint->getTiles();
		BWAPI::Position first = PositionUtils::toPosition(chokeTiles.front());
		BWAPI::Position second = PositionUtils::toPosition(chokeTiles.back());

		MapAnalyser* mapAnalyser = MapAnalyser::Instance();
		const BWAPI::Unitset opponentsInRegion = squad->getOpponentsInRegion();

		for(auto it = units.begin(); it != units.end(); ++it)
		{
			const BWAPI::Unit unit = *it;
			const BWAPI::UnitType unitType = unit->getType();
			const BWAPI::Position unitPos = unit->getPosition();
			const BWAPI::Position target = firstTile ? first : second;
			firstTile = !firstTile;

			if(mapAnalyser->getRegion(unitPos) != region && Distances::getSquaredDistance(target, unitPos) > 640000)	// 640000 = 800 * 800 ~= 25 build tiles
			{
				unit->move(PositionUtils::toPosition(region->getCentroid()));
			}
			else if(squad->isInCombat())
			{
				if(Behaviours::combatBehaviourFight(unit))
					continue;
			}

			if(!opponentsInRegion.empty())
			{
				if(unitType.groundWeapon() != BWAPI::WeaponTypes::None	&& 
					!(UnitUtils::getUnitsInWeaponRange(unit, unitType.groundWeapon(), BWAPI::Filter::IsEnemy)).empty())
				{
					Behaviours::combatBehaviourFight(unit);
				}
				else if(unitType.airWeapon() != BWAPI::WeaponTypes::None	&& 
					!(UnitUtils::getUnitsInWeaponRange(unit, unitType.airWeapon(), BWAPI::Filter::IsEnemy)).empty())
				{
					Behaviours::combatBehaviourFight(unit);
				}
				else
				{
					unit->move(opponentsInRegion.front()->getPosition());
				}
			}
			else if(mapAnalyser->getRegion(unitPos) != region)
			{
				unit->move(PositionUtils::toPosition(region->getCentroid()));
			}
			else if(Distances::getSquaredDistance(target, unitPos) > 36864)		// 36864 = 192*192 ~= 6 build tiles
			{
				if(unit->isIdle())
				{
					unit->move(target);
				}
				else if(Distances::getSquaredDistance(target, unitPos) > 82944								&&	// 82944 = 288*288 ~= 9 build tiles
						unit->getOrderTargetPosition() == PositionUtils::toPosition(region->getCentroid())		)
				{
					unit->stop();
				}
			}
			else
			{
				unit->move(PositionUtils::toPosition(region->getCentroid()));
			}
		}
	}

	inline const bool isFinished() const
	{
		return false;
	}

	const bool isValid() const
	{
		return true;
	}

	inline const bool scheduleConditions() const
	{
		return true;
	}

#ifdef MAASCRAFT_DEBUG
	std::string getDescription() const
	{
		return "Defend Chokepoint";
	}
#endif
};

#endif // ARMY_MANAGER_SCRIPTED