/*	-----------------------------------------------------------------------------
	M A A S C R A F T

	StarCraft: Brood War - Bot

	Author: Dennis Soemers
	Maastricht University
	-----------------------------------------------------------------------------
*/

/*
	A concrete Task to construct an extra Pylon as ''emergency'' because we expect to need more Supply soon
	(http://wiki.teamliquid.net/starcraft/Supply)
*/

#pragma once

#include "../CommonIncludes.h"

#include "../BaseLocation.h"
#include "../BaseManager.h"
#include "../BuildingManager.h"
#include "../ProductionManager.h"
#include "../Task.h"
#include "../WorkerManager.h"
#include "ConstructBuilding.hpp"
#include "ProductionTask.hpp"

class ConstructPylonForSupply : public ProductionTask
{
private:
	mutable BWAPI::TilePosition buildPos;
	mutable BWAPI::Unit builder;
	const BaseLocation* preferredBase;

public:
	ConstructPylonForSupply(const int priority, const BaseLocation* preferredBase = nullptr) : 
		ProductionTask(priority, BWAPI::UnitTypes::Protoss_Pylon), builder(nullptr), buildPos(BWAPI::TilePositions::Invalid), 
		preferredBase(preferredBase ? preferredBase :  BaseManager::Instance()->getMainBaseLocation()) {}

	void onEnd()
	{
		ProductionTask::onEnd();
		BuildingManager::Instance()->transferOwnershipTo(builder, WorkerManager::Instance());
	}

	inline const bool allowInterruption() const
	{
		return (!builder || builder->getLastCommand().getType() != BWAPI::UnitCommandTypes::Build);
	}

	inline const bool repeatTask() const
	{
		return true;
	}

	inline const bool repeatInvalidTask() const
	{
		return true;
	}

	const bool canExecute() const
	{
		if(ProductionManager::Instance()->canConstructBuilding(BWAPI::UnitTypes::Protoss_Pylon))
		{
			buildPos = BuildingManager::Instance()->getBuildLocationNear(preferredBase->getOptimalDepotLocation(),
																		BWAPI::UnitTypes::Protoss_Pylon, preferredBase->getRegion());

#ifdef MAASCRAFT_DEBUG
			if(buildPos == BWAPI::TilePositions::Invalid)
			{
				LOG_WARNING("Can't find valid build location for an Supply Pylon!")
			}
#endif
		}

		return (buildPos != BWAPI::TilePositions::Invalid);
	}

	void execute()
	{
		if(!builder || builder->getHitPoints() == 0)
		{
			builder = WorkerManager::Instance()->requestBuilder(buildPos);

			if(!builder)	// can't execute yet, waiting for a builder
				return;
			else
				WorkerManager::Instance()->transferOwnershipTo(builder, BuildingManager::Instance());
		}

		if(builder->isConstructing())		// probably already executing this command
			return;

		BuildingManager::Instance()->reserveSpaceForBuilding(BWAPI::UnitTypes::Protoss_Pylon, buildPos);
		builder->build(BWAPI::UnitTypes::Protoss_Pylon, buildPos);
	}

	const bool haltsLowerPriorities() const
	{
		return ProductionManager::Instance()->canSoonConstructBuilding(BWAPI::UnitTypes::Protoss_Pylon);
	}

	inline const bool isFinished() const
	{
		BWAPI::Unitset unitsOnTile = BWAPI::Broodwar->getUnitsOnTile(buildPos);

		for(auto it = unitsOnTile.begin(); it != unitsOnTile.end(); ++it)
		{
			if((*it)->getType() == BWAPI::UnitTypes::Protoss_Pylon)
			{
				return true;
			}
		}

		return false;
	}

	const bool isValid() const
	{
		return scheduleConditions();
	}

	inline const bool rescheduleIfInterrupted() const
	{
		return true;
	}

	inline Task* reset()
	{
		if(builder)
			builder = nullptr;

		buildPos = BWAPI::TilePositions::Invalid;

		return this;
	}

	inline const bool scheduleConditions() const
	{
		BWAPI::Player self = BWAPI::Broodwar->self();
		int supplyUsed = self->supplyUsed();
		int supplyBuffer = 2 * Options::ProductionManager::PYLON_SUPPLY_THRESHOLD;

		if(BWAPI::Broodwar->self()->completedUnitCount(BWAPI::UnitTypes::Protoss_Gateway) >= 4)
		{
			supplyBuffer *= 2;
		}

		int expectedSupplyTotal = self->supplyTotal() + BuildingManager::Instance()->getNumPylonsBuilding() * BWAPI::UnitTypes::Protoss_Pylon.supplyProvided();

		return (supplyUsed + supplyBuffer >= expectedSupplyTotal && expectedSupplyTotal < 400);	// 400 corresponds to StarCraft's max supply of 200
	}

#ifdef MAASCRAFT_DEBUG
	std::string getDescription() const
	{
		return "Construct Pylon for Supply";
	}
#endif
};