/*	-----------------------------------------------------------------------------
	M A A S C R A F T

	StarCraft: Brood War - Bot

	Author: Dennis Soemers
	Maastricht University
	-----------------------------------------------------------------------------
*/

/*
	A concrete Task to construct an extra Pylon as ''emergency'' because a certain building cannot be placed due to
	lacking Psionic Matrix (http://wiki.teamliquid.net/starcraft/Psionic_Matrix)
*/

#pragma once

#include "../CommonIncludes.h"

#include "../BaseLocation.h"
#include "../BaseManager.h"
#include "../BuildingManager.h"
#include "../ProductionManager.h"
#include "../Task.h"
#include "../WorkerManager.h"
#include "ConstructBuilding.hpp"
#include "ProductionTask.hpp"

// forward declaration
template <typename SchedulePred, typename IsValidPred>
class ConstructBuilding;

template <typename SchedulePred, typename IsValidPred>
class ConstructPylonForPower : public ProductionTask
{
private:
	mutable BWAPI::TilePosition buildPos;
	mutable BWAPI::Unit builder;
	const ConstructBuilding<typename SchedulePred, typename IsValidPred>* otherBuildTask;
	const BaseLocation* preferredBase;

public:
	ConstructPylonForPower(const int priority, const ConstructBuilding<SchedulePred, IsValidPred>* otherTask, 
																	const BaseLocation* const preferredBase = nullptr) : 
		ProductionTask(priority, BWAPI::UnitTypes::Protoss_Pylon), builder(nullptr), buildPos(BWAPI::TilePositions::Invalid), 
		otherBuildTask(otherTask),	preferredBase(preferredBase ? preferredBase :  BaseManager::Instance()->getMainBaseLocation()) {}

	void onEnd()
	{
		ProductionTask::onEnd();
		BuildingManager::Instance()->transferOwnershipTo(builder, WorkerManager::Instance());
	}

	inline const bool allowInterruption() const
	{
		return (!builder || builder->getLastCommand().getType() != BWAPI::UnitCommandTypes::Build);
	}

	const bool canExecute() const
	{
		if(ProductionManager::Instance()->canConstructBuilding(BWAPI::UnitTypes::Protoss_Pylon))
		{
			buildPos = BuildingManager::Instance()->getBuildLocationNear(preferredBase->getOptimalDepotLocation(),
																		BWAPI::UnitTypes::Protoss_Pylon, preferredBase->getRegion());

#ifdef MAASCRAFT_DEBUG
			if(buildPos == BWAPI::TilePositions::Invalid)
			{
				LOG_WARNING("Can't find valid build location for an Power Pylon!")
			}
#endif
		}

		return (buildPos != BWAPI::TilePositions::Invalid);
	}

	void execute()
	{
		if(!builder || builder->getHitPoints() == 0)
		{
			builder = WorkerManager::Instance()->requestBuilder(buildPos);

			if(!builder)	// can't execute yet, waiting for a builder
				return;
			else
				WorkerManager::Instance()->transferOwnershipTo(builder, BuildingManager::Instance());
		}

		if(builder->isConstructing())		// probably already executing this command
			return;

		BuildingManager::Instance()->reserveSpaceForBuilding(BWAPI::UnitTypes::Protoss_Pylon, buildPos);
		builder->build(BWAPI::UnitTypes::Protoss_Pylon, buildPos);
	}

	const bool haltsLowerPriorities() const
	{
		return ProductionManager::Instance()->canSoonConstructBuilding(BWAPI::UnitTypes::Protoss_Pylon);
	}

	inline const bool isFinished() const
	{
		BWAPI::Unitset unitsOnTile = BWAPI::Broodwar->getUnitsOnTile(buildPos);

		for(auto it = unitsOnTile.begin(); it != unitsOnTile.end(); ++it)
		{
			if((*it)->getType() == BWAPI::UnitTypes::Protoss_Pylon)
			{
				return true;
			}
		}

		return false;
	}

	const bool isValid() const
	{
		return (otherBuildTask->getBuildPos() == BWAPI::TilePositions::Invalid	&& 
				BuildingManager::Instance()->getNumPylonsBuilding() == 0			);
	}

	inline const bool scheduleConditions() const
	{
		return true;
	}

#ifdef MAASCRAFT_DEBUG
	std::string getDescription() const
	{
		return "Construct Pylon for Power";
	}
#endif
};