/*	-----------------------------------------------------------------------------
	M A A S C R A F T

	StarCraft: Brood War - Bot

	Author: Dennis Soemers
	Maastricht University
	-----------------------------------------------------------------------------
*/

/*
	A concrete Task to construct a building
*/

#pragma once

#include "../CommonIncludes.h"

#include "../BaseLocation.h"
#include "../BaseManager.h"
#include "../BuildingManager.h"
#include "../Predicates.h"
#include "../ProductionManager.h"
#include "../Task.h"
#include "../WorkerManager.h"
#include "ConstructPylonForPower.hpp"

template <typename SchedulePred = Predicates::Tautology, typename IsValidPred = Predicates::Tautology>
class ConstructBuilding : public ProductionTask
{
private:
	mutable BWAPI::TilePosition buildPos;
	mutable BWAPI::Unit builder;
	const BaseLocation* preferredBase;

public:
	ConstructBuilding(const int priority, BWAPI::UnitType buildingToConstruct, BaseLocation* preferredBase = nullptr) : 
		ProductionTask(priority, buildingToConstruct), builder(nullptr), buildPos(BWAPI::TilePositions::Invalid), 
		preferredBase(preferredBase ? preferredBase : BaseManager::Instance()->getMainBaseLocation()) {}

	inline const bool allowInterruption() const
	{
		return (!builder || builder->getLastCommand().getType() != BWAPI::UnitCommandTypes::Build);
	}

	void onEnd()
	{
		ProductionTask::onEnd();
		BuildingManager::Instance()->transferOwnershipTo(builder, WorkerManager::Instance());
	}

	const bool canExecute() const
	{
		if(ProductionManager::Instance()->canConstructBuilding(getProductionType()))
		{
			buildPos = BuildingManager::Instance()->getBuildLocationNear(preferredBase->getOptimalDepotLocation(),
																		getProductionType(), preferredBase->getRegion());

			// No positions to build and no pylons under construction, so schedule an emergency-pylon
			if(buildPos == BWAPI::TilePositions::Invalid && BuildingManager::Instance()->getNumPylonsBuilding() == 0)
			{
				ProductionManager::Instance()->addTask(new ConstructPylonForPower<SchedulePred, IsValidPred>
														(priority + 1, this, preferredBase));
			}
		}

		return (buildPos != BWAPI::TilePositions::Invalid);
	}

	void execute()
	{
		if(!builder || builder->getHitPoints() == 0)
		{
			builder = WorkerManager::Instance()->requestBuilder(buildPos);

			if(!builder)	// can't execute yet, waiting for a builder
				return;
			else
				WorkerManager::Instance()->transferOwnershipTo(builder, BuildingManager::Instance());
		}

		if(builder->isConstructing())		// probably already executing this command
			return;

		BuildingManager::Instance()->reserveSpaceForBuilding(getProductionType(), buildPos);
		builder->build(getProductionType(), buildPos);
	}

	const bool haltsLowerPriorities() const
	{
		return ProductionManager::Instance()->canSoonConstructBuilding(getProductionType());
	}

	inline const bool isFinished() const
	{
		BWAPI::Unitset unitsOnTile = BWAPI::Broodwar->getUnitsOnTile(buildPos);

		for(auto it = unitsOnTile.begin(); it != unitsOnTile.end(); ++it)
		{
			if((*it)->getType() == getProductionType())
			{
				return true;
			}
		}

		return false;
	}

	const bool isValid() const
	{
		return true;
	}

	inline const bool scheduleConditions() const
	{
		return SchedulePred()();
	}

	inline const BWAPI::TilePosition getBuildPos() const
	{
		return buildPos;
	}

	inline const bool rescheduleIfInterrupted() const
	{
		return true;
	}

	inline Task* reset()
	{
		if(builder)
			builder = nullptr;

		buildPos = BWAPI::TilePositions::Invalid;

		return this;
	}

#ifdef MAASCRAFT_DEBUG
	std::string getDescription() const
	{
		return (StringBuilder() << "Construct building: " << getProductionType()).getString();
	}
#endif
};