/*	-----------------------------------------------------------------------------
	M A A S C R A F T

	StarCraft: Brood War - Bot

	Author: Dennis Soemers
	Maastricht University
	-----------------------------------------------------------------------------
*/

/*
	A concrete Task to buy an upgrade
*/

#pragma once

#include "../CommonIncludes.h"

#include "../Predicates.h"
#include "../Priorities.h"
#include "../ProductionManager.h"
#include "../Task.h"

template <typename SchedulePred = Predicates::Tautology, typename IsValidPred = Predicates::Tautology>
class BuyUpgrade : public Task
{
private:
	BWAPI::UpgradeType upgradeType;
	int productionID;
	bool placedReservation;

public:
	BuyUpgrade(const int priority, BWAPI::UpgradeType upgradeType) : 
	  Task(priority), upgradeType(upgradeType), productionID(ProductionManager::Instance()->getNextProductionID()),
	  placedReservation(false) {}

	void onStart()
	{
		ProductionManager* productionManager = ProductionManager::Instance();
		int numUpgrades = BWAPI::Broodwar->self()->getUpgradeLevel(upgradeType);
		productionManager->placeGasReservation(Reservation(productionID, upgradeType.gasPrice() + numUpgrades * upgradeType.gasPriceFactor()));
		productionManager->placeMineralReservation(Reservation(productionID, upgradeType.mineralPrice() + numUpgrades * upgradeType.mineralPriceFactor()));
	}

	void onEnd()
	{
		ProductionManager* productionManager = ProductionManager::Instance();
		productionManager->removeGasReservation(productionID);
		productionManager->removeMineralReservation(productionID);
	}

	inline const bool isInstantaneous() const
	{
		return true;
	}

	inline const bool allowInterruption() const
	{
		return true;
	}

	inline const bool canExecute() const
	{
		return ProductionManager::Instance()->canBuyUpgrade(upgradeType);
	}

	void execute()
	{
		BWAPI::Unit building = ProductionManager::Instance()->getIdleUpgrader(upgradeType);
		building->upgrade(upgradeType);
	}

	inline const bool isFinished() const
	{
		return true;
	}

	const bool isValid() const
	{
		return IsValidPred()();
	}

	inline const bool scheduleConditions() const
	{
		return SchedulePred()();
	}

#ifdef MAASCRAFT_DEBUG
	std::string getDescription() const
	{
		return (StringBuilder() << "Buy Upgrade: " << upgradeType).getString();
	}
#endif
};