/*	-----------------------------------------------------------------------------
	M A A S C R A F T

	StarCraft: Brood War - Bot

	Author: Dennis Soemers
	Maastricht University
	-----------------------------------------------------------------------------
*/

/*
	A concrete Task to attack a group of enemy units
*/

#pragma once

#include "../CommonIncludes.h"

#include "../Distances.h"
#include "../MathConstants.h"
#include "../OpponentTracker.h"
#include "../PositionUtils.h"
#include "../Squad.h"
#include "../Task.h"
#include "../UnitBehaviours/CombatBehaviourFight.h"
#include "../UnitUtils.h"

#ifdef ARMY_MANAGER_SCRIPTED

class AttackUnits : public Task
{
private:
	BWAPI::Position destination;
	Squad* squad;

public:
	AttackUnits(const int priority, Squad* squad, const BWAPI::Unitset& targets) : 
	  Task(priority), squad(squad) 
	{
		destination = targets.front()->getPosition();

		if(destination == BWAPI::Positions::Unknown)
			destination = OpponentTracker::Instance()->getLastPosition(targets.front());
	}

	void onStart()
	{
		squad->setRegion(MapAnalyser::Instance()->getRegion(destination));
	}

	inline const bool allowInterruption() const
	{
		return true;
	}

	inline const bool canExecute() const
	{
		return true;
	}

	void execute()
	{
		const BWAPI::Unitset& units = squad->getOwnedUnits();
		const BWAPI::Unitset opponentsInRegion = squad->getOpponentsInRegion();

		for(auto it = units.begin(); it != units.end(); ++it)
		{
			const BWAPI::Unit unit = *it;
			const BWAPI::UnitType unitType = unit->getType();

			if(squad->isInCombat())
			{
				if(Behaviours::combatBehaviourFight(unit))
					continue;
			}

			if(!opponentsInRegion.empty())
			{
				if(unitType.groundWeapon() != BWAPI::WeaponTypes::None	&& 
					!(UnitUtils::getUnitsInWeaponRange(unit, unitType.groundWeapon(), BWAPI::Filter::IsEnemy && BWAPI::Filter::IsDetected)).empty())
				{
					if(!Behaviours::combatBehaviourFight(unit))
					{
						LOG_WARNING(StringBuilder() << "Frame " << BWAPI::Broodwar->getFrameCount() << ": AttackUnits: Enemy in range but combatBehaviourFight() returns false!")
					}
				}
				else if(unitType.airWeapon() != BWAPI::WeaponTypes::None	&& 
					!(UnitUtils::getUnitsInWeaponRange(unit, unitType.airWeapon(), BWAPI::Filter::IsEnemy && BWAPI::Filter::IsDetected)).empty())
				{
					if(!Behaviours::combatBehaviourFight(unit))
					{
						LOG_WARNING(StringBuilder() << "Frame " << BWAPI::Broodwar->getFrameCount() << ": AttackUnits: Enemy in range but combatBehaviourFight() returns false!")
					}
				}
				else
				{
					DebugDrawing::drawLineMap(unit->getPosition(), opponentsInRegion.front()->getPosition(), BWAPI::Colors::Purple);
					unit->move(opponentsInRegion.front()->getPosition());
				}
			}
			else
			{
				DebugDrawing::drawLineMap(unit->getPosition(), destination, BWAPI::Colors::Green);
				unit->move(destination);
			}
		}
	}

	inline const bool isFinished() const
	{
		return (!(BWAPI::Broodwar->getUnitsOnTile(PositionUtils::toTilePosition(destination), BWAPI::Filter::IsAlly && !BWAPI::Filter::IsWorker).empty()));
	}

	const bool isValid() const
	{
		return true;
	}

	inline const bool scheduleConditions() const
	{
		return true;
	}

#ifdef MAASCRAFT_DEBUG
	std::string getDescription() const
	{
		return "Attack Units";
	}
#endif
};

#endif // ARMY_MANAGER_SCRIPTED