/*	-----------------------------------------------------------------------------
	M A A S C R A F T

	StarCraft: Brood War - Bot

	Author: Dennis Soemers
	Maastricht University
	-----------------------------------------------------------------------------
*/

/*
	A concrete Task to attack an enemy Base
*/

#pragma once

#include "../CommonIncludes.h"

#include "../BaseLocation.h"
#include "../BaseManager.h"
#include "../DebugDrawing.h"
#include "../Distances.h"
#include "../MathConstants.h"
#include "../OpponentTracker.h"
#include "../PositionUtils.h"
#include "../Squad.h"
#include "../Task.h"
#include "../UnitBehaviours/CombatBehaviourFight.h"
#include "../UnitUtils.h"

#ifdef ARMY_MANAGER_SCRIPTED

class AttackBase : public Task
{
private:
	BaseLocation* target;
	Squad* squad;

public:
	AttackBase(const int priority, Squad* squad, BaseLocation* baseLocation) : 
	  Task(priority), target(baseLocation), squad(squad) {}

	void onStart()
	{
		squad->setRegion(target->getRegion());
	}

	inline const bool allowInterruption() const
	{
		return true;
	}

	inline const bool canExecute() const
	{
		return true;
	}

	void execute()
	{
		OpponentTracker* opponentTracker = OpponentTracker::Instance();
		Base base = BaseManager::Instance()->getBaseAtLocation(BWAPI::Broodwar->enemy(), target);
		const BWAPI::Unitset& units = squad->getOwnedUnits();

		for(auto it = units.begin(); it != units.end(); ++it)
		{
			const BWAPI::Unit unit = *it;

			if(squad->isInCombat())
			{
				if(Behaviours::combatBehaviourFight(unit))
					continue;
			}

			if(!(base.getBuildings().empty()))
			{
				BWAPI::UnitType unitType = unit->getType();

				if(unitType.groundWeapon() != BWAPI::WeaponTypes::None	&& 
					!(UnitUtils::getUnitsInWeaponRange(unit, unitType.groundWeapon(), BWAPI::Filter::IsEnemy && BWAPI::Filter::IsDetected)).empty())
				{
					if(!Behaviours::combatBehaviourFight(unit))
						LOG_WARNING(StringBuilder() << "Frame " << BWAPI::Broodwar->getFrameCount() << ": AttackBase: Enemy in range but combatBehaviourFight() returns false!")
				}
				else if(unitType.airWeapon() != BWAPI::WeaponTypes::None	&& 
					!(UnitUtils::getUnitsInWeaponRange(unit, unitType.airWeapon(), BWAPI::Filter::IsEnemy && BWAPI::Filter::IsDetected)).empty())
				{
					if(!Behaviours::combatBehaviourFight(unit))
						LOG_WARNING(StringBuilder() << "Frame " << BWAPI::Broodwar->getFrameCount() << ": AttackBase: Enemy in range but combatBehaviourFight() returns false!")
				}
				else
				{
					BWAPI::Unit targetBuilding = base.getBuildings().front();

					if(targetBuilding->isVisible() && UnitUtils::canAttack(unit))
					{
						if(unit->isIdle() || unit->getLastCommand().getType() != BWAPI::UnitCommandTypes::Attack_Unit)
						{
							unit->attack(targetBuilding);
							DebugDrawing::drawLineMap(unit->getPosition(), targetBuilding->getPosition(), BWAPI::Colors::Red);
						}
					}
					else if(UnitUtils::canMove(unit) && unit->isIdle())
					{
						unit->move(opponentTracker->getLastPosition(targetBuilding));
						DebugDrawing::drawLineMap(unit->getPosition(), targetBuilding->getPosition(), BWAPI::Colors::Green);
					}
					else if(unit->isIdle())
					{
						BWAPI::Broodwar->drawTextMap(unit->getPosition(), "Nothing to do in AttackBase Task");
					}
					else
					{
						DebugDrawing::drawLineMap(unit->getPosition(), unit->getLastCommand().getTargetPosition(), BWAPI::Colors::Cyan);
					}
				}
			}
		}
	}

	inline const bool isFinished() const
	{
		return (!squad->isInCombat()	&& 
				BaseManager::Instance()->getBaseAtLocation(BWAPI::Broodwar->enemy(), target).getBuildings().empty());
	}

	const bool isValid() const
	{
		return true;
	}

	inline const bool scheduleConditions() const
	{
		return true;
	}

#ifdef MAASCRAFT_DEBUG
	std::string getDescription() const
	{
		return "Attack Base";
	}
#endif
};

#endif // ARMY_MANAGER_SCRIPTED