/*	-----------------------------------------------------------------------------
	M A A S C R A F T

	StarCraft: Brood War - Bot

	Author: Dennis Soemers
	Maastricht University
	-----------------------------------------------------------------------------
*/

/*
	A TaskSet is a collection of:
	- A current Task (which is currently being executed or SHOULD currently be executed)
	- A set of Tasks which need to be done asap
	- A set of Tasks which will need to be done at some point in the future

	TaskSet will delete all Tasks they are given at some point in time, so users of TaskSet should NOT
	clean up Tasks which they give to a TaskSet
*/

#pragma once

#include "CommonIncludes.h"

#include <set>
#include <vector>

#include "Task.h"

class TaskSet
{
public:
	TaskSet() :
		futureTasks(), tasks(), currentTask(nullptr), finishedInstantTask(nullptr) {}
	~TaskSet();

	void addTask(Task* task);
	void addTasks(const std::vector<Task*>& tasks);

	void clearTasks(const int maxClearPriority);	// clears all tasks with priority <= maxClearPriority

	Task* getCurrentTask() const;

	inline const size_t size()	{	return futureTasks.size() + tasks.size();	}

	void update();

#ifdef MAASCRAFT_DEBUG
	friend std::ostream& operator<< (std::ostream& out, const TaskSet& taskSet)
	{
		out << "[TaskSet: ";
		
		std::vector<Task*> tasksToPrint;
		for(auto it = taskSet.futureTasks.begin(); it != taskSet.futureTasks.end(); ++it)
		{
			tasksToPrint.push_back(*it);
		}
		for(auto it = taskSet.tasks.begin(); it != taskSet.tasks.end(); ++it)
		{
			tasksToPrint.push_back(*it);
		}

		for(auto it = tasksToPrint.begin(); it != tasksToPrint.end(); /**/)
		{
			out << (*it)->getDescription();
			++it;

			if(it != tasksToPrint.end())
				out << ", ";
		}

		out << "]";
		return out;
	}
#endif

private:
	struct TaskSetOrdering
	{
		bool operator() (const Task* t1, const Task* t2) const
		{
			return (t1->getPriority() > t2->getPriority());
		}
	};

	std::vector<Task*> futureTasks;
	std::multiset<Task*, TaskSetOrdering> tasks;
	Task* currentTask;
	Task* finishedInstantTask;
};