/*	-----------------------------------------------------------------------------
	M A A S C R A F T

	StarCraft: Brood War - Bot

	Author: Dennis Soemers
	Maastricht University
	-----------------------------------------------------------------------------
*/

/*
	Implementation of TaskSet.h
*/

#include "CommonIncludes.h"

#include "Task.h"
#include "TaskSet.h"
#include "VectorUtils.h"

using namespace std;

TaskSet::~TaskSet()
{
	for(auto it = futureTasks.begin(); it != futureTasks.end(); ++it)
	{
		delete (*it);
	}

	for(auto it = tasks.begin(); it != tasks.end(); ++it)
	{
		delete (*it);
	}

	if(currentTask)
	{
		currentTask->onEnd();
		delete currentTask;
	}
}

void TaskSet::addTask(Task* task)
{
	futureTasks.push_back(task);
}

void TaskSet::addTasks(const vector<Task*>& newTasks)
{
	futureTasks.reserve(futureTasks.size() + newTasks.size());
	for(auto it = newTasks.begin(); it != newTasks.end(); ++it)
	{
		futureTasks.push_back(*it);
	}
}

void TaskSet::clearTasks(const int maxClearPriority)
{
	vector<Task*> toDelete = VectorUtils::popFromVectorIfNot(futureTasks, [maxClearPriority](const Task* t) 
																		{	return t->getPriority() > maxClearPriority;	});

	for(auto it = toDelete.begin(); it != toDelete.end(); ++it)
	{
		delete (*it);
	}

	for(auto it = tasks.begin(); it != tasks.end(); /**/)
	{
		Task* t = *it;
		if(t->getPriority() <= maxClearPriority)
		{
			it = tasks.erase(it);
			delete t;
			t = nullptr;
		}
		else
			++it;
	}

	if(currentTask && currentTask->getPriority() < maxClearPriority)
	{
		currentTask->onEnd();
		delete currentTask;
		currentTask = nullptr;
	}
}

Task* TaskSet::getCurrentTask() const
{
	return currentTask;
}

void TaskSet::update()
{
	if(finishedInstantTask)
	{
		finishedInstantTask->onEnd();
		if(finishedInstantTask->repeatTask())
		{
			finishedInstantTask = finishedInstantTask->reset();
			futureTasks.push_back(finishedInstantTask);
		}
		else
		{
			delete finishedInstantTask;
		}

		finishedInstantTask = nullptr;
	}

	// Deal properly with currentTask
	bool allowNewTask = true;
	bool changedTask = false;
	if(currentTask)
	{
		if(currentTask->isFinished())
		{
			currentTask->onEnd();
			if(currentTask->repeatTask())
				futureTasks.push_back(currentTask->reset());
			else
				delete currentTask;

			currentTask = nullptr;
		}
		else if(!currentTask->isValid())
		{
			currentTask->onEnd();
			if(currentTask->repeatInvalidTask())
				futureTasks.push_back(currentTask->reset());
			else
				delete currentTask;

			currentTask = nullptr;
		}
		else if(!currentTask->allowInterruption())
			allowNewTask = false;
	}

	// Add futureTasks to tasks if they have become schedulable
	vector<Task*> newTasks = VectorUtils::popFromVectorIfNot(futureTasks, [](const Task* t){	return !(t->scheduleConditions());	});

	for(auto it = newTasks.begin(); it != newTasks.end(); ++it)
	{
		tasks.insert(*it);
	}

	// Check if tasks are still valid
	for(auto it = tasks.begin(); it != tasks.end(); /**/)
	{
		Task* t = *it;
		if(t->isValid() || t->rescheduleIfInterrupted())
			++it;
		else
		{
			it = tasks.erase(it);
			delete t;
		}
	}

	// Replace current task with higher priority new tasks if allowed
	if(allowNewTask)
	{
		const int currentPriority = currentTask ? currentTask->getPriority() : -1;

		// find highest priority task which is ready to be executed
		for(auto it = tasks.begin(); it != tasks.end(); ++it)
		{
			Task* t = *it;

			if(t->getPriority() <= currentPriority)
				break;

			if(t->canExecute())
			{
				tasks.erase(it);

				if(currentTask)
				{
					currentTask->onEnd();

					if(currentTask->rescheduleIfInterrupted())
						tasks.insert(currentTask->reset());
					else
						delete currentTask;
				}

				currentTask = t;
				t->onStart();
				changedTask = true;
				break;
			}
			else if(t->haltsLowerPriorities() && t->isValid())
				return;
		}
	}

	if(!changedTask)	// didn't change tasks this frame, so allow execution of one additional instantaneous task
	{
		for(auto it = tasks.begin(); it != tasks.end(); ++it)
		{
			Task* t = *it;

			if(t->isInstantaneous() && t->canExecute())
			{
				t->onStart();
				t->execute();
				finishedInstantTask = t;
				tasks.erase(it);
				return;
			}
		}
	}
}