/*	-----------------------------------------------------------------------------
	M A A S C R A F T

	StarCraft: Brood War - Bot

	Author: Dennis Soemers
	Maastricht University
	-----------------------------------------------------------------------------
*/

/*
	An abstract Task class, which represents a Task which can be completed by some unit, has some priority
	and some conditions under which the Task remains valid

	Specific Tasks should inherit from this class
*/

#pragma once

class Task
{
private:
	Task();

protected:
	const int priority;					// Priority level should be a value declared in Priorities.h

	Task(const int priority) : 
		priority(priority) {}

public:
	// Functions which CAN be implemented by new Tasks
	virtual inline const int getPriority() const		{	return priority;	}
	virtual const bool haltsLowerPriorities() const		{	return false;		}	// implement to return true if it should halt lower priority tasks
	virtual const bool isInstantaneous() const			{	return false;		}	// implement to return true if it is allowed to execute this task whilst other tasks are being executed.
	virtual const bool repeatInvalidTask() const		{	return false;		}	// implement to return true if this task should be repeated when found to be invalid
	virtual const bool repeatTask() const				{	return false;		}	// implement to return true if this task should be repeated after completion
	virtual const bool rescheduleIfInterrupted() const	{	return true;		}	// implement to return true if this task should be rescheduled for later completion
																					//	when interrupted
	virtual Task* reset()								{	return this;		}	// called when a repeatable Task is repeated

	virtual void onEnd() {}								// will be called when a Task transitions from being currentTask to not being currentTask
	virtual void onStart() {}							// will be called when a Task transitions from not being currentTask to being currentTask
	virtual ~Task() {}

	// Functions which MUST be implemented by new Tasks
	virtual const bool allowInterruption() const = 0;	// should return true if it is allowed to interrupt this task in case higher priority tasks show up
	virtual const bool canExecute() const = 0;			// should return true if the task could be executed right now
	virtual void execute() = 0;							// should execute the command
	virtual const bool isFinished() const = 0;			// should return true as soon as the Task has been finished completely
	virtual const bool isValid() const = 0;				// a function which should return true if the Task is still valid.
	virtual const bool scheduleConditions() const = 0;	// Should return true if the Task is ready to be scheduled (pre-conditions such as required tech built)

#ifdef MAASCRAFT_DEBUG
	virtual std::string getDescription() const = 0;
#endif
};