/*	-----------------------------------------------------------------------------
	M A A S C R A F T

	StarCraft: Brood War - Bot

	Author: Dennis Soemers
	Maastricht University
	-----------------------------------------------------------------------------
*/

/*
	Implementation of Squad.h
*/

#include "CommonIncludes.h"

#include "ArmyManager.h"
#include "OpponentTracker.h"
#include "Priorities.h"
#include "Squad.h"
#include "Tasks/FightSquad.hpp"
#include "Tasks/MoveSquad.hpp"
#include "Tasks/RetreatSquad.hpp"
#include "UnitUtils.h"

using namespace BWAPI;

#ifdef ARMY_MANAGER_UCT

Squad::Squad() : task(nullptr), region(nullptr), inCombat(false) 
{}

Squad::~Squad()
{
	if(task)
	{
		task->onEnd();
		delete task;
		task = nullptr;
	}
}

const Unitset Squad::getOpponentsInRegion() const
{
	return OpponentTracker::Instance()->getOpponentsInRegion(region);
}

const Unitset Squad::getPotentialTargets(Unit squadMember) const
{
	MoveSquad* moveTask = dynamic_cast<MoveSquad*>(task);

	if(moveTask)
	{
		return moveTask->getPotentialTargets(squadMember);
	}
	else
	{
		FightSquad* fightTask = dynamic_cast<FightSquad*>(task);
		
		if(fightTask)
		{
			return fightTask->getPotentialTargets(squadMember);
		}

		Unitset potentialTargets = UnitUtils::getUnitsInRadius(squadMember, 640, Filter::IsEnemy && Filter::IsDetected);

		if(potentialTargets.empty())
			potentialTargets = getOpponentsInRegion();

		return potentialTargets;
	}
}

const MapRegion* Squad::getRegion() const
{
	return region;
}

const Task* Squad::getTask() const
{
	return task;
}

const int Squad::getSize() const
{
	return getOwnedUnits().size();
}

const bool Squad::isInCombat() const
{
	return inCombat;
}

void Squad::setRegion(MapRegion* newRegion)
{
	region = newRegion;
}

void Squad::setTask(Action action)
{
#ifdef MAASCRAFT_DEBUG
	if(action.type() == RESOLVE_ACTIONS)
	{
		LOG_WARNING("Squad::setTask() called with action of type RESOLVE_ACTIONS")
		return;
	}
#endif

	if(action.type() == RETREAT_TO)
	{
		setTask(new RetreatSquad(Priorities::PRIORITY_COMBAT_RETREAT, this, action.destination()->getPosition()));
	}
	else if(action.type() == MOVE_TO)
	{
		setTask(new MoveSquad(Priorities::PRIORITY_COMBAT_ATTACK, this, action.destination(), false));
	}
	else if(action.type() == DO_NOTHING)
	{
		setTask(new MoveSquad(Priorities::PRIORITY_COMBAT_ATTACK, this, action.destination(), true));
	}
	else if(action.type() == FIGHT)
	{
		setTask(new FightSquad(Priorities::PRIORITY_COMBAT_ATTACK, this, action.destination()));
	}
}

void Squad::setTask(Task* newTask)
{
	if(task)
	{
		task->onEnd();
		delete task;
	}

	task = newTask;

	if(task)
		task->onStart();
}

void Squad::onFrame()
{
	inCombat = false;
	const Unitset& units = getOwnedUnits();

	for(auto it = units.begin(); it != units.end(); ++it)
	{
		if(UnitUtils::isInCombat(*it))
		{
			inCombat = true;
			break;
		}
	}

	if(task)
		task->execute();
}

#endif // ARMY_MANAGER_UCT

#ifdef ARMY_MANAGER_SCRIPTED

const Unitset Squad::getOpponentsInRegion() const
{
	return OpponentTracker::Instance()->getOpponentsInRegion(region);
}

const Unitset Squad::getPotentialTargets(Unit squadMember) const
{
	Unitset potentialTargets = UnitUtils::getUnitsInRadius(squadMember, 640, Filter::IsEnemy);

	if(potentialTargets.empty())
		potentialTargets = getOpponentsInRegion();

	return potentialTargets;
}

const MapRegion* Squad::getRegion() const
{
	return region;
}

void Squad::setRegion(MapRegion* newRegion)
{
	region = newRegion;
}

void Squad::addTask(Task* task)
{
	squadTaskSet.addTask(task);
}

void Squad::addTasks(const vector<Task*>& tasks)
{
	squadTaskSet.addTasks(tasks);
}

void Squad::clearTasks(const int maxClearPriority)
{
	squadTaskSet.clearTasks(maxClearPriority);
}

const Task* Squad::getCurrentTask() const
{
	return squadTaskSet.getCurrentTask();
}

const int Squad::getSize() const
{
	return getOwnedUnits().size();
}

const bool Squad::isInCombat() const
{
	return inCombat;
}

void Squad::onFrame()
{
	inCombat = false;
	const Unitset& units = getOwnedUnits();

	for(auto it = units.begin(); it != units.end(); ++it)
	{
		if(UnitUtils::isInCombat(*it))
		{
			inCombat = true;
			break;
		}
	}

	squadTaskSet.update();
	Task* currentTask = squadTaskSet.getCurrentTask();

	if(currentTask)
	{
		currentTask->execute();
	}
	else
	{
		Task* newTask = ArmyManager::Instance()->requestTask(this);

		if(newTask)
			addTask(newTask);
	}
}

#endif // ARMY_MANAGER_SCRIPTED