/*	-----------------------------------------------------------------------------
	M A A S C R A F T

	StarCraft: Brood War - Bot

	Author: Dennis Soemers
	Maastricht University
	-----------------------------------------------------------------------------
*/

/*
	Implementation of SimSquad.h
*/

#include "CommonIncludes.h"

#ifdef ARMY_MANAGER_UCT

#include "Distances.h"
#include "MapGraph.h"
#include "MapGraphEdge.h"
#include "MapGraphNode.h"
#include "MathConstants.h"
#include "OpponentTracker.h"
#include "SimSquad.h"

using namespace BWAPI;
using namespace std;

#pragma warning( push )
#pragma warning( disable:4244 )

SimSquad::SimSquad(const Unitset& units, const bool flyers) : 
	units(units), position(0, 0), topSpeed(MathConstants::MAX_DOUBLE), player(units.front()->getPlayer()), graphNode(nullptr), 
	previousNode(nullptr), airHP(0), groundHP(0), numGroundUnits(0.0f), numFlyers(0.0f), airDPS(0.0f), groundDPS(0.0f), flyers(flyers), 
	idle(true), inCombat(false), retreating(false), base(false), startSimInCombat(false), retreatAllowed(true)
{
	OpponentTracker* opponentTracker = OpponentTracker::Instance();
	Player self = Broodwar->self();
	Player enemy = Broodwar->enemy();

	int minDistSq = MathConstants::MAX_INT;
	Unit centreUnit = nullptr;

	Position squadPos;

	if(player == self)		// Squad of units owned by us
	{
		bool includesZealots = false;
		position = units.getPosition();

		for(auto it = units.begin(); it != units.end(); ++it)
		{
			Unit unit = *it;
			UnitType type = unit->getType();

			if(type.topSpeed() < topSpeed)
				topSpeed = type.topSpeed();

			WeaponType airWeapon = type.airWeapon();
			WeaponType groundWeapon = type.groundWeapon();

			if(airWeapon != WeaponTypes::None && airWeapon != WeaponTypes::Unknown)
			{
				airDPS += (self->damage(airWeapon) * (24.0f / airWeapon.damageCooldown()));

#ifdef MAASCRAFT_DEBUG
				if(airWeapon.damageCooldown() == 0)
				{
					LOG_WARNING(StringBuilder() << "Weapon cooldown = 0 for " << type)
				}
#endif
			}
			if(groundWeapon != WeaponTypes::None && groundWeapon != WeaponTypes::Unknown)
			{
				groundDPS += (self->damage(groundWeapon) * (24.0f / self->weaponDamageCooldown(type)));

				if(type == UnitTypes::Protoss_Zealot)
				{
					groundDPS += (self->damage(groundWeapon) * (24.0f / self->weaponDamageCooldown(type)));
					includesZealots = true;
				}

#ifdef MAASCRAFT_DEBUG
				if(self->weaponDamageCooldown(type) == 0)
				{
					LOG_WARNING(StringBuilder() << "Weapon cooldown = 0 for " << type)
				}
#endif
			}

			if(type.isFlyer())
			{
				airHP += unit->getHitPoints() + unit->getShields();
				numFlyers += 1.0f;
			}
			else if(!type.isWorker())
			{
				groundHP += unit->getHitPoints() + unit->getShields();
				numGroundUnits += 1.0f;
			}

			Position pos = unit->getPosition();
			const int distSq = Distances::getSquaredDistance(pos, position);

			if(distSq < minDistSq)
			{
				minDistSq = distSq;
				centreUnit = unit;
			}
		}

		squadPos = centreUnit->getPosition();

		if(includesZealots && units.size() >= 4)
			retreatAllowed = false;
	}
	else								// Squad of units owned by enemy
	{
		bool containsNonWorker = false;
		for(auto it = units.begin(); it != units.end(); ++it)
		{
			position += opponentTracker->getLastPosition(*it);

			if(!(opponentTracker->getLastType(*it).isWorker()))
			{
				containsNonWorker = true;
				break;
			}
		}

		position /= units.size();

		for(auto it = units.begin(); it != units.end(); ++it)
		{
			float addGroundDPS = 0.0f;
			float addAirDPS = 0.0f;

			Unit unit = *it;
			UnitType type = opponentTracker->getLastType(unit);

			if(type.topSpeed() < topSpeed)
				topSpeed = type.topSpeed();

			WeaponType airWeapon = type.airWeapon();
			WeaponType groundWeapon = type.groundWeapon();

			if(airWeapon != WeaponTypes::None && airWeapon != WeaponTypes::Unknown)
			{
				addAirDPS += (enemy->damage(airWeapon) * (24.0f / airWeapon.damageCooldown()));

#ifdef MAASCRAFT_DEBUG
				if(airWeapon.damageCooldown() == 0)
				{
					LOG_WARNING(StringBuilder() << "Weapon cooldown = 0 for " << type)
				}
#endif
			}
			if(groundWeapon != WeaponTypes::None && groundWeapon != WeaponTypes::Unknown && !type.isWorker())
			{
				addGroundDPS += (enemy->damage(groundWeapon) * (24.0f / enemy->weaponDamageCooldown(type)));

				if(type == UnitTypes::Protoss_Zealot)
					addGroundDPS += (enemy->damage(groundWeapon) * (24.0f / enemy->weaponDamageCooldown(type)));

#ifdef MAASCRAFT_DEBUG
				if(enemy->weaponDamageCooldown(type) == 0)
				{
					LOG_WARNING(StringBuilder() << "Weapon cooldown = 0 for " << type)
				}
#endif
			}

			if(type == UnitTypes::Terran_Bunker && (unit->isCompleted() || !unit->isVisible()))
			{
				addGroundDPS += 38.4f;
				addAirDPS += 38.4f;
			}

			if(type == UnitTypes::Terran_Bunker && !unit->isVisible())
			{
				groundHP += 0.25 * type.maxHitPoints();
			}
			else if(type == UnitTypes::Protoss_Photon_Cannon && !unit->isVisible())
			{
				groundHP += type.maxShields();
			}

			Position pos = opponentTracker->getLastPosition(unit);

			float hpRatio = 0.0f;

			if(type.maxHitPoints() > 0)
			{
				hpRatio = (opponentTracker->getLastHP(unit) + unit->getShields()) / (type.maxHitPoints() + type.maxShields());
			}

			if(!unit->isVisible())
				hpRatio = 1.0f;

			if(hpRatio > 1.1f)
				hpRatio = 1.1f;

			airDPS += addAirDPS * hpRatio;
			groundDPS += addGroundDPS * hpRatio;

			if(type.isFlyer())
			{
				airHP += opponentTracker->getLastHP(unit) + unit->getShields();
				numFlyers += 1.0f;
			}
			else if(!type.isWorker())
			{
				if(unit->isVisible() || opponentTracker->wasCompleted(unit))
					groundHP += opponentTracker->getLastHP(unit) + unit->getShields();
				else
					groundHP += type.maxHitPoints() + type.maxShields();

				numGroundUnits += 1.0f;
			}
			else if(!containsNonWorker)
			{
				groundHP += 0.25 * opponentTracker->getLastHP(unit);
				numGroundUnits += 0.05f;
			}

			const int distSq = Distances::getSquaredDistance(pos, position);

			if(distSq < minDistSq)
			{
				minDistSq = distSq;
				centreUnit = unit;
			}
		}

		squadPos = opponentTracker->getLastPosition(centreUnit);

		if(squadPos == Positions::Unknown)
		{
			airHP = 0;
			groundHP = 0;
			numGroundUnits = 0.0f;
			numFlyers = 0.0f;
			idle = true;
			return;
		}
	}

	if(topSpeed <= 0.0)
		idle = false;

#ifdef MAASCRAFT_DEBUG
	if(!centreUnit)
	{
		LOG_WARNING("SimSquad constructor: No centre unit found!")
		LOG_WARNING(StringBuilder() << "Num units = " << units.size())
		return;
	}
#endif

	MapAnalyser* mapAnalyser = MapAnalyser::Instance();
	MapRegion* region = mapAnalyser->getRegion(squadPos);

	const MapGraph* graph = mapAnalyser->getGraph();
	const vector<MapGraphNode*>& potentialNodes = graph->getRegionNodes(region);

	minDistSq = MathConstants::MAX_INT;
	MapGraphNode* nearestNode = nullptr;

	for(auto it = potentialNodes.begin(); it != potentialNodes.end(); ++it)
	{
		MapGraphNode* node = *it;

		const int distSq = Distances::getSquaredDistance(squadPos, node->getPosition());
		if(distSq < minDistSq)
		{
			minDistSq = distSq;
			nearestNode = node;
		}
	}

	graphNode = nearestNode;

	if(topSpeed == 0.0)
		idle = false;
}

SimSquad::SimSquad(const Base& base) : 
	units(base.getBuildings()), position(base.getDepotCenter()), topSpeed(0.0), player(base.getOwner()), graphNode(nullptr), 
	previousNode(nullptr), airHP(0), groundHP(2500), numGroundUnits(1.0f), numFlyers(0.0f), airDPS(0.01f), groundDPS(0.01f), flyers(false), 
	idle(false), inCombat(false), retreating(false), base(true), startSimInCombat(false), retreatAllowed(true)
{
	MapAnalyser* mapAnalyser = MapAnalyser::Instance();
	MapRegion* region = base.getLocation()->getRegion();

	const MapGraph* graph = mapAnalyser->getGraph();
	const vector<MapGraphNode*>& potentialNodes = graph->getRegionNodes(region);

	int minDistSq = MathConstants::MAX_INT;
	MapGraphNode* nearestNode = nullptr;

	for(auto it = potentialNodes.begin(); it != potentialNodes.end(); ++it)
	{
		MapGraphNode* node = *it;

		const int distSq = Distances::getSquaredDistance(position, node->getPosition());
		if(distSq < minDistSq)
		{
			minDistSq = distSq;
			nearestNode = node;
		}
	}

	graphNode = nearestNode;
}

SimSquad::~SimSquad()
{}

const bool SimSquad::allowRetreat() const
{
	return retreatAllowed;
}

MapGraphNode* SimSquad::getGraphNode() const
{
	return graphNode;
}

Position SimSquad::getPosition() const
{
	return position;
}

MapGraphNode* SimSquad::getPreviousNode() const
{
	return previousNode;
}

const bool SimSquad::flyersOnly() const
{
	return flyers;
}

const int SimSquad::getAirHP() const
{
	static const float RETREAT_HEALTH_BONUS_FACTOR = 1.5f;

	if(startSimInCombat && player == Broodwar->self())
	{
		if(retreating)
			return airHP;
		else
			return airHP * 1.60f;
	}
	else
	{
		if(retreating)
			return airHP * RETREAT_HEALTH_BONUS_FACTOR;
		else
			return airHP;
	}
}

const int SimSquad::getGroundHP() const
{
	static const float RETREAT_HEALTH_BONUS_FACTOR = 1.5f;

	if(startSimInCombat && player == Broodwar->self())
	{
		if(retreating)
			return groundHP;
		else
			return groundHP * 1.60f;
	}
	else
	{
		if(retreating)
			return groundHP * RETREAT_HEALTH_BONUS_FACTOR;
		else
			return groundHP;
	}
}

const float SimSquad::getAirDPS() const
{
	static const float RETREAT_DAMAGE_DEALT_FACTOR = 0.01f;

	if(retreating)
		return airDPS * RETREAT_DAMAGE_DEALT_FACTOR;
	else
		return airDPS;
}

const float SimSquad::getGroundDPS() const
{
	static const float RETREAT_DAMAGE_DEALT_FACTOR = 0.01f;

	if(getPlayer() == Broodwar->self())
	{
		if(retreating)
			return groundDPS * RETREAT_DAMAGE_DEALT_FACTOR;
		else
			return groundDPS;
	}
	else
	{
		if(retreating)
			return groundDPS * RETREAT_DAMAGE_DEALT_FACTOR;
		else
			return groundDPS;
	}
}

const float SimSquad::getNumFlyers() const
{
	return numFlyers;
}

const float SimSquad::getNumGroundUnits() const
{
	return numGroundUnits;
}

const double SimSquad::getTopSpeed() const
{
	return topSpeed;
}

const int SimSquad::getTrueAirHP() const
{
	return airHP;
}

const int SimSquad::getTrueGroundHP() const
{
	return groundHP;
}

const Player SimSquad::getPlayer() const
{
	return player;
}

const Unitset& SimSquad::getUnits() const
{
	return units;
}

const bool SimSquad::isBase() const
{
	return base;
}

const bool SimSquad::isIdle() const
{
	return idle;
}

const bool SimSquad::isInCombat() const
{
	return inCombat;
}

const bool SimSquad::isRetreating() const
{
	return retreating;
}

void SimSquad::modAirHP(const int mod)
{
	static const float RETREAT_DAMAGE_TAKEN_FACTOR = 1.0f / 1.5f;

	if(retreating)
		airHP += RETREAT_DAMAGE_TAKEN_FACTOR * mod;
	else
		airHP += mod;
}

void SimSquad::modGroundHP(const int mod)
{
	static const float RETREAT_DAMAGE_TAKEN_FACTOR = 1.0f / 1.5f; 

	if(retreating)
		groundHP += RETREAT_DAMAGE_TAKEN_FACTOR * mod;
	else
		groundHP += mod;
}

void SimSquad::modNumFlyers(const float mod)
{
	numFlyers += mod;
}

void SimSquad::modNumGroundUnits(const float mod)
{
	numGroundUnits += mod;
}

void SimSquad::setGraphNode(MapGraphNode* newNode)
{
	graphNode = newNode;
}

void SimSquad::setIdle(const bool status)
{
	idle = status;
}

void SimSquad::setInCombat(const bool status)
{
	inCombat = status;
}

void SimSquad::setNumFlyers(const int num)
{
	numFlyers = num;
}

void SimSquad::setNumGroundUnits(const int num)
{
	numGroundUnits = num;
}

void SimSquad::setPosition(const BWAPI::Position& pos)
{
	position = pos;
}

void SimSquad::setPreviousNode(MapGraphNode* node)
{
	previousNode = node;
}

void SimSquad::setRetreating(const bool status)
{
	retreating = status;
}

void SimSquad::setAirHP(const int hp)
{
	airHP = hp;
}

void SimSquad::setGroundHP(const int hp)
{
	groundHP = hp;
}

void SimSquad::setAirDPS(const float dps)
{
	airDPS = dps;
}

void SimSquad::setGroundDPS(const float dps)
{
	groundDPS = dps;
}

void SimSquad::setStartsInCombat(const bool flag)
{
	startSimInCombat = flag;
}

const bool SimSquad::startedSimInCombat() const
{
	return startSimInCombat;
}

#pragma warning( pop )

#endif // ARMY_MANAGER_UCT