/*	-----------------------------------------------------------------------------
	M A A S C R A F T

	StarCraft: Brood War - Bot

	Author: Dennis Soemers
	Maastricht University
	-----------------------------------------------------------------------------
*/

/*
	Implementation of SimBattle.h
*/

#include "CommonIncludes.h"

#ifdef ARMY_MANAGER_UCT

#include <set>
#include <vector>

#include "SimBattle.h"
#include "SimSquad.h"

#pragma warning( push )
#pragma warning( disable:4244 )

using namespace std;

SimBattle::SimBattle(MapGraphNode* node) : 
	allySquadIndices(), enemySquadIndices(), allyBaseIndices(), enemyBaseIndices(), lanchester(), node(node), totEnemyAirDPF(0.0f), 
	totEnemyGroundDPF(0.0f), totSelfAirDPF(0.0f), totSelfGroundDPF(0.0f), totEnemyAirHP(0), totEnemyGroundHP(0), 
	totSelfAirHP(0), totSelfGroundHP(0), totNumEnemyFlyers(0.0f), totNumEnemyGroundUnits(0.0f), totNumSelfFlyers(0.0f),
	totNumSelfGroundUnits(0.0f), totNumEnemyUnits(0.0f), totNumSelfUnits(0.0f), durationRemaining(0)
{}

SimBattle::~SimBattle()
{}

void SimBattle::addAllySquad(const int squadIndex, GameState& gameState)
{
	allySquadIndices.push_back(squadIndex);
}

void SimBattle::addEnemySquad(const int squadIndex, GameState& gameState)
{
	enemySquadIndices.push_back(squadIndex);
}

void SimBattle::addAllyBase(const int baseIndex, GameState& gameState)
{
	allyBaseIndices.push_back(baseIndex);
}

void SimBattle::addEnemyBase(const int baseIndex, GameState& gameState)
{
	enemyBaseIndices.push_back(baseIndex);
}

MapGraphNode* SimBattle::getGraphNode() const
{
	return node;
}

const int SimBattle::getRemainingDuration() const
{
	return durationRemaining;
}

const bool SimBattle::isFinished() const
{
	return ((allySquadIndices.empty() && allyBaseIndices.empty()) || (enemySquadIndices.empty() && enemyBaseIndices.empty()));
}

#ifdef BATTLE_MODEL_LANCHESTER
void SimBattle::recomputeRemainingDuration(GameState& gameState)
{
	const vector<SimSquad>& enemySquads = gameState.getEnemySquads();
	const vector<SimSquad>& selfSquads = gameState.getSelfSquads();
	const vector<SimSquad>& enemyBases = gameState.getEnemyBases();
	const vector<SimSquad>& selfBases = gameState.getSelfBases();

	totEnemyAirHP = 0;
	totEnemyGroundHP = 0;
	totSelfAirHP = 0;
	totSelfGroundHP = 0;

	totEnemyAirDPF = 0.001f;
	totEnemyGroundDPF = 0.001f;
	totSelfAirDPF = 0.001f;
	totSelfGroundDPF = 0.001f;

	for(auto it = allySquadIndices.begin(); it != allySquadIndices.end(); ++it)
	{
		const int i = *it;
		const SimSquad& squad = selfSquads.at(i);

		totSelfAirHP += squad.getAirHP();
		totSelfGroundHP += squad.getGroundHP();
		totSelfAirDPF += squad.getAirDPS();
		totSelfGroundDPF += squad.getGroundDPS();

		totNumSelfGroundUnits += squad.getNumGroundUnits();
		totNumSelfFlyers += squad.getNumFlyers();
	}

	for(auto it = enemySquadIndices.begin(); it != enemySquadIndices.end(); ++it)
	{
		const int i = *it;
		const SimSquad& squad = enemySquads.at(i);

		totEnemyAirHP += squad.getAirHP();
		totEnemyGroundHP += squad.getGroundHP();
		totEnemyAirDPF += squad.getAirDPS();
		totEnemyGroundDPF += squad.getGroundDPS();

		totNumEnemyGroundUnits += squad.getNumGroundUnits();
		totNumEnemyFlyers += squad.getNumFlyers();
	}

	for(auto it = allyBaseIndices.begin(); it != allyBaseIndices.end(); ++it)
	{
		const int i = *it;
		const SimSquad& base = selfBases.at(i);

		totSelfAirHP += base.getAirHP();
		totSelfGroundHP += base.getGroundHP();
		totSelfAirDPF += base.getAirDPS();
		totSelfGroundDPF += base.getGroundDPS();

		totNumSelfGroundUnits += base.getNumGroundUnits();
	}

	for(auto it = enemyBaseIndices.begin(); it != enemyBaseIndices.end(); ++it)
	{
		const int i = *it;
		const SimSquad& base = enemyBases.at(i);

		totEnemyAirHP += base.getAirHP();
		totEnemyGroundHP += base.getGroundHP();
		totEnemyAirDPF += base.getAirDPS();
		totEnemyGroundDPF += base.getGroundDPS();

		totNumEnemyGroundUnits += base.getNumGroundUnits();
	}

	totNumEnemyUnits = totNumEnemyGroundUnits + totNumEnemyFlyers;
	totNumSelfUnits = totNumSelfGroundUnits + totNumSelfFlyers;

	totEnemyAirDPF = (totEnemyAirDPF / 24.0f);
	totEnemyGroundDPF = (totEnemyGroundDPF / 24.0f);
	totSelfAirDPF = (totSelfAirDPF / 24.0f);
	totSelfGroundDPF = (totSelfGroundDPF / 24.0f);

	float averageHealthAirEnemy = totEnemyAirHP == 0.0f ? 0.0f : totEnemyAirHP / float(totNumEnemyFlyers);
	float averageHealthAirSelf = totSelfAirHP == 0.0f ? 0.0f : totSelfAirHP / float(totNumSelfFlyers);
	float averageHealthGroundEnemy = totEnemyGroundHP == 0.0f ? 0.0f : totEnemyGroundHP / float(totNumEnemyGroundUnits);
	float averageHealthGroundSelf = totSelfGroundHP == 0.0f ? 0.0f : totSelfGroundHP / float(totNumSelfGroundUnits);

	float sumAverageHealthsEnemy = averageHealthAirEnemy + averageHealthGroundEnemy;
	float sumAvergeHealthsSelf = averageHealthAirSelf + averageHealthGroundSelf;

	float averageHealthEnemy = (totEnemyAirHP + totEnemyGroundHP) / float(totNumEnemyUnits);
	float averageHealthSelf = (totSelfAirHP + totSelfGroundHP) / float(totNumSelfUnits);

	float averageEnemyAirDPF = totEnemyAirDPF / totNumEnemyUnits;
	float averageEnemyGroundDPF = totEnemyGroundDPF / totNumEnemyUnits;
	float averageSelfAirDPF = totSelfAirDPF / totNumSelfUnits;
	float averageSelfGroundDPF = totSelfGroundDPF / totNumSelfUnits;

	float averageEnemyDPF = averageEnemyAirDPF * (averageHealthAirSelf / sumAvergeHealthsSelf)	+ 
							averageEnemyGroundDPF * (averageHealthGroundSelf / sumAvergeHealthsSelf);
	float averageSelfDPF = averageSelfAirDPF * (averageHealthAirEnemy / sumAverageHealthsEnemy)	+ 
							averageSelfGroundDPF * (averageHealthGroundEnemy / sumAverageHealthsEnemy);

	lanchester.a = averageEnemyDPF / averageHealthSelf;
	lanchester.b = averageSelfDPF / averageHealthEnemy;

	lanchester.P = 0.0;
	lanchester.Q = 0.0;
	
	/* For now, results seemed to be better without using P and Q
	if(allyBaseIndices.size() == 0)	// reinforcements only allowed if the base itself is not in combat
	{
		for(auto it = selfBases.begin(); it != selfBases.end(); ++it)
		{
			const SimSquad& base = *it;

			if(base.getGraphNode() == node)		// we have a base at this battle node
			{
				lanchester.P = (2.0 / BWAPI::UnitTypes::Protoss_Zealot.buildTime());
				break;
			}
		}
	}

	if(enemyBaseIndices.size() == 0)
	{
		for(auto it = enemyBases.begin(); it != enemyBases.end(); ++it)
		{
			const SimSquad& base = *it;

			if(base.getGraphNode() == node)		// enemy has a base at this battle node
			{
				lanchester.Q = (0.5 / BWAPI::UnitTypes::Protoss_Zealot.buildTime());
				break;
			}
		}
	}
	*/

	// weird variable names correspond to variables and equations in thesis
	double root_ab = sqrt(lanchester.a * lanchester.b);
	double m0_term = (totNumSelfUnits + (lanchester.P / root_ab));
	double n0_term = (totNumEnemyUnits + (lanchester.Q / root_ab));

	lanchester.E = 0.5 * ( m0_term - (root_ab / lanchester.b) * n0_term );
	lanchester.F = 0.5 * ( (root_ab / lanchester.a) * m0_term + n0_term );

	if(lanchester.E > 0.0)			// we will win the battle
	{
		double discriminant = lanchester.P * lanchester.P + 4 * lanchester.a * lanchester.E * lanchester.F * root_ab;
		double z = (-lanchester.P + sqrt(discriminant)) / (2.0 * lanchester.a * lanchester.F);
		durationRemaining = ceil((-log(z)) / root_ab);
	}
	else if(lanchester.E < 0.0)		// enemy will win the battle
	{
		double discriminant = lanchester.Q * lanchester.Q - 4 * lanchester.b * lanchester.E * lanchester.F * root_ab;
		double z = (-lanchester.Q + sqrt(discriminant)) / (2.0 * lanchester.F * root_ab);
		durationRemaining = ceil((-log(z)) / root_ab);
	}
	else							// draw
	{
		durationRemaining = ceil((-log(0.001)) / root_ab);		// this seemed to approximate something reasonable in a couple of tests
	}
}

void SimBattle::simulateBattle(const int frameDuration, GameState& gameState)
{
	vector<SimSquad>& enemySquads = gameState.getEnemySquads();
	vector<SimSquad>& selfSquads = gameState.getSelfSquads();
	vector<SimSquad>& enemyBases = gameState.getEnemyBases();
	vector<SimSquad>& selfBases = gameState.getSelfBases();

	double root_ab = sqrt(lanchester.a * lanchester.b);

	float numSelfUnitsRemaining = ((lanchester.Q / lanchester.b)						+ 
									lanchester.E * std::exp(frameDuration*root_ab)		+ 
									(root_ab / lanchester.b) * lanchester.F * std::exp(-frameDuration*root_ab));

	float numEnemyUnitsRemaining = ((lanchester.P / lanchester.a)												-
									(root_ab / lanchester.a) * lanchester.E * std::exp(frameDuration*root_ab)	+
									lanchester.F * std::exp(-frameDuration*root_ab)								);

	if(numSelfUnitsRemaining < 0.0f)
		numSelfUnitsRemaining = 0.0f;
	
	if(numEnemyUnitsRemaining < 0.0f)
		numEnemyUnitsRemaining = 0.0f;

	for(auto it = allySquadIndices.begin(); it != allySquadIndices.end(); ++it)
	{
		if(totNumSelfUnits <= numSelfUnitsRemaining)
			break;

		SimSquad& squad = selfSquads.at(*it);
		float squadGroundUnits = squad.getNumGroundUnits();
		float squadFlyers = squad.getNumFlyers();
		float previousNumUnits = squadFlyers + squadGroundUnits;

		float averageSquadGroundHP = squadGroundUnits <= 0.0f ? 0.0f : float(squad.getGroundHP()) / squadGroundUnits;
		float reduction = std::min(totNumSelfUnits - numSelfUnitsRemaining, squadGroundUnits);
		squad.modNumGroundUnits(-reduction);
		squad.modGroundHP(-averageSquadGroundHP * reduction);
		totNumSelfUnits -= reduction;

		float averageSquadAirHP = squadFlyers <= 0.0f ? 0.0f : float(squad.getAirHP()) / squadFlyers;
		reduction = std::min(totNumSelfUnits - numSelfUnitsRemaining, squadFlyers);
		squad.modNumFlyers(-reduction);
		squad.modAirHP(-averageSquadAirHP * reduction);
		totNumSelfUnits -= reduction;

		float newNumUnits = squad.getNumFlyers() + squad.getNumGroundUnits();
		squad.setAirDPS(squad.getAirDPS() * newNumUnits / previousNumUnits);
		squad.setGroundDPS(squad.getGroundDPS() * newNumUnits / previousNumUnits);
	}

	for(auto it = allyBaseIndices.begin(); it != allyBaseIndices.end(); ++it)
	{
		if(totNumSelfUnits <= numSelfUnitsRemaining)
			break;

		SimSquad& base = selfBases.at(*it);
		float previousNumUnits = base.getNumGroundUnits();

		float averageBaseGroundHP = float(base.getGroundHP()) / base.getNumGroundUnits();
		int reduction = std::min(totNumSelfUnits - numSelfUnitsRemaining, base.getNumGroundUnits());
		base.modNumGroundUnits(-reduction);
		base.modGroundHP(-averageBaseGroundHP * reduction);
		totNumSelfUnits -= reduction;

		float newNumUnits = base.getNumGroundUnits();
		base.setAirDPS(base.getAirDPS() * newNumUnits / previousNumUnits);
		base.setGroundDPS(base.getGroundDPS() * newNumUnits / previousNumUnits);
	}

	for(auto it = enemySquadIndices.begin(); it != enemySquadIndices.end(); ++it)
	{
		if(totNumEnemyUnits <= numEnemyUnitsRemaining)
			break;

		SimSquad& squad = enemySquads.at(*it);
		float squadGroundUnits = squad.getNumGroundUnits();
		float squadFlyers = squad.getNumFlyers();
		float previousNumUnits = squadFlyers + squadGroundUnits;

		float averageSquadGroundHP = squadGroundUnits <= 0.0f ? 0.0f : float(squad.getGroundHP()) / squadGroundUnits;
		float reduction = std::min(totNumEnemyUnits - numEnemyUnitsRemaining, squadGroundUnits);
		squad.modNumGroundUnits(-reduction);
		squad.modGroundHP(-averageSquadGroundHP * reduction);
		totNumEnemyUnits -= reduction;

		float averageSquadAirHP = squadFlyers <= 0.0f ? 0.0f : float(squad.getAirHP()) / squadFlyers;
		reduction = std::min(totNumEnemyUnits - numEnemyUnitsRemaining, squadFlyers);
		squad.modNumFlyers(-reduction);
		squad.modAirHP(-averageSquadAirHP * reduction);
		totNumEnemyUnits -= reduction;

		float newNumUnits = squad.getNumFlyers() + squad.getNumGroundUnits();
		squad.setAirDPS(squad.getAirDPS() * newNumUnits / previousNumUnits);
		squad.setGroundDPS(squad.getGroundDPS() * newNumUnits / previousNumUnits);
	}

	for(auto it = enemyBaseIndices.begin(); it != enemyBaseIndices.end(); ++it)
	{
		if(totNumEnemyUnits <= numEnemyUnitsRemaining)
			break;

		SimSquad& base = enemyBases.at(*it);
		float previousNumUnits = base.getNumGroundUnits();

		float averageBaseGroundHP = float(base.getGroundHP()) / base.getNumGroundUnits();
		float reduction = std::min(totNumEnemyUnits - numEnemyUnitsRemaining, base.getNumGroundUnits());
		base.modNumGroundUnits(-reduction);
		base.modGroundHP(-averageBaseGroundHP * reduction);
		totNumEnemyUnits -= reduction;

		float newNumUnits = base.getNumGroundUnits();
		base.setAirDPS(base.getAirDPS() * newNumUnits / previousNumUnits);
		base.setGroundDPS(base.getGroundDPS() * newNumUnits / previousNumUnits);
	}

	// remove squads with 0 health remaining
	for(auto it = allySquadIndices.begin(); it != allySquadIndices.end(); /**/)
	{
		const SimSquad& squad = selfSquads.at(*it);

		if(squad.getAirHP() <= 0 && squad.getGroundHP() <= 0)
		{
			it = allySquadIndices.erase(it);
		}
		else
		{
			++it;
		}
	}

	for(auto it = enemySquadIndices.begin(); it != enemySquadIndices.end(); /**/)
	{
		const SimSquad& squad = enemySquads.at(*it);

		if(squad.getAirHP() <= 0 && squad.getGroundHP() <= 0)
		{
			it = enemySquadIndices.erase(it);
		}
		else
		{
			++it;
		}
	}

	// remove bases with 0 health remaining
	for(auto it = allyBaseIndices.begin(); it != allyBaseIndices.end(); /**/)
	{
		const SimSquad& base = selfBases.at(*it);

		if(base.getAirHP() <= 0 && base.getGroundHP() <= 0)
		{
			it = allyBaseIndices.erase(it);
		}
		else
		{
			++it;
		}
	}

	for(auto it = enemyBaseIndices.begin(); it != enemyBaseIndices.end(); /**/)
	{
		const SimSquad& base = enemyBases.at(*it);

		if(base.getAirHP() <= 0 && base.getGroundHP() <= 0)
		{
			it = enemyBaseIndices.erase(it);
		}
		else
		{
			++it;
		}
	}

	durationRemaining -= frameDuration;
}
#else

void SimBattle::recomputeRemainingDuration(GameState& gameState)
{
	const vector<SimSquad>& enemySquads = gameState.getEnemySquads();
	const vector<SimSquad>& selfSquads = gameState.getSelfSquads();
	const vector<SimSquad>& enemyBases = gameState.getEnemyBases();
	const vector<SimSquad>& selfBases = gameState.getSelfBases();

	totEnemyAirHP = 0;
	totEnemyGroundHP = 0;
	totSelfAirHP = 0;
	totSelfGroundHP = 0;

	totEnemyAirDPF = 0.001f;
	totEnemyGroundDPF = 0.001f;
	totSelfAirDPF = 0.001f;
	totSelfGroundDPF = 0.001f;

	for(auto it = allySquadIndices.begin(); it != allySquadIndices.end(); ++it)
	{
		const int i = *it;
		const SimSquad& squad = selfSquads[i];

		totSelfAirHP += squad.getAirHP();
		totSelfGroundHP += squad.getGroundHP();
		totSelfAirDPF += squad.getAirDPS();
		totSelfGroundDPF += squad.getGroundDPS();
	}

	for(auto it = enemySquadIndices.begin(); it != enemySquadIndices.end(); ++it)
	{
		const int i = *it;
		const SimSquad& squad = enemySquads[i];

		totEnemyAirHP += squad.getAirHP();
		totEnemyGroundHP += squad.getGroundHP();
		totEnemyAirDPF += squad.getAirDPS();
		totEnemyGroundDPF += squad.getGroundDPS();
	}

	for(auto it = allyBaseIndices.begin(); it != allyBaseIndices.end(); ++it)
	{
		const int i = *it;
		const SimSquad& base = selfBases[i];

		totSelfAirHP += base.getAirHP();
		totSelfGroundHP += base.getGroundHP();
		totSelfAirDPF += base.getAirDPS();
		totSelfGroundDPF += base.getGroundDPS();
	}

	for(auto it = enemyBaseIndices.begin(); it != enemyBaseIndices.end(); ++it)
	{
		const int i = *it;
		const SimSquad& base = enemyBases[i];

		totEnemyAirHP += base.getAirHP();
		totEnemyGroundHP += base.getGroundHP();
		totEnemyAirDPF += base.getAirDPS();
		totEnemyGroundDPF += base.getGroundDPS();
	}

	totEnemyAirDPF = (totEnemyAirDPF / 24.0f);
	totEnemyGroundDPF = (totEnemyGroundDPF / 24.0f);
	totSelfAirDPF = (totSelfAirDPF / 24.0f);
	totSelfGroundDPF = (totSelfGroundDPF / 24.0f);

	float enemyTimeRequired = (totSelfAirHP / totEnemyAirDPF) + (totSelfGroundHP / totEnemyGroundDPF);
	float selfTimeRequired = (totEnemyAirHP / totSelfAirDPF) + (totEnemyGroundHP / totSelfGroundDPF);

	durationRemaining = ceil(min(enemyTimeRequired, selfTimeRequired));
}

void SimBattle::simulateBattle(const int frameDuration, GameState& gameState)
{
	vector<SimSquad>& enemySquads = gameState.getEnemySquads();
	vector<SimSquad>& selfSquads = gameState.getSelfSquads();
	vector<SimSquad>& enemyBases = gameState.getEnemyBases();
	vector<SimSquad>& selfBases = gameState.getSelfBases();

	int enemyAirDamage = ceil(totEnemyAirDPF * frameDuration);
	int enemyGroundDamage = ceil(totEnemyGroundDPF * frameDuration);
	int selfAirDamage = ceil(totSelfAirDPF * frameDuration);
	int selfGroundDamage = ceil(totSelfGroundDPF * frameDuration);
	
	for(auto it = allySquadIndices.begin(); it != allySquadIndices.end(); ++it)
	{
		SimSquad& squad = selfSquads[*it];
		int airDamage = min(enemyAirDamage, squad.getAirHP());
		int groundDamage = min(enemyGroundDamage, squad.getGroundHP());

		squad.modAirHP(-airDamage);
		enemyAirDamage -= airDamage;
		totSelfAirHP -= airDamage;

		squad.modGroundHP(-groundDamage);
		enemyGroundDamage -= groundDamage;
		totSelfGroundHP -= groundDamage;

		if(enemyAirDamage == 0 && enemyGroundDamage == 0)
			break;
	}

	for(auto it = allyBaseIndices.begin(); it != allyBaseIndices.end(); ++it)
	{
		SimSquad& base = selfBases[*it];
		int airDamage = min(enemyAirDamage, base.getAirHP());
		int groundDamage = min(enemyGroundDamage, base.getGroundHP());

		base.modAirHP(-airDamage);
		enemyAirDamage -= airDamage;
		totSelfAirHP -= airDamage;

		base.modGroundHP(-groundDamage);
		enemyGroundDamage -= groundDamage;
		totSelfGroundHP -= groundDamage;

		if(enemyAirDamage == 0 && enemyGroundDamage == 0)
			break;
	}

	for(auto it = enemySquadIndices.begin(); it != enemySquadIndices.end(); ++it)
	{
		SimSquad& squad = enemySquads[*it];
		int airDamage = min(selfAirDamage, squad.getAirHP());
		int groundDamage = min(selfGroundDamage, squad.getGroundHP());

		squad.modAirHP(-airDamage);
		selfAirDamage -= airDamage;
		totEnemyAirHP -= airDamage;

		squad.modGroundHP(-groundDamage);
		selfGroundDamage -= groundDamage;
		totEnemyGroundHP -= groundDamage;

		if(selfAirDamage == 0 && selfGroundDamage == 0)
			break;
	}

	for(auto it = enemyBaseIndices.begin(); it != enemyBaseIndices.end(); ++it)
	{
		SimSquad& base = enemyBases[*it];
		int airDamage = min(selfAirDamage, base.getAirHP());
		int groundDamage = min(selfGroundDamage, base.getGroundHP());

		base.modAirHP(-airDamage);
		selfAirDamage -= airDamage;
		totEnemyAirHP -= airDamage;

		base.modGroundHP(-groundDamage);
		selfGroundDamage -= groundDamage;
		totEnemyGroundHP -= groundDamage;

		if(selfAirDamage == 0 && selfGroundDamage == 0)
			break;
	}

	// remove squads with 0 health remaining
	for(auto it = allySquadIndices.begin(); it != allySquadIndices.end(); /**/)
	{
		const SimSquad& squad = selfSquads[*it];

		if(squad.getAirHP() <= 0 && squad.getGroundHP() <= 0)
		{
			it = allySquadIndices.erase(it);
		}
		else
		{
			++it;
		}
	}

	for(auto it = enemySquadIndices.begin(); it != enemySquadIndices.end(); /**/)
	{
		const SimSquad& squad = enemySquads[*it];

		if(squad.getAirHP() <= 0 && squad.getGroundHP() <= 0)
		{
			it = enemySquadIndices.erase(it);
		}
		else
		{
			++it;
		}
	}

	// remove bases with 0 health remaining
	for(auto it = allyBaseIndices.begin(); it != allyBaseIndices.end(); /**/)
	{
		const SimSquad& base = selfBases[*it];

		if(base.getAirHP() <= 0 && base.getGroundHP() <= 0)
		{
			it = allyBaseIndices.erase(it);
		}
		else
		{
			++it;
		}
	}

	for(auto it = enemyBaseIndices.begin(); it != enemyBaseIndices.end(); /**/)
	{
		const SimSquad& base = enemyBases[*it];

		if(base.getAirHP() <= 0 && base.getGroundHP() <= 0)
		{
			it = enemyBaseIndices.erase(it);
		}
		else
		{
			++it;
		}
	}

	durationRemaining -= frameDuration;

#ifdef MAASCRAFT_DEBUG
	if(durationRemaining == 0 && !isFinished())
	{
		LOG_WARNING("SimBattle durationRemaining = 0 but battle is not yet finished!")
		LOG_MESSAGE(StringBuilder() << "totEnemyAirHP = " << totEnemyAirHP)
		LOG_MESSAGE(StringBuilder() << "totEnemyGroundHP = " << totEnemyGroundHP)
		LOG_MESSAGE(StringBuilder() << "totSelfAirHP = " << totSelfAirHP)
		LOG_MESSAGE(StringBuilder() << "totSelfGroundHP = " << totSelfGroundHP)
		for(auto it = allySquadIndices.begin(); it != allySquadIndices.end(); ++it)
		{
			const SimSquad& squad = selfSquads[*it];
			LOG_MESSAGE(StringBuilder() << "Self squad has airHP = " << squad.getAirHP() << " and groundHP = " << squad.getGroundHP())
		}
		for(auto it = enemySquadIndices.begin(); it != enemySquadIndices.end(); ++it)
		{
			const SimSquad& squad = enemySquads[*it];
			LOG_MESSAGE(StringBuilder() << "Enemy squad has airHP = " << squad.getAirHP() << " and groundHP = " << squad.getGroundHP())
		}
		LOG_MESSAGE("")
	}
#endif
}

#endif // BATTLE_MODEL_LANCHESTER

#pragma warning( pop )

#endif // ARMY_MANAGER_UCT