/*	-----------------------------------------------------------------------------
	M A A S C R A F T

	StarCraft: Brood War - Bot

	Author: Dennis Soemers
	Maastricht University
	-----------------------------------------------------------------------------
*/

#include "CommonIncludes.h"

#include <vector>

#include "Base.h"
#include "BaseLocation.h"
#include "BaseManager.h"
#include "MapAnalyser.h"
#include "MathConstants.h"
#include "ScoutManager.h"
#include "Task.h"
#include "Tasks/AssignScoutWorker.hpp"
#include "Tasks/TaskSchedulePreds.h"
#include "Tasks/ScoutBaseLocation.hpp"
#include "UnitTracker.h"

/*
	Implementation of the ScoutManager
*/

using namespace BWAPI;
using namespace std;

void ScoutManager::addTask(Task* task)
{
	scoutManagerTaskSet.addTask(task);
}

void ScoutManager::addTasks(const vector<Task*>& tasks)
{
	scoutManagerTaskSet.addTasks(tasks);
}

void ScoutManager::clearTasks(const int maxClearPriority)
{
	scoutManagerTaskSet.clearTasks(maxClearPriority);
}

const Task* ScoutManager::getCurrentTask() const
{
	return scoutManagerTaskSet.getCurrentTask();
}

void ScoutManager::onFrame()
{
	// first the Scout Manager's tasks...
	scoutManagerTaskSet.update();
	Task* currentTask = scoutManagerTaskSet.getCurrentTask();

	if(currentTask)
		currentTask->execute();

	// clean up tasksets belonging to units which we no longer own
	UnitTracker* unitTracker = UnitTracker::Instance();
	for(auto it = unitTasksMap.begin(); it != unitTasksMap.end(); /**/)
	{
		Unit unit = (*it).first;

		if(unitTracker->getUnitOwner(unit) != this)
		{
			TaskSet taskSet = (*it).second;
			taskSet.update();
			it = unitTasksMap.erase(it);
		}
		else
		{
			++it;
		}
	}

	// Finally the individual Tasks of the Units owned by the ScoutManager
	Unitset scouts = getOwnedUnits();

	for(auto it = scouts.begin(); it != scouts.end(); ++it)
	{
		Unit u = *it;

		if(unitTasksMap.count(u) == 0)
			continue;

		TaskSet& tasks = unitTasksMap[u];
		tasks.update();
		Task* currentTask = tasks.getCurrentTask();

		if(currentTask)
		{
			currentTask->execute();
		}
		else
		{
			Task* newTask = requestNewScoutTask(u);

			if(newTask)
			{
				tasks.addTask(newTask);
			}
		}
	}

	if(scouts.size() == 0 && BaseManager::Instance()->getOpponentBases().size() == 0)	// don't have any scouts and dont know about opposing bases
	{
		if(Broodwar->getFrameCount() > 7200)	// 5 minutes of gameplay @ 24FPS
			this->addTask(new AssignScoutWorker<>());
	}
}

Task* ScoutManager::requestNewScoutTask(const Unit scout)
{
	BaseManager* baseManager = BaseManager::Instance();

	// if we don't know about any opponent base locations yet, scout start positions of the map
	if(baseManager->getOpponentBases().size() == 0)
	{
		const vector<BaseLocation*>& baseLocs = MapAnalyser::Instance()->getBaseLocations();
		int minLastObservedFrame = 5;
		BaseLocation* minLastObservedLoc = nullptr;

		for(auto it = baseLocs.begin(); it != baseLocs.end(); ++it)
		{
			BaseLocation* loc = *it;

			if(!loc->isStartLocation())						// initially only want to scout start locations
				continue;

			if(!MapAnalyser::Instance()->canReach(scout, PositionUtils::toWalkPosition(loc->getOptimalDepotLocation())))
				continue;

			const int lastObserved = loc->getLastObserved();

			if(lastObserved < minLastObservedFrame)
			{
				minLastObservedFrame = lastObserved;
				minLastObservedLoc = loc;
			}
		}

		if(minLastObservedLoc)
		{
			return new ScoutBaseLocation<TaskSchedulePreds::NoEnemyBaseFound, TaskSchedulePreds::NoEnemyBaseFound>(scout, minLastObservedLoc);
		}
		else		// search again, but this time search in ALL base locations, since we've already seen all of the starting locations at least once
		{
			minLastObservedFrame = MathConstants::MAX_INT;

			for(auto it = baseLocs.begin(); it != baseLocs.end(); ++it)
			{
				BaseLocation* loc = *it;

				if(!MapAnalyser::Instance()->canReach(scout, PositionUtils::toWalkPosition(loc->getOptimalDepotLocation())))
					continue;

				const int lastObserved = loc->getLastObserved();

				if(lastObserved < minLastObservedFrame)
				{
					minLastObservedFrame = lastObserved;
					minLastObservedLoc = loc;
				}
			}

			if(minLastObservedLoc)
			{
				return new ScoutBaseLocation<>(scout, minLastObservedLoc);
			}
			else
			{
				LOG_WARNING("Scout Manager doesn't know about any bases but can't find any to scout either")
				return nullptr;
			}
		}
	}
	else	// already know about an enemy base, so should scout base locations around that for expansions
	{
		vector<BaseLocation*> scoutLocs = BaseManager::Instance()->getScoutLocations();
		int minLastObservedFrame = MathConstants::MAX_INT;
		BaseLocation* minLastObservedLoc = nullptr;

		for(auto it = scoutLocs.begin(); it != scoutLocs.end(); ++it)
		{
			BaseLocation* loc = *it;

			if(loc->isCurrentScoutTarget())					// some other scout going for this location already
				continue;

			if(!MapAnalyser::Instance()->canReach(scout, PositionUtils::toWalkPosition(loc->getOptimalDepotLocation())))
				continue;

			const int lastObserved = loc->getLastObserved();

			if(lastObserved < minLastObservedFrame)
			{
				minLastObservedFrame = lastObserved;
				minLastObservedLoc = loc;
			}
		}

		if(minLastObservedLoc)
		{
			return new ScoutBaseLocation<>(scout, minLastObservedLoc);
		}
		else
		{
			LOG_WARNING("No expansions to scout...")
			return nullptr;
		}
	}
}

void ScoutManager::receiveOwnership(const Unit unit)
{
	if(!unit)
		return;

	UnitOwner::receiveOwnership(unit);
	unitTasksMap[unit];
}