/*	-----------------------------------------------------------------------------
	M A A S C R A F T

	StarCraft: Brood War - Bot

	Author: Dennis Soemers
	Maastricht University
	-----------------------------------------------------------------------------
*/

#pragma once

/*
	Interface to the Strategy Manager Singleton.
*/

#include "CommonIncludes.h"

#include <unordered_map>
#include <unordered_set>

#include "Base.h"
#include "BaseLocation.h"
#include "Task.h"
#include "TaskSet.h"
#include "Tuples.h"

struct Reservation
{
	Reservation(int productionID) :
		productionID(productionID), reservedAmount(0) {}

	Reservation(int productionID, int reservedAmount) :
		productionID(productionID), reservedAmount(reservedAmount) {}

	int productionID;
	int reservedAmount;
};

struct HashFunctionReservation
{
	size_t operator() (const Reservation& reservation) const
	{
		return std::hash<int>()(reservation.productionID);
	}
};

struct EqualityFunctionReservations
{
	const bool operator() (const Reservation& r1, const Reservation& r2) const
	{
		return (r1.productionID == r2.productionID);
	}
};

class ProductionManager
{
public:
	static ProductionManager* Instance()
	{
		static ProductionManager instance;
		static ProductionManager* instance_ptr = &instance;
		return instance_ptr;
	}

	const int availableGas() const;
	const int availableMinerals() const;

	const bool placeGasReservation(const Reservation& reservation);
	const bool placeMineralReservation(const Reservation& reservation);
	void removeGasReservation(const int productionID);
	void removeMineralReservation(const int productionID);

	const bool canBuyUpgrade(const BWAPI::UpgradeType upgradeType) const;
	const bool canConstructBuilding(const BWAPI::UnitType buildingType) const;
	const bool canSoonConstructBuilding(const BWAPI::UnitType buildingType) const;
	const bool canTrainUnit(const BWAPI::UnitType unitType) const;

	BWAPI::Unit getIdleNexus(const BaseLocation* preferredBase = nullptr) const;
	BWAPI::Unit getIdleGateway(const BaseLocation* preferredBase = nullptr) const;
	BWAPI::Unit getIdleRoboticsFacility(const BaseLocation* preferredBase = nullptr) const;
	BWAPI::Unit getIdleStargate(const BaseLocation* preferredBase = nullptr) const;
	BWAPI::Unit getIdleTrainer(const BWAPI::UnitType unitType, const BaseLocation* preferredBase = nullptr) const;
	BWAPI::Unit getIdleUpgrader(const BWAPI::UpgradeType upgradeType) const;

	const int getNextProductionID();

	const int getExpectedFrameCountTillMinerals(const int requiredMinerals) const;
	const int getExpectedFrameCountTillGas(const int requiredGas) const;
	const int getNumUnitsProducing(const BWAPI::UnitType unitType);

	// convenience methods to affect the production manager's tasks
	void addTask(Task* task);
	void addTasks(const std::vector<Task*>& tasks);
	void clearTasks(const int maxClearPriority);

	const Task* getCurrentTask() const;

	// EVENTS
	void onBuildingComplete(const BWAPI::Unit building);
	void onBuildingDestroy(const BWAPI::Unit building);
	void onFinishProduction(const BWAPI::UnitType unitType);
	void onFrame();
	void onStartProduction(const BWAPI::UnitType unitType);

private:
	TaskSet productionManagerTaskSet;
	std::unordered_set<Reservation, HashFunctionReservation, EqualityFunctionReservations> gasReservations;
	std::unordered_set<Reservation, HashFunctionReservation, EqualityFunctionReservations> mineralReservations;

	BWAPI::Unitset nexuses;
	BWAPI::Unitset gateways;
	BWAPI::Unitset roboticsFacilities;
	BWAPI::Unitset stargates;

	int nextProductionID;
	int reservedGas;
	int reservedMinerals;

	// counts of units currently in production
	int producingProbes;
	int producingZealots;
	int producingDragoons;
	int producingHighTemplars;
	int producingDarkTemplars;
	int producingReavers;
	int producingObservers;
	int producingShuttles;
	int producingScouts;
	int producingCarriers;
	int producingArbiters;
	int producingCorsairs;

	// keep track of average rate of resource gain
	int previousMineralGain;
	int previousGasGain;
	float averageMineralGain;
	float averageGasGain;

	ProductionManager() :
		productionManagerTaskSet(), gasReservations(), mineralReservations(), 
		nexuses(), gateways(), roboticsFacilities(), stargates(), 
		nextProductionID(-1), reservedGas(0), reservedMinerals(0),
		producingProbes(BWAPI::Broodwar->self()->completedUnitCount(BWAPI::UnitTypes::Protoss_Probe)),
		producingZealots(BWAPI::Broodwar->self()->completedUnitCount(BWAPI::UnitTypes::Protoss_Zealot)), 
		producingDragoons(BWAPI::Broodwar->self()->completedUnitCount(BWAPI::UnitTypes::Protoss_Dragoon)), 
		producingHighTemplars(BWAPI::Broodwar->self()->completedUnitCount(BWAPI::UnitTypes::Protoss_High_Templar)), 
		producingDarkTemplars(BWAPI::Broodwar->self()->completedUnitCount(BWAPI::UnitTypes::Protoss_Dark_Templar)), 
		producingReavers(BWAPI::Broodwar->self()->completedUnitCount(BWAPI::UnitTypes::Protoss_Reaver)), 
		producingObservers(BWAPI::Broodwar->self()->completedUnitCount(BWAPI::UnitTypes::Protoss_Observer)), 
		producingShuttles(BWAPI::Broodwar->self()->completedUnitCount(BWAPI::UnitTypes::Protoss_Shuttle)), 
		producingScouts(BWAPI::Broodwar->self()->completedUnitCount(BWAPI::UnitTypes::Protoss_Scout)), 
		producingCarriers(BWAPI::Broodwar->self()->completedUnitCount(BWAPI::UnitTypes::Protoss_Carrier)), 
		producingArbiters(BWAPI::Broodwar->self()->completedUnitCount(BWAPI::UnitTypes::Protoss_Arbiter)), 
		producingCorsairs(BWAPI::Broodwar->self()->completedUnitCount(BWAPI::UnitTypes::Protoss_Corsair)),
		previousMineralGain(0), previousGasGain(0), averageMineralGain(0.0f), averageGasGain(0.0f) {}

	~ProductionManager() {}

	ProductionManager(ProductionManager const &);
};