/*	-----------------------------------------------------------------------------
	M A A S C R A F T

	StarCraft: Brood War - Bot

	Author: Dennis Soemers
	Maastricht University
	-----------------------------------------------------------------------------
*/

#include "CommonIncludes.h"

#include "BuildingManager.h"
#include "Distances.h"
#include "MathConstants.h"
#include "ProductionManager.h"
#include "UnitTracker.h"

/*
	Implementation of the ProductionManager
*/

#pragma warning( push )
#pragma warning( disable: 4244 )

using namespace BWAPI;
using namespace std;

const int ProductionManager::availableGas() const
{
	return Broodwar->self()->gas() - reservedGas;
}

const int ProductionManager::availableMinerals() const
{
	return Broodwar->self()->minerals() - reservedMinerals;
}

const bool ProductionManager::placeGasReservation(const Reservation& reservation)
{
#ifdef MAASCRAFT_DEBUG
	if(gasReservations.count(reservation) > 0)
		LOG_WARNING("ProductionManager: Attempted to place a gas reservation with already existing production ID!")

	if(reservation.reservedAmount > availableGas())
	{
		LOG_WARNING("ProductionManager: Attempted to reserve more gas than available!")
		return false;
	}
#endif

	gasReservations.insert(reservation);
	reservedGas += reservation.reservedAmount;
	return true;
}

const bool ProductionManager::placeMineralReservation(const Reservation& reservation)
{
#ifdef MAASCRAFT_DEBUG
	if(mineralReservations.count(reservation) > 0)
		LOG_WARNING("ProductionManager: Attempted to place a minerals reservation with already existing production ID!")

	if(reservation.reservedAmount > availableMinerals())
	{
		LOG_WARNING("ProductionManager: Attempted to reserve more minerals than available!")
		LOG_WARNING(StringBuilder() << "Available minerals = " << this->availableMinerals())
		LOG_WARNING(StringBuilder() << "Attempted reservation = " << reservation.reservedAmount)
		return false;
	}
#endif

	mineralReservations.insert(reservation);
	reservedMinerals += reservation.reservedAmount;
	return true;
}

void ProductionManager::removeGasReservation(const int productionID)
{
	auto it = gasReservations.find(Reservation(productionID));
#ifdef MAASCRAFT_DEBUG
	if(it == gasReservations.end())
	{
		LOG_WARNING("ProductionManager: Attempted to remove a non-existing gas reservation!")
		return;
	}
#endif

	reservedGas -= it->reservedAmount;
	gasReservations.erase(it);
}

void ProductionManager::removeMineralReservation(const int productionID)
{
	auto it = mineralReservations.find(Reservation(productionID));
#ifdef MAASCRAFT_DEBUG
	if(it == mineralReservations.end())
	{
		LOG_WARNING("ProductionManager: Attempted to remove a non-existing mineral reservation!")
		return;
	}
#endif

	reservedMinerals -= it->reservedAmount;
	mineralReservations.erase(it);
}

const bool ProductionManager::canBuyUpgrade(const UpgradeType upgradeType) const
{
	Player self = BWAPI::Broodwar->self();

	int currentLevel = self->getUpgradeLevel(upgradeType);
	int nextLevel = currentLevel + 1;

	if(nextLevel > upgradeType.maxRepeats())
	{
		return false;
	}

	if(self->isUpgrading(upgradeType))
	{
		return false;
	}

	UnitType requirement = upgradeType.whatsRequired(nextLevel);

	if(requirement != UnitTypes::None && self->completedUnitCount(requirement) <= 0)
	{
		return false;
	}

	int numUpgrades = self->getUpgradeLevel(upgradeType);

	if(availableGas() < upgradeType.gasPrice() + numUpgrades * upgradeType.gasPriceFactor())
	{
		return false;
	}

	if(availableMinerals() < upgradeType.mineralPrice() + numUpgrades * upgradeType.mineralPriceFactor())
	{
		return false;
	}

	UnitType upgrader = upgradeType.whatUpgrades();

	return (getIdleUpgrader(upgradeType) != nullptr);
}

const bool ProductionManager::canConstructBuilding(const UnitType buildingType) const
{
	Player self = BWAPI::Broodwar->self();

	const Unitset& selfBuildings = UnitTracker::Instance()->getSelfBuildings();
	const map<UnitType, int>& requirements = buildingType.requiredUnits();

	for(auto it_req = requirements.begin(); it_req != requirements.end(); ++it_req)
	{
		UnitType requirement = (*it_req).first;
		if(!requirement.isBuilding())			// dont care about requiring probes
		{
#ifdef MAASCRAFT_DEBUG
			if(requirement != UnitTypes::Protoss_Probe)
				LOG_WARNING(StringBuilder() << buildingType << " has non-building requirement which is not probe; " << requirement)
#endif
			continue;
		}

		bool found = false;

		for(auto it_building = selfBuildings.begin(); it_building != selfBuildings.end(); ++it_building)
		{
			Unit building = *it_building;
			if(building->getType() == requirement)
			{
				found = true;
				break;
			}
		}

		if(!found)
			return false;
	}

	return (availableGas() >= buildingType.gasPrice()			&&
			availableMinerals() >= buildingType.mineralPrice()		);
}

const bool ProductionManager::canSoonConstructBuilding(const BWAPI::UnitType buildingType) const
{
	const int soon = Options::ProductionManager::MAX_FRAME_BLOCK_PRODUCTION;

	Player self = BWAPI::Broodwar->self();

	const Unitset& selfBuildings = UnitTracker::Instance()->getSelfBuildings();
	const map<UnitType, int>& requirements = buildingType.requiredUnits();

	for(auto it_req = requirements.begin(); it_req != requirements.end(); ++it_req)
	{
		UnitType requirement = (*it_req).first;
		if(!requirement.isBuilding())			// dont care about requiring probes
		{
#ifdef MAASCRAFT_DEBUG
			if(requirement != UnitTypes::Protoss_Probe)
				LOG_WARNING(StringBuilder() << buildingType << " has non-building requirement which is not probe; " << requirement)
#endif
			continue;
		}

		bool found = false;

		for(auto it_building = selfBuildings.begin(); it_building != selfBuildings.end(); ++it_building)
		{
			Unit building = *it_building;
			if(building->getType() == requirement)
			{
				found = true;
				break;
			}
		}

		if(!found)
			return false;
	}

	return (availableGas() + soon*averageGasGain >= buildingType.gasPrice()					&&
			availableMinerals() + soon*averageMineralGain >= buildingType.mineralPrice()		);
}

const bool ProductionManager::canTrainUnit(const UnitType unitType) const
{
	Player self = BWAPI::Broodwar->self();

	const Unitset& selfBuildings = UnitTracker::Instance()->getSelfBuildings();
	const map<UnitType, int>& requirements = unitType.requiredUnits();

	for(auto it_req = requirements.begin(); it_req != requirements.end(); ++it_req)
	{
		bool found = false;
		UnitType requirement = (*it_req).first;

		for(auto it_building = selfBuildings.begin(); it_building != selfBuildings.end(); ++it_building)
		{
			Unit building = *it_building;
			if(building->getType() == requirement)
			{
				found = true;
				break;
			}
		}

		if(!found)
			return false;
	}

	Unit trainer = nullptr;
	UnitType trainerType = unitType.whatBuilds().first;

	if(trainerType == UnitTypes::Protoss_Nexus)
		trainer = getIdleNexus();
	else if(trainerType == UnitTypes::Protoss_Gateway)
		trainer = getIdleGateway();
	else if(trainerType == UnitTypes::Protoss_Robotics_Facility)
		trainer = getIdleRoboticsFacility();
	else if(trainerType == UnitTypes::Protoss_Stargate)
		trainer = getIdleStargate();

	return (trainer																	&&
			availableGas() >= unitType.gasPrice()									&&
			availableMinerals() >= unitType.mineralPrice()							&&
			self->supplyTotal() >= self->supplyUsed() + unitType.supplyRequired()		);
}

Unit ProductionManager::getIdleNexus(const BaseLocation* preferredBase) const
{
	if(preferredBase)		// need to keep track of distances since there's a preference
	{
		TilePosition baseLoc = preferredBase->getOptimalDepotLocation();

		Unit bestNexus = nullptr;
		int minDistSq = MathConstants::MAX_INT;

		for(auto it = nexuses.begin(); it != nexuses.end(); ++it)
		{
			Unit nexus = *it;

			if(!nexus->isTraining())
			{
				const int distSq = Distances::getSquaredDistance(baseLoc, nexus->getTilePosition());

				if(distSq < minDistSq)
				{
					minDistSq = distSq;
					bestNexus = nexus;
				}
			}
		}
	}
	else
	{
		for(auto it = nexuses.begin(); it != nexuses.end(); ++it)
		{
			Unit nexus = *it;

			if(!nexus->isTraining())
				return nexus;
		}
	}

	return nullptr;
}

Unit ProductionManager::getIdleGateway(const BaseLocation* preferredBase) const
{
	if(preferredBase)		// need to keep track of distances since there's a preference
	{
		TilePosition baseLoc = preferredBase->getOptimalDepotLocation();

		Unit bestGateway = nullptr;
		int minDistSq = MathConstants::MAX_INT;

		for(auto it = gateways.begin(); it != gateways.end(); ++it)
		{
			Unit gateway = *it;

			if(!gateway->isTraining())
			{
				const int distSq = Distances::getSquaredDistance(baseLoc, gateway->getTilePosition());

				if(distSq < minDistSq)
				{
					minDistSq = distSq;
					bestGateway = gateway;
				}
			}
		}
	}
	else
	{
		for(auto it = gateways.begin(); it != gateways.end(); ++it)
		{
			Unit gateway = *it;

			if(!gateway->isTraining())
				return gateway;
		}
	}

	return nullptr;
}

Unit ProductionManager::getIdleRoboticsFacility(const BaseLocation* preferredBase) const
{
	if(preferredBase)		// need to keep track of distances since there's a preference
	{
		TilePosition baseLoc = preferredBase->getOptimalDepotLocation();

		Unit bestFacility = nullptr;
		int minDistSq = MathConstants::MAX_INT;

		for(auto it = roboticsFacilities.begin(); it != roboticsFacilities.end(); ++it)
		{
			Unit facility = *it;

			if(!facility->isTraining())
			{
				const int distSq = Distances::getSquaredDistance(baseLoc, facility->getTilePosition());

				if(distSq < minDistSq)
				{
					minDistSq = distSq;
					bestFacility = facility;
				}
			}
		}
	}
	else
	{
		for(auto it = roboticsFacilities.begin(); it != roboticsFacilities.end(); ++it)
		{
			Unit facility = *it;

			if(!facility->isTraining())
				return facility;
		}
	}

	return nullptr;
}

Unit ProductionManager::getIdleStargate(const BaseLocation* preferredBase) const
{
	if(preferredBase)		// need to keep track of distances since there's a preference
	{
		TilePosition baseLoc = preferredBase->getOptimalDepotLocation();

		Unit bestStargate = nullptr;
		int minDistSq = MathConstants::MAX_INT;

		for(auto it = stargates.begin(); it != stargates.end(); ++it)
		{
			Unit stargate = *it;

			if(!stargate->isTraining())
			{
				const int distSq = Distances::getSquaredDistance(baseLoc, stargate->getTilePosition());

				if(distSq < minDistSq)
				{
					minDistSq = distSq;
					bestStargate = stargate;
				}
			}
		}
	}
	else
	{
		for(auto it = stargates.begin(); it != stargates.end(); ++it)
		{
			Unit stargate = *it;

			if(!stargate->isTraining())
				return stargate;
		}
	}

	return nullptr;
}

Unit ProductionManager::getIdleTrainer(const UnitType unitType, const BaseLocation* preferredBase) const
{
	UnitType trainerType = unitType.whatBuilds().first;

	if(trainerType == UnitTypes::Protoss_Nexus)
		return getIdleNexus();
	else if(trainerType == UnitTypes::Protoss_Gateway)
		return getIdleGateway();
	else if(trainerType == UnitTypes::Protoss_Robotics_Facility)
		return getIdleRoboticsFacility();
	else if(trainerType == UnitTypes::Protoss_Stargate)
		return getIdleStargate();

#ifdef MAASCRAFT_DEBUG
	LOG_WARNING("ProductionManager::getIdleTrainer() called with unknown trainerType!")
#endif
	return nullptr;
}

Unit ProductionManager::getIdleUpgrader(const UpgradeType upgradeType) const
{
	UnitType type = upgradeType.whatUpgrades();

	const Unitset& selfBuildings = UnitTracker::Instance()->getSelfBuildings();
	for(auto it = selfBuildings.begin(); it != selfBuildings.end(); ++it)
	{
		Unit building = *it;

		if(building->getType() == type)
		{
			if(!building->isUpgrading())
				return building;
		}
	}

	return nullptr;
}

const int ProductionManager::getNextProductionID()
{
	return nextProductionID++;
}

const int ProductionManager::getExpectedFrameCountTillMinerals(const int requiredMinerals) const
{
	return ((requiredMinerals - availableMinerals()) / averageMineralGain);
}

const int ProductionManager::getExpectedFrameCountTillGas(const int requiredGas) const
{
	return ((requiredGas - availableGas()) / averageGasGain);
}

const int ProductionManager::getNumUnitsProducing(const BWAPI::UnitType unitType)
{
	if(unitType == UnitTypes::Protoss_Probe)
		return producingProbes;
	else if(unitType == UnitTypes::Protoss_Zealot)
		return producingZealots;
	else if(unitType == UnitTypes::Protoss_Dragoon)
		return producingDragoons;
	else if(unitType == UnitTypes::Protoss_High_Templar)
		return producingHighTemplars;
	else if(unitType == UnitTypes::Protoss_Dark_Templar)
		return producingDarkTemplars;
	else if(unitType == UnitTypes::Protoss_Reaver)
		return producingReavers;
	else if(unitType == UnitTypes::Protoss_Observer)
		return producingObservers;
	else if(unitType == UnitTypes::Protoss_Shuttle)
		return producingShuttles;
	else if(unitType == UnitTypes::Protoss_Scout)
		return producingScouts;
	else if(unitType == UnitTypes::Protoss_Carrier)
		return producingCarriers;
	else if(unitType == UnitTypes::Protoss_Arbiter)
		return producingArbiters;
	else if(unitType == UnitTypes::Protoss_Corsair)
		return producingCorsairs;

	LOG_WARNING(StringBuilder() << "ProductionManager: Requested number of units producing of unknown unit type: " << unitType)
	return 0;
}

void ProductionManager::addTask(Task* task)
{
	productionManagerTaskSet.addTask(task);
}

void ProductionManager::addTasks(const vector<Task*>& tasks)
{
	productionManagerTaskSet.addTasks(tasks);
}

void ProductionManager::clearTasks(const int maxClearPriority)
{
	productionManagerTaskSet.clearTasks(maxClearPriority);
}

const Task* ProductionManager::getCurrentTask() const
{
	return productionManagerTaskSet.getCurrentTask();
}

void ProductionManager::onBuildingComplete(const Unit building)
{
	UnitType type = building->getType();

	if(type == UnitTypes::Protoss_Nexus)
		nexuses.push_back(building);
	else if(type == UnitTypes::Protoss_Gateway)
		gateways.push_back(building);
	else if(type == UnitTypes::Protoss_Robotics_Facility)
		roboticsFacilities.push_back(building);
	else if(type == UnitTypes::Protoss_Stargate)
		stargates.push_back(building);
}

void ProductionManager::onBuildingDestroy(const Unit building)
{
	UnitType type = building->getType();

	if(type == UnitTypes::Protoss_Nexus)
		nexuses.erase(building);
	else if(type == UnitTypes::Protoss_Gateway)
		gateways.erase(building);
	else if(type == UnitTypes::Protoss_Robotics_Facility)
		roboticsFacilities.erase(building);
	else if(type == UnitTypes::Protoss_Stargate)
		stargates.erase(building);
}

void ProductionManager::onFinishProduction(const UnitType unitType)
{
	if(unitType == UnitTypes::Protoss_Probe)
		producingProbes--;
	else if(unitType == UnitTypes::Protoss_Zealot)
		producingZealots--;
	else if(unitType == UnitTypes::Protoss_Dragoon)
		producingDragoons--;
	else if(unitType == UnitTypes::Protoss_High_Templar)
		producingHighTemplars--;
	else if(unitType == UnitTypes::Protoss_Dark_Templar)
		producingDarkTemplars--;
	else if(unitType == UnitTypes::Protoss_Reaver)
		producingReavers--;
	else if(unitType == UnitTypes::Protoss_Observer)
		producingObservers--;
	else if(unitType == UnitTypes::Protoss_Shuttle)
		producingShuttles--;
	else if(unitType == UnitTypes::Protoss_Scout)
		producingScouts--;
	else if(unitType == UnitTypes::Protoss_Carrier)
		producingCarriers--;
	else if(unitType == UnitTypes::Protoss_Arbiter)
		producingArbiters--;
	else if(unitType == UnitTypes::Protoss_Corsair)
		producingCorsairs--;
}

void ProductionManager::onFrame()
{
	productionManagerTaskSet.update();
	Task* currentTask = productionManagerTaskSet.getCurrentTask();

	if(currentTask)
		currentTask->execute();

	// re-compute average resource gains
	if((Broodwar->getFrameCount() % Options::ProductionManager::UPDATE_AVG_RESOURCE_GAIN_FRAME_COUNT) == 0)
	{
		int gatheredMinerals = Broodwar->self()->gatheredMinerals();
		int gatheredGas = Broodwar->self()->gatheredGas();

		float mineralDifference = gatheredMinerals - previousMineralGain;
		float gasDifference = gatheredGas - previousGasGain;

		previousMineralGain = gatheredMinerals;
		previousGasGain = gatheredGas;

		averageMineralGain = mineralDifference / Options::ProductionManager::UPDATE_AVG_RESOURCE_GAIN_FRAME_COUNT;
		averageGasGain = gasDifference / Options::ProductionManager::UPDATE_AVG_RESOURCE_GAIN_FRAME_COUNT;
	}

#ifdef MAASCRAFT_DEBUG
	if(Options::Debug::ScreenInfo::DEBUG_PRODUCTION_MANAGER_INFO)
	{
		Broodwar->drawTextScreen(350, 20, "PRODUCTION MANAGER");
		if(currentTask)
			Broodwar->drawTextScreen(350, 40, "Current Task: %s", currentTask->getDescription());
		else
			Broodwar->drawTextScreen(350, 40, "Current Task: None");

		Broodwar->drawTextScreen(350, 60, "Number of Pylons building: %d", BuildingManager::Instance()->getNumPylonsBuilding());
		/*
		Broodwar->drawTextScreen(350, 60, "Unfinished tasks in queue: %d", productionManagerTaskSet.size());
		
		Broodwar->drawTextScreen(350, 80, "Average Mineral Gain = %f", averageMineralGain);
		Broodwar->drawTextScreen(350, 100, "Average Gas Gain = %f", averageGasGain);
		
		Broodwar->drawTextScreen(350, 300, "Producing Probes: %d", producingProbes);
		Broodwar->drawTextScreen(350, 320, "Producing Zealots: %d", producingZealots);
		Broodwar->drawTextScreen(350, 340, "Producing Dragoons: %d", producingDragoons);
		Broodwar->drawTextScreen(350, 120, "Producing High Templars: %d", producingHighTemplars);
		Broodwar->drawTextScreen(350, 140, "Producing Dark Templars: %d", producingDarkTemplars);
		Broodwar->drawTextScreen(350, 160, "Producing Reavers: %d", producingReavers);
		Broodwar->drawTextScreen(350, 180, "Producing Observers: %d", producingObservers);
		Broodwar->drawTextScreen(350, 200, "Producing Shuttles: %d", producingShuttles);
		Broodwar->drawTextScreen(350, 220, "Producing Scouts: %d", producingScouts);
		Broodwar->drawTextScreen(350, 240, "Producing Carriers: %d", producingCarriers);
		Broodwar->drawTextScreen(350, 260, "Producing Arbiters: %d", producingArbiters);
		Broodwar->drawTextScreen(350, 280, "Producing Corsairs: %d", producingCorsairs);
		*/
	}
#endif
}

void ProductionManager::onStartProduction(const UnitType unitType)
{
	if(unitType == UnitTypes::Protoss_Probe)
		producingProbes++;
	else if(unitType == UnitTypes::Protoss_Zealot)
		producingZealots++;
	else if(unitType == UnitTypes::Protoss_Dragoon)
		producingDragoons++;
	else if(unitType == UnitTypes::Protoss_High_Templar)
		producingHighTemplars++;
	else if(unitType == UnitTypes::Protoss_Dark_Templar)
		producingDarkTemplars++;
	else if(unitType == UnitTypes::Protoss_Reaver)
		producingReavers++;
	else if(unitType == UnitTypes::Protoss_Observer)
		producingObservers++;
	else if(unitType == UnitTypes::Protoss_Shuttle)
		producingShuttles++;
	else if(unitType == UnitTypes::Protoss_Scout)
		producingScouts++;
	else if(unitType == UnitTypes::Protoss_Carrier)
		producingCarriers++;
	else if(unitType == UnitTypes::Protoss_Arbiter)
		producingArbiters++;
	else if(unitType == UnitTypes::Protoss_Corsair)
		producingCorsairs++;
}

#pragma warning( pop )