/*	-----------------------------------------------------------------------------
	M A A S C R A F T

	StarCraft: Brood War - Bot

	Author: Dennis Soemers
	Maastricht University
	-----------------------------------------------------------------------------
*/

#pragma once

/*
	Interface to a container around a BWAPI::WalkPosition to represent states in the A* algorithm
*/

#include "CommonIncludes.h"

#include "AStar.hpp"
#include "Distances.h"
#include "MathConstants.h"

#pragma warning( push )
#pragma warning( disable: 4244 )

class PositionState
{
private:
	BWAPI::WalkPosition walkPosition;

public:
	PositionState();
	PositionState(const int x, const int y);
	PositionState(const BWAPI::WalkPosition position);

	inline BWAPI::WalkPosition getWalkPosition() const	{	return walkPosition;	}

	inline const int x() const	{	return walkPosition.x;		}
	inline const int y() const	{	return walkPosition.y;		}

	inline const bool operator==(const PositionState& other) const {	return (x() == other.x() && y() == other.y());		}

	~PositionState();
};

// This namespace contains implementations of functions which A* can use when using PositionStates
namespace PositionStateFunctions
{
	//===============================================
	//				Get_Cost Functions
	//===============================================

	// Euclidean distance
	//	Because A* guarantees the function will only be called on neighbouring states, 
	//	can simply use costs of 1 and SQRT(2) and avoid more expensive computations
	struct Get_Cost_Euclidean
	{
		const float operator() (const PositionState& from, const PositionState& to) const
		{
			if((from.x() == to.x()) || (from.y() == to.y()))
				return 1.0f;
			else
				return MathConstants::SQRT_2F;
		}
	};

	//===============================================
	//				Get_Successors Functions
	//===============================================

	// This function adds all walkable neighbouring tiles as successors
	template <typename Get_Cost,
				typename Goal_Distance_Estimate,
				typename F_function,
				typename Is_Same_State = Predicates::Equality<PositionState> >
	struct Get_Successors_Walkable
	{
		void operator() (const PositionState& node, const PositionState* parent, 
							AStar<PositionState, 
									Get_Cost, 
									Get_Successors_Walkable,
									Goal_Distance_Estimate, 
									F_function, 
									Is_Same_State					>* const aStarSearch) const
		{
			MapAnalyser* mapAnalyser = MapAnalyser::Instance();
			int parentX = -1;
			int parentY = -1;

			// if parent node is specified, dont want to add him to the successors again
			if(parent)
			{
				parentX = parent->x();
				parentY = parent->y();
			}

			BWAPI::WalkPosition parentPos(parentX, parentY);

			int nodeX = node.x();
			int nodeY = node.y();

			int dx = -1;
			int dy = -1;
			BWAPI::WalkPosition pos(nodeX + dx, nodeY + dy);
			if(pos.isValid() && pos != parentPos && mapAnalyser->isWalkable(pos))
				aStarSearch->addSuccessor(PositionState(pos));

			dy = 0;
			pos = BWAPI::WalkPosition(nodeX + dx, nodeY + dy);
			if(pos.isValid() && pos != parentPos && mapAnalyser->isWalkable(pos))
				aStarSearch->addSuccessor(PositionState(pos));

			dy = 1;
			pos = BWAPI::WalkPosition(nodeX + dx, nodeY + dy);
			if(pos.isValid() && pos != parentPos && mapAnalyser->isWalkable(pos))
				aStarSearch->addSuccessor(PositionState(pos));

			dx = 0;
			dy = -1;
			pos = BWAPI::WalkPosition(nodeX + dx, nodeY + dy);
			if(pos.isValid() && pos != parentPos && mapAnalyser->isWalkable(pos))
				aStarSearch->addSuccessor(PositionState(pos));

			dy = 1;
			pos = BWAPI::WalkPosition(nodeX + dx, nodeY + dy);
			if(pos.isValid() && pos != parentPos && mapAnalyser->isWalkable(pos))
				aStarSearch->addSuccessor(PositionState(pos));

			dx = 1;
			dy = -1;
			pos = BWAPI::WalkPosition(nodeX + dx, nodeY + dy);
			if(pos.isValid() && pos != parentPos && mapAnalyser->isWalkable(pos))
				aStarSearch->addSuccessor(PositionState(pos));

			dy = 0;
			pos = BWAPI::WalkPosition(nodeX + dx, nodeY + dy);
			if(pos.isValid() && pos != parentPos && mapAnalyser->isWalkable(pos))
				aStarSearch->addSuccessor(PositionState(pos));

			dy = 1;
			pos = BWAPI::WalkPosition(nodeX + dx, nodeY + dy);
			if(pos.isValid() && pos != parentPos && mapAnalyser->isWalkable(pos))
				aStarSearch->addSuccessor(PositionState(pos));
		}
	};

	// This function adds all neighbouring tiles (walkable AND non-walkable) as successors
	template <typename Get_Cost,
				typename Goal_Distance_Estimate,
				typename F_function,
				typename Is_Same_State = Predicates::Equality<PositionState> >
	struct Get_Successors_Flyer
	{
		void operator() (const PositionState& node, const PositionState* parent, 
							AStar<PositionState, 
									Get_Cost, Get_Successors_Flyer,
									Goal_Distance_Estimate, 
									F_function, 
									Is_Same_State					>* const aStarSearch) const
		{
			int parentX = -1;
			int parentY = -1;

			// if parent node is specified, dont want to add him to the successors again
			if(parent)
			{
				parentX = parent->x();
				parentY = parent->y();
			}

			BWAPI::WalkPosition parentPos(parentX, parentY);

			int nodeX = node.x();
			int nodeY = node.y();

			int dx = -1;
			int dy = -1;
			BWAPI::WalkPosition pos(nodeX + dx, nodeY + dy);
			if(pos.isValid() && pos != parentPos)
				aStarSearch->addSuccessor(PositionState(pos));

			dx = -1;
			dy = 0;
			pos = BWAPI::WalkPosition(nodeX + dx, nodeY + dy);
			if(pos.isValid() && pos != parentPos)
				aStarSearch->addSuccessor(PositionState(pos));

			dx = -1;
			dy = 1;
			pos = BWAPI::WalkPosition(nodeX + dx, nodeY + dy);
			if(pos.isValid() && pos != parentPos)
				aStarSearch->addSuccessor(PositionState(pos));

			dx = 0;
			dy = -1;
			pos = BWAPI::WalkPosition(nodeX + dx, nodeY + dy);
			if(pos.isValid() && pos != parentPos)
				aStarSearch->addSuccessor(PositionState(pos));

			dx = 0;
			dy = 1;
			pos = BWAPI::WalkPosition(nodeX + dx, nodeY + dy);
			if(pos.isValid() && pos != parentPos)
				aStarSearch->addSuccessor(PositionState(pos));

			dx = 1;
			dy = -1;
			pos = BWAPI::WalkPosition(nodeX + dx, nodeY + dy);
			if(pos.isValid() && pos != parentPos)
				aStarSearch->addSuccessor(PositionState(pos));

			dx = 1;
			dy = 0;
			pos = BWAPI::WalkPosition(nodeX + dx, nodeY + dy);
			if(pos.isValid() && pos != parentPos)
				aStarSearch->addSuccessor(PositionState(pos));

			dx = 1;
			dy = 1;
			pos = BWAPI::WalkPosition(nodeX + dx, nodeY + dy);
			if(pos.isValid() && pos != parentPos)
				aStarSearch->addSuccessor(PositionState(pos));
		}
	};

	//===============================================
	//			Goal_Distance_Estimate Functions
	//===============================================

	// Euclidean distance
	struct Goal_Distance_Estimate_Euclid_Dist
	{
		const float operator() (const PositionState& self, const PositionState& goal) const
		{
			return Distances::getDistanceF(self.x(), self.y(), goal.x(), goal.y());
		}
	};

	// Squared Euclidean distance
	struct Goal_Distance_Estimate_Squared_Euclid_Dist
	{
		const float operator() (const PositionState& self, const PositionState& goal) const
		{
			return Distances::getSquaredDistanceF(self.x(), self.y(), goal.x(), goal.y());
		}
	};
}

#pragma warning( pop )