/*	-----------------------------------------------------------------------------
	M A A S C R A F T

	StarCraft: Brood War - Bot

	Author: Dennis Soemers
	Maastricht University
	-----------------------------------------------------------------------------
*/

#pragma once

#include <cmath>
#include <iostream>

/*	==========================================================================================
			Declaration of Point2D class, for simple 2-dimensional points/vectors
	========================================================================================== */

class Point2D
{

public:
	// The coordinates
	double x;
	double y;

	// Constructors
	Point2D(const double px = 0, const double py = 0);
	Point2D(const Point2D& p);

	// Deconstructor
	~Point2D();

	// Overloaded operators
	const bool		operator==(const Point2D& p) const;		// test equality
	const bool		operator!=(const Point2D& p) const;		// test inequality

	const Point2D	operator-() const;							// negate point
	Point2D&		operator+=(const Point2D& p);				// addition assignment
	Point2D&		operator-=(const Point2D& p);				// subtraction assignment
	Point2D&		operator*=(const double scalar);			// multiplication assignment
	Point2D&		operator/=(const double scalar);			// division assignment

	const Point2D	operator+(const Point2D& p) const;			// add up 2 points
	const Point2D	operator-(const Point2D& p) const;			// subtract p from point
	const Point2D	operator*(const double scalar) const;		// scale point by scalar
	const Point2D	operator/(const double scalar) const;		// scale point by 1/scalar

	const double	operator*(const Point2D& p) const;			// dot product of 2 points

	// Public methods
	const double	getLength() const;						// returns length of vector from origin to this point
	const double	getSquaredLength() const;				// returns the length of vector from origin to this point squared. This function is strictly faster than getLength()!
	const Point2D	getNormalizedPoint() const;				// returns normalized version of given point
	void			normalize();							// normalizes the given point
};

typedef Point2D Vector2D;		// sometimes it makes more sense to think of a Point2D as a vector, when doing linear algebra stuff

/*	==========================================================================================
			Definition of declarations above

			Implementing all functions in header file since they are all very short functions.
			Allows for better compiler optimizations through inlining
	========================================================================================== */

// Constructor with given coordinates
inline Point2D::Point2D(const double px, const double py) :
	x(px), y(py)
	{}

// Copy constructor
inline Point2D::Point2D(const Point2D& p) :
	x(p.x), y(p.y)
	{}

// Deconstructor
inline Point2D::~Point2D()
{}

/* =====================================================================
		OVERLOADED OPERATORS
   ===================================================================== */

// negate point
inline const Point2D Point2D::operator-() const
{
	return Point2D(-x, -y);
}

// test equality
inline const bool Point2D::operator==(const Point2D& p) const
{
	return ((x == p.x) && (y == p.y));
}

// test inequality
inline const bool Point2D::operator!=(const Point2D& p) const
{
	return ((x != p.x) || (y != p.y));
}

// addition assignment
inline Point2D& Point2D::operator+=(const Point2D& p)
{
	x += p.x;
	y += p.y;

	return (*this);
}

 // subtraction assignment
inline Point2D& Point2D::operator-=(const Point2D& p)
{
	x -= p.x;
	y -= p.y;

	return (*this);
}

 // multiplication assignment
inline Point2D& Point2D::operator*=(const double scalar)
{
	x *= scalar;
	y *= scalar;

	return (*this);
}

// division assignment
inline Point2D& Point2D::operator/=(const double scalar)
{
	double scalar_inv = 1/scalar;

	x *= scalar_inv;
	y *= scalar_inv;

	return (*this);
}

// multiplication
inline const Point2D Point2D::operator*(const double scalar) const
{
	return Point2D(x * scalar, y * scalar);
}

// division
inline const Point2D Point2D::operator/(const double scalar) const
{
	double scalar_inv = 1/scalar;

	return Point2D(x * scalar_inv, y * scalar_inv);
}

// addition
inline const Point2D Point2D::operator+(const Point2D& p) const
{
	return Point2D(x + p.x, y + p.y);
}

// subtraction
inline const Point2D Point2D::operator-(const Point2D& p) const
{
	return Point2D(x - p.x, y - p.y);
}
 
// dot product
inline const double Point2D::operator*(const Point2D& p) const
{
	return (x * p.x + y * p.y);
}

// output stream operator
inline std::ostream & operator<<(std::ostream & os, const Point2D & p)
{
	os << "(" << p.x << ", " << p.y << ")";
}


/* =====================================================================
		PUBLIC METHODS
   ===================================================================== */
inline const double Point2D::getLength() const
{
	return std::sqrt(x*x + y*y);
}
 
inline const double Point2D::getSquaredLength() const
{
	return (x*x + y*y);
}

inline const Point2D Point2D::getNormalizedPoint() const
{
	double scalar = 1 / (this->getLength());
	return Point2D(x * scalar, y * scalar);
}

inline void Point2D::normalize()
{
	double scalar = 1 / (this->getLength());

	x *= scalar;
	y *= scalar;
}