/*	-----------------------------------------------------------------------------
	M A A S C R A F T

	StarCraft: Brood War - Bot

	Author: Dennis Soemers
	Maastricht University
	-----------------------------------------------------------------------------
*/

#include "CommonIncludes.h"

#include "ArrayUtils.h"
#include "BaseManager.h"
#include "Distances.h"
#include "MapAnalyser.h"
#include "OpponentTracker.h"
#include "PositionUtils.h"

#pragma warning( push )
#pragma warning( disable:4244 )

/*
	Implementation of the OpponentManager
*/

using namespace BWAPI;
using namespace std;
using namespace UnitUtils;

OpponentTracker::OpponentTracker() :
	buildingDataMap(), buildingTypeCounts(),
	unitDataMap(), unitTypeCounts(), opponentClusters(),
	opponentUnits(), opponentBuildings(), opponentDefensiveStructures()
{
	airThreatMap = new float[Broodwar->mapWidth() * Broodwar->mapHeight()];
	groundThreatMap = new float[Broodwar->mapWidth() * Broodwar->mapHeight()];
}

OpponentTracker::~OpponentTracker()
{
	delete[] airThreatMap;
	delete[] groundThreatMap;
}

void OpponentTracker::onFrame()
{
	validateOpponentUnits();
	updateBuildingData();
	updateUnitData();
	updateThreatMaps();
	computeClusters();
}

void OpponentTracker::onUnitDestroy(Unit unit)
{
	if(!unit)
		return;

	UnitType type = unit->getType();

	if(type.isBuilding())
	{
		BaseManager::Instance()->onOpponentBuildingDestroy(unit);
		MapAnalyser::Instance()->onBuildingDestroy(unit->getPosition(), type);

		if(buildingDataMap.erase(unit) == 1)
			buildingTypeCounts[type]--;

		opponentDefensiveStructures.erase(unit);
	}
	else
	{
		if(unitDataMap.erase(unit) == 1)
			unitTypeCounts[type]--;
	}
}

// Currently deals with following cases:
//	- Old Building which was placed on Vespene Geyser disappeared and turned back into Vespene Geyser
//	- Building used to be visible in a certain location and no longer is visible
//
// Because it currently only deals with buildings, also only called in the updateBuildingData() function!
const bool OpponentTracker::shouldRemoveUnitData(const UnitData& unitData) const
{
	return (unitData.unitPtr != nullptr && unitData.unitPtr->getType() == UnitTypes::Resource_Vespene_Geyser) || 
			((unitData.unitType.isBuilding() || unitData.unitType.isAddon()) && Broodwar->isVisible(unitData.lastPosition.x / 32, unitData.lastPosition.y / 32) && !unitData.unitPtr->isVisible());
}

const int OpponentTracker::getNumUnits(const BWAPI::UnitType type) const
{
	if(type.isBuilding())
	{
		if(buildingTypeCounts.count(type) == 0)
			return 0;
		else
			return buildingTypeCounts.at(type);
	}
	else
	{
		if(unitTypeCounts.count(type) == 0)
			return 0;
		else
			return unitTypeCounts.at(type);
	}
}

void OpponentTracker::updateBuildingData()
{
	for(auto it = opponentBuildings.begin(); it != opponentBuildings.end(); ++it)
	{
		Unit building = *it;
		auto it_data = buildingDataMap.find(building);

		if(it_data != buildingDataMap.end())		// building's data already in the map
		{
			UnitData& data = it_data->second;
			UnitType currentType = building->getType();

			if(data.unitType != currentType && currentType != UnitTypes::Unknown)	// building changed into some other type
			{
				buildingTypeCounts[data.unitType]--;
				data.unitType = currentType;
				buildingTypeCounts[data.unitType]++;
			}

			data.completed = building->isCompleted();
			data.id = building->getID();
			data.lastHP = building->getHitPoints();
			data.lastPosition = building->getPosition();
			data.lastSeen = Broodwar->getFrameCount();
			data.unitPtr = building;
		}
		else							// building not seen before, so add it
		{
			UnitType buildingType = building->getType();
			buildingTypeCounts[buildingType]++;
			buildingDataMap[building] = UnitData(building);
			
			if(buildingType == UnitTypes::Protoss_Photon_Cannon		||
				buildingType == UnitTypes::Terran_Bunker			||
				buildingType == UnitTypes::Terran_Missile_Turret	||
				buildingType == UnitTypes::Zerg_Spore_Colony		||
				buildingType == UnitTypes::Zerg_Sunken_Colony			)
			{
				opponentDefensiveStructures.push_back(building);
			}

			if(!building->isLifted())
			{
				BaseManager::Instance()->onOpponentBuildingShow(building);
				MapAnalyser::Instance()->onBuildingShow(building);
			}
		}
	}

	// cleanup
	for(auto it = buildingDataMap.begin(); it != buildingDataMap.end(); /**/)
	{
		if(shouldRemoveUnitData(it->second) || it->second.lastHP <= 0)
		{
			UnitType buildingType = it->second.unitType;
			buildingTypeCounts[buildingType]--;
			MapAnalyser::Instance()->onBuildingDestroy(it->second.lastPosition, buildingType);
			BaseManager::Instance()->onOpponentBuildingDestroy(it->second.unitPtr);
			it = buildingDataMap.erase(it);

			opponentDefensiveStructures.erase(it->first);
		}
		else
			++it;
	}
}

void OpponentTracker::updateUnitData()
{
	for(auto it = opponentUnits.begin(); it != opponentUnits.end(); ++it)
	{
		Unit unit = *it;

		if(!unit->isVisible())
			continue;

		auto it_data = unitDataMap.find(unit);

		if(it_data != unitDataMap.end())		// unit's data already in the map
		{
			UnitData& data = it_data->second;
			UnitType currentType = unit->getType();

			if(data.unitType != currentType && currentType != UnitTypes::Unknown)	// unit changed into some other type
			{
				unitTypeCounts[data.unitType]--;

				if(currentType.isBuilding())		// non-building unit morphed into a building
				{
					unitDataMap.erase(unit);
					buildingDataMap[unit] = UnitData(unit);
					buildingTypeCounts[currentType]++;

					BaseManager::Instance()->onOpponentBuildingShow(unit);
					MapAnalyser::Instance()->onBuildingShow(unit);
				}
				else
				{
					data.unitType = currentType;
					unitTypeCounts[data.unitType]++;
				}
			}

			data.completed = unit->isCompleted();
			data.id = unit->getID();
			data.lastHP = unit->getHitPoints();
			data.lastPosition = unit->getPosition();
			data.lastSeen = Broodwar->getFrameCount();
			data.unitPtr = unit;

			if(data.lastPosition != Positions::Unknown									&& 
				Broodwar->isVisible(data.lastPosition.x / 32, data.lastPosition.y / 32)	&& 
				!data.unitPtr->isVisible()													)
			{
				data.lastPosition = Positions::Unknown;
			}
		}
		else							// unit not seen before, so add it
		{
			unitTypeCounts[unit->getType()]++;
			unitDataMap[unit] = UnitData(unit);
		}
	}
}

void OpponentTracker::validateOpponentUnits()
{
	opponentBuildings.clear();
	opponentUnits.clear();

	Unitset units = Broodwar->enemy()->getUnits();

	for(auto it = units.begin(); it != units.end(); ++it)
	{
		if(UnitUtils::isUnitValid(*it))
		{
			Unit u = *it;
			UnitType type = u->getType();

			if(type.isBuilding() || type.isAddon())
			{
				opponentBuildings.push_back(u);
			}
			else if(type != UnitTypes::Protoss_Scarab		&&
					type != UnitTypes::Protoss_Interceptor	&&
					type != UnitTypes::Terran_Vulture_Spider_Mine)
			{
				opponentUnits.push_back(u);
			}
		}
	}
}

// Implementation note: DPS and range formulas based on the code of the Nova bot
void OpponentTracker::updateThreatMaps()
{
	const int currentFrame = Broodwar->getFrameCount();
	const int mapWidth = Broodwar->mapWidth();
	const int mapHeight = Broodwar->mapHeight();

	// reset maps
	ArrayUtils::initializeFloatArray(airThreatMap, mapWidth*mapHeight, 0.0f);
	ArrayUtils::initializeFloatArray(groundThreatMap, mapWidth*mapHeight, 0.0f);

	// first loop through buildings and set threat of defensive structures
	for(auto it = buildingDataMap.begin(); it != buildingDataMap.end(); ++it)
	{
		Unit building = (*it).first;
		UnitData data = (*it).second;
		UnitType type = data.unitType;

		if(type == UnitTypes::Zerg_Creep_Colony && building->isMorphing())
			type = UnitTypes::Zerg_Sunken_Colony;

		if(type == UnitTypes::Protoss_Photon_Cannon		|| 
			type == UnitTypes::Terran_Bunker			||
			type == UnitTypes::Terran_Missile_Turret	||
			type == UnitTypes::Zerg_Spore_Colony		||
			type == UnitTypes::Zerg_Sunken_Colony			)
		{
			TilePosition pos = PositionUtils::toTilePosition(data.lastPosition);
			WeaponType airWeapon = type.airWeapon();
			WeaponType groundWeapon = type.groundWeapon();

			float airDPS = dpsAir(data);
			float groundDPS = dpsGround(data);

			if(type == UnitTypes::Terran_Bunker)
			{
				airDPS = 38.4f;
				groundDPS = 38.4f;
			}

			// air threat
			if(airDPS > 0)
			{
				int maxRange = ceil(UnitUtils::rangeAir(building) / 32.0f) + 2;

				if(type == UnitTypes::Protoss_Photon_Cannon)
					maxRange++;
				else if(type == UnitTypes::Terran_Bunker)
					maxRange = 10;

				float distFactor = airDPS / maxRange;

				for(int ix = -maxRange; ix <= maxRange; ++ix)
				{
					if(ArrayUtils::boundsCheck(pos.x + ix, mapWidth))
					{
						for(int iy = -maxRange; iy <= maxRange; ++iy)
						{
							if(ArrayUtils::boundsCheck(pos.y + iy, mapHeight))
							{
								int dist = sqrt(float(ix * ix) + float(iy * iy));

								if(dist <= maxRange)
									ArrayUtils::addMatrixValue(airThreatMap, pos.y + iy, pos.x + ix, mapWidth, (airDPS - (distFactor * dist)));
							}
						}
					}
				}
			}

			// ground threat
			if(groundDPS > 0)
			{
				int maxRange = ceil(Broodwar->enemy()->weaponMaxRange(groundWeapon) / 32.0f) + 2;

				if(type == UnitTypes::Protoss_Photon_Cannon || type == UnitTypes::Zerg_Sunken_Colony)
					maxRange++;
				else if(type == UnitTypes::Terran_Bunker)
					maxRange = 10;

				float distFactor = groundDPS / maxRange;

				for(int ix = -maxRange; ix <= maxRange; ++ix)
				{
					if(ArrayUtils::boundsCheck(pos.x + ix, mapWidth))
					{
						for(int iy = -maxRange; iy <= maxRange; ++iy)
						{
							if(ArrayUtils::boundsCheck(pos.y + iy, mapHeight))
							{
								int dist = sqrt(float(ix * ix) + float(iy * iy));

								if(dist <= maxRange)
									ArrayUtils::addMatrixValue(groundThreatMap, pos.y + iy, pos.x + ix, mapWidth, (groundDPS - (distFactor * dist)));
							}
						}
					}
				}
			}
		}
	}

	// now loop through all other units and set their threat
	for(auto it = unitDataMap.begin(); it != unitDataMap.end(); ++it)
	{
		Unit unit = (*it).first;
		UnitData data = (*it).second;

		if(currentFrame - data.lastSeen > 120)	// been more than 5 seconds since we saw unit, so no longer consider this unit's threat
			continue;

		UnitType type = data.unitType;

		if(type.canAttack()								||
			type == UnitTypes::Protoss_High_Templar		||
			type == UnitTypes::Protoss_Dark_Archon		||
			type == UnitTypes::Protoss_Reaver			||
			type == UnitTypes::Protoss_Carrier			||
			type == UnitTypes::Terran_Medic					)
		{
			TilePosition pos = PositionUtils::toTilePosition(data.lastPosition);
			WeaponType airWeapon = type.airWeapon();
			WeaponType groundWeapon = type.groundWeapon();

			float airDPS = dpsAir(data);
			float groundDPS = dpsGround(data);

			// air threat
			if(airDPS > 0)
			{
				int maxRange = ceil(UnitUtils::rangeAir(data) / 32.0f) + 2;
				int doubleMaxRange = 2*maxRange;
				float doubleAirDPS = 2.0f*airDPS;
				float distFactor = airDPS / maxRange;

				for(int ix = -doubleMaxRange; ix <= doubleMaxRange; ++ix)
				{
					if(ArrayUtils::boundsCheck(pos.x + ix, mapWidth))
					{
						for(int iy = -doubleMaxRange; iy <= doubleMaxRange; ++iy)
						{
							if(ArrayUtils::boundsCheck(pos.y + iy, mapHeight))
							{
								int dist = sqrt(float(ix * ix) + float(iy * iy));

								if(dist <= doubleMaxRange)
									ArrayUtils::addMatrixValue(airThreatMap, pos.y + iy, pos.x + ix, mapWidth, (doubleAirDPS - (distFactor * dist)));
							}
						}
					}
				}
			}

			// ground threat
			if(groundDPS > 0)
			{
				int maxRange = ceil(UnitUtils::rangeGround(data) / 32.0f) + 2;

				int doubleMaxRange = 2 * maxRange;
				float doubleGroundDPS = 2 * groundDPS;
				float distFactor = groundDPS / maxRange;

				if(type.isWorker())
				{
					maxRange = 2;
					doubleMaxRange = 3;
				}
				else if(type == UnitTypes::Protoss_Zealot || type == UnitTypes::Zerg_Zergling || type == UnitTypes::Terran_Medic)
				{
					maxRange = 3;
					doubleMaxRange = 4;
				}
				else if(type == UnitTypes::Terran_Vulture_Spider_Mine)
				{
					maxRange = 3;
					doubleMaxRange = 4;
				}

				for(int ix = -doubleMaxRange; ix <= doubleMaxRange; ++ix)
				{
					if(ArrayUtils::boundsCheck(pos.x + ix, mapWidth))
					{
						for(int iy = -doubleMaxRange; iy <= doubleMaxRange; ++iy)
						{
							if(ArrayUtils::boundsCheck(pos.y + iy, mapHeight))
							{
								int dist = sqrt(float(ix * ix) + float(iy * iy));

								if(dist <= doubleMaxRange)
									ArrayUtils::addMatrixValue(groundThreatMap, pos.y + iy, pos.x + ix, mapWidth, (doubleGroundDPS - (distFactor * dist)));
							}
						}
					}
				}
			}
		}
	}
}

void OpponentTracker::computeClusters()
{
	Unitset clusterCandidates;

	opponentClusters = Clustering::getClustersForUnits(opponentDefensiveStructures, 192, 1);

	for(auto it = unitDataMap.begin(); it != unitDataMap.end(); ++it)
	{
		Unit unit = (*it).first;
		UnitType type = (*it).second.unitType;

		if(type == UnitTypes::Zerg_Overlord || type == UnitTypes::Zerg_Egg || type == UnitTypes::Terran_Vulture_Spider_Mine)
			continue;

		// don't add marines which are near a bunker, they'll probably enter the bunker
		if(type == UnitTypes::Terran_Marine)
		{
			bool skipMarine = false;

			for(auto defensiveCluster = opponentClusters.begin(); defensiveCluster != opponentClusters.end(); ++defensiveCluster)
			{
				const Unitset& defenses = *defensiveCluster;

				for(auto defense = defenses.begin(); defense != defenses.end(); ++defense)
				{
					Unit u = *defense;

					if(buildingDataMap.count(u) > 0 && buildingDataMap.at(u).unitType == UnitTypes::Terran_Bunker)
					{
						if(Distances::getDistance(buildingDataMap.at(u).lastPosition, (*it).second.lastPosition) < 200)
						{
							skipMarine = true;
							break;
						}
					}
				}

				if(skipMarine)
				{
					break;
				}
			}

			if(skipMarine)
				continue;
		}

		Position lastPosition = (*it).second.lastPosition;

		if(lastPosition != Positions::Unknown && (unit->isVisible() || !Broodwar->isVisible(PositionUtils::toTilePosition(lastPosition))))
		{
			clusterCandidates.push_back(unit);
		}
	}

	VectorUtils::append(opponentClusters, Clustering::getClustersForUnits(clusterCandidates, 192, 1));
}

const float OpponentTracker::getAirThreat(const int tileX, const int tileY) const
{
	return ArrayUtils::getMatrixValue(airThreatMap, tileY, tileX, Broodwar->mapWidth());
}

const float OpponentTracker::getAirThreat(const BWAPI::Position& position) const
{
	return getAirThreat(PositionUtils::toTilePosition(position));
}

const float OpponentTracker::getAirThreat(const BWAPI::TilePosition& tilePos) const
{
	return ArrayUtils::getMatrixValue(airThreatMap, tilePos.y, tilePos.x, Broodwar->mapWidth());
}

const float OpponentTracker::getAirThreat(const BWAPI::WalkPosition& walkPos) const
{
	return getAirThreat(PositionUtils::toTilePosition(walkPos));
}

const float OpponentTracker::getGroundThreat(const int tileX, const int tileY) const
{
	return ArrayUtils::getMatrixValue(groundThreatMap, tileY, tileX, Broodwar->mapWidth());
}

const float OpponentTracker::getGroundThreat(const BWAPI::Position& position) const
{
	return getGroundThreat(PositionUtils::toTilePosition(position));
}

const float OpponentTracker::getGroundThreat(const BWAPI::TilePosition& tilePos) const
{
	return ArrayUtils::getMatrixValue(groundThreatMap, tilePos.y, tilePos.x, Broodwar->mapWidth());
}

const float OpponentTracker::getGroundThreat(const BWAPI::WalkPosition& walkPos) const
{
	return getGroundThreat(PositionUtils::toTilePosition(walkPos));
}

const vector<Unitset>& OpponentTracker::getOpponentClusters() const
{
	return opponentClusters;
}

const int OpponentTracker::getLastHP(const Unit unit) const
{
	if(buildingDataMap.count(unit) > 0)
	{
		const UnitData& data = buildingDataMap.at(unit);
		return data.lastHP;
	}
	else if(unitDataMap.count(unit) > 0)
	{
		const UnitData& data = unitDataMap.at(unit);
		return data.lastHP;
	}
	else
		return 0;
}

const Position OpponentTracker::getLastPosition(const Unit unit) const
{
	if(buildingDataMap.count(unit) > 0)
	{
		const UnitData& data = buildingDataMap.at(unit);
		return data.lastPosition;
	}
	else if(unitDataMap.count(unit) > 0)
	{
		const UnitData& data = unitDataMap.at(unit);
		return data.lastPosition;
	}
	else
		return Positions::Unknown;
}

const UnitType OpponentTracker::getLastType(const Unit unit) const
{
	if(buildingDataMap.count(unit) > 0)
	{
		const UnitData& data = buildingDataMap.at(unit);
		return data.unitType;
	}
	else if(unitDataMap.count(unit) > 0)
	{
		const UnitData& data = unitDataMap.at(unit);
		return data.unitType;
	}
	else
		return UnitTypes::Unknown;
}

const bool OpponentTracker::wasCompleted(const Unit unit) const
{
	if(buildingDataMap.count(unit) > 0)
	{
		const UnitData& data = buildingDataMap.at(unit);
		return data.completed;
	}
	else if(unitDataMap.count(unit) > 0)
	{
		const UnitData& data = unitDataMap.at(unit);
		return data.completed;
	}
	else
		return true;
}

const Unitset& OpponentTracker::getOpponentDefensiveStructures() const
{
	return opponentDefensiveStructures; 
}

Unitset OpponentTracker::getOpponentsInRegion(const MapRegion* region) const
{
	Unitset opponents;
	MapAnalyser* mapAnalyser = MapAnalyser::Instance();

	for(auto it = opponentUnits.begin(); it != opponentUnits.end(); ++it)
	{
		Unit unit = (*it);
		Position pos = unit->getPosition();

		if(mapAnalyser->getRegion(pos) == region)
			opponents.push_back(unit);
	}

	for(auto it = buildingDataMap.begin(); it != buildingDataMap.end(); ++it)
	{
		Unit building = (*it).first;
		Position pos = getLastPosition(building);

		if(mapAnalyser->getRegion(pos) == region)
			opponents.push_back(building);
	}

	return opponents;
}

const unordered_map<Unit, UnitUtils::UnitData, UnitUtils::HashFunctionUnit>& OpponentTracker::getUnitDataMap() const
{
	return unitDataMap;
}

#pragma warning( pop )