/*	-----------------------------------------------------------------------------
	M A A S C R A F T

	StarCraft: Brood War - Bot

	Author: Dennis Soemers
	Maastricht University
	-----------------------------------------------------------------------------
*/

/*
	Implementation of MidGame.h
*/

#include "CommonIncludes.h"

#include "Distances.h"
#include "MapAnalyser.h"
#include "MathConstants.h"
#include "MidGame.h"
#include "StrategyManager.h"
#include "Tasks/BuyUpgrade.hpp"
#include "Tasks/ChooseTransition.hpp"
#include "Tasks/ConstructBuilding.hpp"
#include "Tasks/ConstructBuildingAtDistance.hpp"
#include "Tasks/Expand.hpp"
#include "Tasks/SpamUnitSequence.hpp"
#include "Tasks/TaskSchedulePreds.h"

using namespace BWAPI;
using namespace std;

/*
	PROTOSS vs PROTOSS
*/
const int MidGame::PvP_3_GateGoon()
{
	MapAnalyser* mapAnalyser = MapAnalyser::Instance();
	ProductionManager* productionManager = ProductionManager::Instance();
	BaseManager* baseManager = BaseManager::Instance();

	// Army Composition of 2 Dragoons + 2 Zealots
	vector<UnitType> armyComposition;
	armyComposition.push_back(UnitTypes::Protoss_Dragoon);
	armyComposition.push_back(UnitTypes::Protoss_Zealot);
	armyComposition.push_back(UnitTypes::Protoss_Dragoon);
	armyComposition.push_back(UnitTypes::Protoss_Zealot);

	productionManager->addTask(new SpamUnitSequence(Priorities::PRIORITY_ARMY_BUILDING_MID + 1, 
													armyComposition, Strategies::PvP_3_GateGoon));

	// also ensure we have 3 gateways
	int currentNumGateways = Broodwar->self()->allUnitCount(UnitTypes::Protoss_Gateway);

	while(currentNumGateways < 3)
	{
		productionManager->addTask(new ConstructBuilding<>
								(Priorities::PRIORITY_BASE_BUILDING_MID, UnitTypes::Protoss_Gateway));

		++currentNumGateways;
	}

	// Get range upgrade for Dragoons
	productionManager->addTask(new BuyUpgrade<TaskSchedulePreds::AtOrAboveTotalUnitCount<UnitTypes::Enum::Protoss_Dragoon, 1>>
												(Priorities::PRIORITY_ARMY_BUILDING_MID, UpgradeTypes::Singularity_Charge));

	// Get a fourth gateway
	productionManager->addTask(new ConstructBuilding<TaskSchedulePreds::HasOrIsUpgrading<UpgradeTypes::Enum::Singularity_Charge>>
							(Priorities::PRIORITY_BASE_BUILDING_MID, UnitTypes::Protoss_Gateway));

	// Next build Forge to unlock photon cannons
	productionManager->addTask(new ConstructBuilding<TaskSchedulePreds::AtOrAboveCompletedUnitCount<UnitTypes::Enum::Protoss_Gateway, 4>>
										(Priorities::PRIORITY_BASE_BUILDING_EARLY, UnitTypes::Protoss_Forge)								);

	// build photon cannons near every chokepoint
	const vector<Base>& bases = BaseManager::Instance()->getSelfBases();
	for(auto it_base = bases.begin(); it_base != bases.end(); ++it_base)
	{
		Base base = *it_base;
		MapRegion* region = mapAnalyser->getRegion(base.getDepotCenter());
		const vector<MapChokePoint*>& chokepoints = region->getChokepoints();

		for(auto it = chokepoints.begin(); it != chokepoints.end(); ++it)
		{
			MapChokePoint* choke = *it;

			productionManager->addTask(new ConstructBuildingAtDistance<TaskSchedulePreds::AtOrAboveCompletedUnitCount<UnitTypes::Enum::Protoss_Forge, 1>>
											(Priorities::PRIORITY_BASE_BUILDING_EARLY,
											UnitTypes::Protoss_Photon_Cannon, 
											PositionUtils::toTilePosition(choke->getCenter()), 2, 18));

			productionManager->addTask(new ConstructBuildingAtDistance<TaskSchedulePreds::AtOrAboveCompletedUnitCount<UnitTypes::Enum::Protoss_Forge, 1>>
											(Priorities::PRIORITY_BASE_BUILDING_EARLY,
											UnitTypes::Protoss_Photon_Cannon, 
											PositionUtils::toTilePosition(choke->getCenter()), 2, 18));
		}
	}

	// create first expansion
	const BaseLocation* mainBase = baseManager->getMainBaseLocation();
	const vector<BaseLocation*>& baseLocs = mapAnalyser->getBaseLocations();
	BaseLocation* nearest = nullptr;
	int minDistSq = MathConstants::MAX_INT;

	for(auto it = baseLocs.begin(); it != baseLocs.end(); ++it)
	{
		BaseLocation* loc = *it;

		if(baseManager->hasBaseAtLocation(Broodwar->self(), loc))
			continue;

		const int distSq = Distances::getSquaredDistance(loc->getOptimalDepotLocation(), mainBase->getOptimalDepotLocation());

		if(distSq < minDistSq && mapAnalyser->tilesConnected(loc->getOptimalDepotLocation(), mainBase->getOptimalDepotLocation()))
		{
			minDistSq = distSq;
			nearest = loc;
		}
	}

	if(nearest)
	{
		productionManager->addTask(new Expand<TaskSchedulePreds::AtOrAboveCompletedUnitCount<UnitTypes::Enum::Protoss_Forge, 1>>
										(Priorities::PRIORITY_ARMY_BUILDING_MID + 2,
										nearest->getOptimalDepotLocation()));

		productionManager->addTask(new ConstructBuilding<TaskSchedulePreds::AtOrAboveTotalUnitCount<UnitTypes::Enum::Protoss_Nexus, 2>>
										(Priorities::PRIORITY_BASE_BUILDING_LATE, 
										UnitTypes::Protoss_Pylon,
										nearest));

		productionManager->addTask(new ConstructBuilding<TaskSchedulePreds::AtOrAboveCompletedUnitCount<UnitTypes::Enum::Protoss_Nexus, 2>>
										(Priorities::PRIORITY_BASE_BUILDING_LATE, 
										UnitTypes::Protoss_Gateway,
										nearest));

		productionManager->addTask(new ConstructBuilding<TaskSchedulePreds::AtOrAboveCompletedUnitCount<UnitTypes::Enum::Protoss_Nexus, 2>>
										(Priorities::PRIORITY_BASE_BUILDING_LATE, 
										UnitTypes::Protoss_Photon_Cannon,
										nearest));
	}

	// Zealot speed upgrade
	productionManager->addTask(new ConstructBuilding<TaskSchedulePreds::AtOrAboveTotalUnitCount<UnitTypes::Enum::Protoss_Nexus, 2>>
									(Priorities::PRIORITY_BASE_BUILDING_LATE, 
									UnitTypes::Protoss_Citadel_of_Adun));

	productionManager->addTask(new BuyUpgrade<TaskSchedulePreds::AtOrAboveCompletedUnitCount<UnitTypes::Enum::Protoss_Citadel_of_Adun, 1>>
												(Priorities::PRIORITY_ARMY_BUILDING_MID, UpgradeTypes::Leg_Enhancements));

	//create second expansion
	BaseLocation* previous = nearest;
	nearest = nullptr;
	minDistSq = MathConstants::MAX_INT;

	for(auto it = baseLocs.begin(); it != baseLocs.end(); ++it)
	{
		BaseLocation* loc = *it;

		if(baseManager->hasBaseAtLocation(Broodwar->self(), loc) || loc == previous)
			continue;

		const int distSq = Distances::getSquaredDistance(loc->getOptimalDepotLocation(), mainBase->getOptimalDepotLocation());

		if(distSq < minDistSq && mapAnalyser->tilesConnected(loc->getOptimalDepotLocation(), mainBase->getOptimalDepotLocation()))
		{
			minDistSq = distSq;
			nearest = loc;
		}
	}

	if(nearest)
	{
		productionManager->addTask(new Expand<TaskSchedulePreds::AtOrAboveCompletedUnitCount<UnitTypes::Enum::Protoss_Gateway, 5>>
										(Priorities::PRIORITY_ARMY_BUILDING_MID + 2,
										nearest->getOptimalDepotLocation()));

		productionManager->addTask(new ConstructBuilding<TaskSchedulePreds::AtOrAboveTotalUnitCount<UnitTypes::Enum::Protoss_Nexus, 3>>
										(Priorities::PRIORITY_BASE_BUILDING_LATE, 
										UnitTypes::Protoss_Pylon,
										nearest));

		productionManager->addTask(new ConstructBuilding<TaskSchedulePreds::AtOrAboveCompletedUnitCount<UnitTypes::Enum::Protoss_Nexus, 3>>
										(Priorities::PRIORITY_BASE_BUILDING_LATE, 
										UnitTypes::Protoss_Gateway,
										nearest));

		productionManager->addTask(new ConstructBuilding<TaskSchedulePreds::AtOrAboveCompletedUnitCount<UnitTypes::Enum::Protoss_Nexus, 3>>
										(Priorities::PRIORITY_BASE_BUILDING_LATE, 
										UnitTypes::Protoss_Photon_Cannon,
										nearest));
	}

	// damage / armor upgrades
	productionManager->addTask(new BuyUpgrade<TaskSchedulePreds::AtOrAboveTotalUnitCount<UnitTypes::Enum::Protoss_Nexus, 3>>
												(Priorities::PRIORITY_ARMY_BUILDING_MID, UpgradeTypes::Protoss_Ground_Weapons));

	productionManager->addTask(new BuyUpgrade<TaskSchedulePreds::AtOrAboveTotalUnitCount<UnitTypes::Enum::Protoss_Nexus, 3>>
												(Priorities::PRIORITY_ARMY_BUILDING_MID, UpgradeTypes::Protoss_Ground_Armor));

	return Strategies::PvP_3_GateGoon;
}