/*	-----------------------------------------------------------------------------
	M A A S C R A F T

	StarCraft: Brood War - Bot

	Author: Dennis Soemers
	Maastricht University
	-----------------------------------------------------------------------------
*/

/*
	Implementation of MapGraph.h

	TODO: create an optimizeMemory() function to trim the all the vectors in graph/nodes/edges once processing is done
*/

#include "CommonIncludes.h"

#include "Distances.h"
#include "MapGraph.h"
#include "MapGraphEdge.h"
#include "MapGraphNode.h"
#include "PathFinder.h"
#include "PositionUtils.h"

using namespace BWAPI;
using namespace std;

MapGraph::MapGraph() : regions_to_nodes(), edges(), nodes() {}

MapGraph::~MapGraph()
{
	for(auto it = edges.begin(); it != edges.end(); ++it)
	{
		MapGraphEdge* edge = *it;
		delete edge;
		edge = nullptr;
	}
	edges.clear();

	for(auto it = nodes.begin(); it != nodes.end(); ++it)
	{
		MapGraphNode* node = *it;
		delete node;
		node = nullptr;
	}
	nodes.clear();
}

void MapGraph::initialize(const vector<MapRegion*>& mapRegions, const vector<MapChokePoint*>& chokes, const vector<BaseLocation*>& baseLocs)
{
	for(auto it = mapRegions.begin(); it != mapRegions.end(); ++it)
	{
		MapRegion* region = *it;

		if(region->getChokepoints().size() > 0)
			regions_to_nodes[region] = vector<MapGraphNode*>();
	}

	// add nodes for base locations
	for(auto it = baseLocs.begin(); it != baseLocs.end(); ++it)
	{
		BaseLocation* loc = *it;
		Position pos = PositionUtils::toPosition(loc->getOptimalDepotLocation());

		MapGraphNode* node = new MapGraphNode(pos, loc);
		nodes.push_back(node);

		MapRegion* baseRegion = node->getRegion();
		vector<MapGraphNode*>& regionNodes = regions_to_nodes[baseRegion];
		regionNodes.push_back(node);
	}

	// make sure each region has at least one node and make sure all nodes within a region are connected to each other
	for(auto it = regions_to_nodes.begin(); it != regions_to_nodes.end(); ++it)
	{
		MapRegion* region = (*it).first;
		vector<MapGraphNode*>& regionNodes = (*it).second;

		bool centroidNodeNecessary = true;
		for(auto it = regionNodes.begin(); it != regionNodes.end(); ++it)
		{
			if(Distances::getDistance((*it)->getPosition(), PositionUtils::toPosition(region->getCentroid())) < 150)
			{
				centroidNodeNecessary = false;
				break;
			}
		}

		// create a node in region centroid
		if(centroidNodeNecessary)
		{
			MapGraphNode* newNode = new MapGraphNode(PositionUtils::toPosition(region->getCentroid()));
			nodes.push_back(newNode);
			regionNodes.push_back(newNode);
		}

		// link all nodes in the region to each other
		if(regionNodes.size() > 1)
		{
			for(auto it_1 = regionNodes.begin(); it_1 != regionNodes.end(); ++it_1)
			{
				MapGraphNode* n1 = *it_1;

				for(auto it_2 = it_1 + 1; it_2 != regionNodes.end(); ++it_2)
				{
					MapGraphNode* n2 = *it_2;

					createEdge(n1, n2);
				}
			}
		}
	}

	// now add a node to each chokepoint and link them to region nodes
	for(auto it = chokes.begin(); it != chokes.end(); ++it)
	{
		MapChokePoint* choke = *it;

		MapRegion* region1 = choke->getRegions().first;
		MapRegion* region2 = choke->getRegions().second;

		Position centre = PositionUtils::toPosition(choke->getCenter());

		const Unitset& statics = Broodwar->getStaticMinerals();
		bool blocked = false;

		for(auto it = statics.begin(); it != statics.end(); ++it)
		{
			Unit mineral = *it;

			if(mineral->getInitialResources() > 0)
				continue;

			if(Distances::getDistance(centre, mineral->getInitialPosition()) < 50)
			{
				blocked = true;
				break;
			}
		}

		if(blocked)
		{
			continue;
		}

		MapGraphNode* node = new MapGraphNode(PositionUtils::toPosition(choke->getCenter()), choke);
		nodes.push_back(node);

		// loop through previous, non-choke nodes in an adjacent region, and link them to choke node
		vector<MapGraphNode*>& region1Nodes = regions_to_nodes[region1];
		vector<MapGraphNode*>& region2Nodes = regions_to_nodes[region2];

		size_t size1 = region1Nodes.size();
		size_t size2 = region2Nodes.size();

		for(size_t i = 0; i < size1; ++i)
		{
			MapGraphNode* otherNode = region1Nodes.at(i);

			if(otherNode->getType() == CHOKEPOINT)
				break;

			createEdge(node, otherNode);
		}

		for(size_t i = 0; i < size2; ++i)
		{
			MapGraphNode* otherNode = region2Nodes.at(i);

			if(otherNode->getType() == CHOKEPOINT)
				break;

			createEdge(node, otherNode);
		}

		region1Nodes.push_back(node);
		region2Nodes.push_back(node);
	}

	// make sure all chokes within a region are connected to each other
	for(auto it = regions_to_nodes.begin(); it != regions_to_nodes.end(); ++it)
	{
		MapRegion* region = (*it).first;
		vector<MapGraphNode*>& regionNodes = (*it).second;

		for(auto it_1 = regionNodes.begin(); it_1 != regionNodes.end(); ++it_1)
		{
			MapGraphNode* n1 = *it_1;

			if(!n1->getType() == CHOKEPOINT)
				continue;

			for(auto it_2 = it_1 + 1; it_2 != regionNodes.end(); ++it_2)
			{
				MapGraphNode* n2 = *it_2;

				if(!n1->isAdjacentTo(n2))
					createEdge(n1, n2);
			}
		}
	}

	// clean up nodes with degree = 0
	for(auto it = nodes.begin(); it != nodes.end(); /**/)
	{
		MapGraphNode* node = *it;
		
		if(node->degree() == 0)
		{
			if(node->getType() == CHOKEPOINT)
			{
				MapChokePoint* choke = node->getChokePoint();
				MapRegion* region1 = choke->getRegions().first;
				MapRegion* region2 = choke->getRegions().second;

				vector<MapGraphNode*>& nodes1 = regions_to_nodes[region1];
				vector<MapGraphNode*>& nodes2 = regions_to_nodes[region2];

				nodes1.erase(find(nodes1.begin(), nodes1.end(), node));
				nodes2.erase(find(nodes2.begin(), nodes2.end(), node));
			}
			else
			{
				MapRegion* region = node->getRegion();
				vector<MapGraphNode*>& regionNodes = regions_to_nodes[region];
				regionNodes.erase(find(regionNodes.begin(), regionNodes.end(), node));
			}

			it = nodes.erase(it);
			delete node;
		}
		else
		{
			++it;
		}
	}
}

void MapGraph::createEdge(MapGraphNode* n1, MapGraphNode* n2)
{
	if(n1->isAdjacentTo(n2))	// already adjacent
		return;

	WalkPosition pos1 = PositionUtils::toWalkPosition(n1->getPosition());
	WalkPosition pos2 = PositionUtils::toWalkPosition(n2->getPosition());

#ifdef MAASCRAFT_DEBUG
	//if(walkCost == MathConstants::MAX_FLOAT)
	//{
	//	LOG_WARNING("MapGraph::createEdges() for 2 nodes with walk cost == MathConstants::MAX_FLOAT")
	//	return;
	//}
#endif

	int straightLineDist = int(Distances::getDistance(PositionUtils::toPosition(pos1), PositionUtils::toPosition(pos2)));

#ifdef MAASCRAFT_DEBUG
	//if(pixelCost < straightLineDist)
	//{
	//	LOG_WARNING("MapGraph::createEdges() pixelCost < straightLineDist!")
	//}
#endif

	MapGraphEdge* walkEdge = new MapGraphEdge(n1, n2, straightLineDist, 1.1f * straightLineDist);

	n1->addEdge(walkEdge);
	n2->addEdge(walkEdge);

	edges.push_back(walkEdge);
}

const vector<MapGraphEdge*>& MapGraph::getEdges() const
{
	return edges;
}

const vector<MapGraphNode*>& MapGraph::getNodes() const
{
	return nodes;
}

const vector<MapGraphNode*>& MapGraph::getRegionNodes(MapRegion* region) const
{
	if(regions_to_nodes.count(region) > 0)
		return regions_to_nodes.at(region);
	else
		return nodes;
}