/*	-----------------------------------------------------------------------------
	M A A S C R A F T

	StarCraft: Brood War - Bot

	Author: Dennis Soemers
	Maastricht University
	-----------------------------------------------------------------------------
*/

#pragma once

#include "CommonIncludes.h"

#include <vector>

#include "ArrayUtils.h"
#include "BaseLocation.h"
#include "MapConnectedArea.h"
#include "MapChokePoint.h"
#include "MapGraph.h"
#include "MapRegion.h"
#include "PositionUtils.h"
#include "Tuples.h"

namespace Connectivities
{
	static const short LARGE_OBSTACLE = -2;
	static const short SMALL_OBSTACLE = -1;
	static const short UNASSIGNED = 0;
}

/*
	MapAnalyser objects can analyse a map and then be queried for the information, which will be cached.
*/
class MapAnalyser
{
private:
	int mapWidth;
	int mapHeight;
	int numWalkTiles;

	std::vector<MapConnectedArea*> walkableAreas;
	std::vector<MapConnectedArea*> obstacles;
	std::vector<MapChokePoint*> chokepoints;
	std::vector<MapRegion*> mapRegions;
	std::vector<BaseLocation*> baseLocations;

	bool* walkable;
	bool* chokeTiles;
	float* clearances;
	short* connectivities;		// -2 = big obstacle, -1 = small obstacle, 0 = unassigned, > 0 = connected walkable region
	BWAPI::WalkPosition* nearestObstacleTiles;
	short* regionNumbers;

	MapGraph graph;

	void computeWalkability();

	void computeConnectivity();
	void processConnectivity(int x, int y, short connectivity);

	void computeClearance();

	void computeRegions();
	const Pair<BWAPI::WalkPosition, BWAPI::WalkPosition> findChokeSides(const BWAPI::WalkPosition chokeCenter) const;

	void connectChokesToRegions();

	void pruneRegions();
	void mergeRegionInto(MapRegion* region1, MapRegion* region2, MapChokePoint* choke);		// merge region1 into region2

	void updateRegionNumbers();

	void findBaseLocations();

	void identifyStartLocations();

	void cleanup();

	MapAnalyser();
	~MapAnalyser();

	MapAnalyser(MapAnalyser const &){}

public:
	static MapAnalyser* Instance()
	{
		static MapAnalyser instance;
		static MapAnalyser* instance_ptr = &instance;
		return instance_ptr;
	}

	static const BWAPI::WalkPosition NULL_TILE;

	// Analyses the map currently being played
	void analyseCurrentMap();

	/*
		NOTE: all functions other than analyseCurrentMap() will have undefined results if called before analyseCurrentMap() has been called
	*/

	/*
		MAP ANALYSIS METHODS
	*/
	inline const bool isWalkable(const int walkX, const int walkY) const
	{
		return ArrayUtils::getMatrixValue(walkable, walkY, walkX, mapWidth);
	}

	inline const bool isWalkable(const BWAPI::WalkPosition& walkPos) const
	{
		return ArrayUtils::getMatrixValue(walkable, walkPos.y, walkPos.x, mapWidth);
	}


	void setWalkable(const BWAPI::TilePosition& tilePos, const bool isWalkable) const
	{
		BWAPI::WalkPosition leftTop = PositionUtils::toWalkPosition(tilePos);

		for(int y = 0; y < 4; ++y)
		{
			int walkY = leftTop.y + y;
			for(int x = 0; x < 4; ++x)
			{
				int walkX = leftTop.x + x;
				ArrayUtils::setMatrixValue(walkable, walkY, walkX, mapWidth, isWalkable);
			}
		}
	}

	void onBuildingShow(BWAPI::Unit building)
	{
		BWAPI::TilePosition leftTop = building->getTilePosition();
		BWAPI::UnitType buildingType = building->getType();

		const int left = leftTop.x;
		const int right = leftTop.x + buildingType.tileWidth();
		const int top = leftTop.y;
		const int bot = leftTop.y + buildingType.tileHeight();

		for(int y = top; y < bot; ++y)
		{
			for(int x = left; x < right; ++x)
			{
				setWalkable(BWAPI::TilePosition(x, y), false);
			}
		}
	}

	void onBuildingDestroy(BWAPI::Position& center, BWAPI::UnitType buildingType)
	{
		BWAPI::TilePosition leftTop = PositionUtils::toTilePosition(center - BWAPI::Position(buildingType.dimensionLeft(), 
																								buildingType.dimensionUp()));

		const int left = leftTop.x;
		const int right = leftTop.x + buildingType.tileWidth();
		const int top = leftTop.y;
		const int bot = leftTop.y + buildingType.tileHeight();

		for(int y = top; y < bot; ++y)
		{
			for(int x = left; x < right; ++x)
			{
				setWalkable(BWAPI::TilePosition(x, y), true);
			}
		}
	}

	const MapGraph* getGraph() const
	{
		return &graph;
	}

	inline const float getClearance(const int walkX, const int walkY) const			
	{	
		return ArrayUtils::getMatrixValue(clearances, walkY, walkX, mapWidth);
	}

	inline const float getClearance(const BWAPI::WalkPosition& walkPos) const
	{
		return ArrayUtils::getMatrixValue(clearances, walkPos.y, walkPos.x, mapWidth);
	}

	inline void setClearance(const int x, const int y, const float clearance)
	{
		ArrayUtils::setMatrixValue(clearances, y, x, mapWidth, clearance);
	}

	inline const short getConnectivity(const int x, const int y) const		
	{	
		return ArrayUtils::getMatrixValue(connectivities, y, x, mapWidth);
	}

	inline const short getConnectivity(BWAPI::WalkPosition& walkPos) const
	{
		return ArrayUtils::getMatrixValue(connectivities, walkPos.y, walkPos.x, mapWidth);
	}

	inline void setConnectivity(const int x, const int y, const short connectivity)
	{
		ArrayUtils::setMatrixValue(connectivities, y, x, mapWidth, connectivity);
	}

	inline const BWAPI::WalkPosition getNearestObstacleTile(const int x, const int y) const
	{
		return ArrayUtils::getMatrixValue(nearestObstacleTiles, y, x, mapWidth);
	}

	inline void setNearestObstacleTile(const int x, const int y, const BWAPI::WalkPosition nearestObstacleTile)
	{
		ArrayUtils::setMatrixValue(nearestObstacleTiles, y, x, mapWidth, nearestObstacleTile);
	}

	inline const bool isChokeTile(const int x, const int y) const		
	{	
		return ArrayUtils::getMatrixValue(chokeTiles, y, x, mapWidth);
	}

	inline void setChokeTile(const int x, const int y, const bool chokeTile)
	{
		ArrayUtils::setMatrixValue(chokeTiles, y, x, mapWidth, chokeTile);
	}

	// Returns map width and height in walk-tile resolution
	inline const int getMapHeightWalkTiles() const							{	return mapHeight;	}
	inline const int getMapWidthWalkTiles() const							{	return mapWidth;	}

	inline MapRegion* getRegion(const int walkX, const int walkY) const
	{
		int index = getRegionNumber(walkX, walkY) - 1;

		if(index < 0)
			return nullptr;

		return mapRegions[index];
	}

	inline MapRegion* getRegion(const short regionNumber) const
	{
		int index = regionNumber - 1;

		if(index < 0)
			return nullptr;

		return mapRegions[index];
	}

	inline MapRegion* getRegion(const BWAPI::Position& pos) const
	{
		int index = getRegionNumber(pos.x / 8, pos.y / 8) - 1;

		if(index < 0)
			return nullptr;

		return mapRegions[index];
	}

	inline MapRegion* getRegion(const BWAPI::TilePosition& tilePos) const
	{
		int index = getRegionNumber(tilePos.x * 4, tilePos.y * 4) - 1;

		if(index < 0)
			return nullptr;

		return mapRegions[index];
	}

	inline MapRegion* getRegion(const BWAPI::WalkPosition& walkPos) const
	{
		int index = getRegionNumber(walkPos.x, walkPos.y) - 1;

		if(index < 0)
			return nullptr;

		return mapRegions[index];
	}

	inline const short getRegionNumber(const int walkX, const int walkY) const		
	{	
		return ArrayUtils::getMatrixValue(regionNumbers, walkY, walkX, mapWidth);				
	}

	inline const short getRegionNumber(const BWAPI::TilePosition& tilePos) const
	{
		return ArrayUtils::getMatrixValue(regionNumbers, tilePos.y * 4, tilePos.x * 4, mapWidth);
	}

	inline void setRegionNumber(const int x, const int y, const short regionNumber)
	{
		ArrayUtils::setMatrixValue(regionNumbers, y, x, mapWidth, regionNumber);
	}

	inline const bool isSmallObstacle(BWAPI::WalkPosition t) const			{	return (getConnectivity(t.x, t.y) == Connectivities::SMALL_OBSTACLE);	}

	const bool tileExists(const int x, const int y) const;
	const bool tileExists(const BWAPI::TilePosition& position) const;
	const bool tileExists(const BWAPI::WalkPosition& position) const;

	inline const bool tilesConnected(const BWAPI::TilePosition& t1, const BWAPI::TilePosition& t2) const
	{
		return (getConnectivity(PositionUtils::toWalkPosition(t1)) == getConnectivity(PositionUtils::toWalkPosition(t2)));
	}

	inline const bool tilesConnected(const BWAPI::WalkPosition& t, int x, int y) const		
	{
		return (getConnectivity(x, y) == getConnectivity(t.x, t.y));
	}

	inline bool tilesConnected(const BWAPI::WalkPosition& pos1, const BWAPI::WalkPosition& pos2) const
	{
		return (getConnectivity(pos1.x, pos1.y) == getConnectivity(pos2.x, pos2.y));	
	}

	/*
		Methods useful outside of Map Analysis, to obtain information on the map
	*/

	// Returns whether a given unit can reach a given goal Position. Will keep in mind whether the unit flies or walks.
	const bool canReach(const BWAPI::Unit unit, const BWAPI::WalkPosition goal) const;

	// Returns whether or not a path exists from p1 to p2 for ground units.
	const bool pathExists(const BWAPI::WalkPosition& p1, const BWAPI::WalkPosition& p2) const;

	inline const std::vector<BaseLocation*>& getBaseLocations()
	{
		return baseLocations;
	}

	inline const std::vector<MapChokePoint*>& getChokepoints()
	{
		return chokepoints;
	}

	inline const bool positionsInSameRegion(const BWAPI::Position& p1, const BWAPI::Position& p2) const
	{
		BWAPI::WalkPosition w1 = PositionUtils::toWalkPosition(p1);
		BWAPI::WalkPosition w2 = PositionUtils::toWalkPosition(p2);
		return (getRegionNumber(w1.x, w1.y) == getRegionNumber(w2.x, w2.y));
	}

	inline const bool walkPositionsInSameRegion(const BWAPI::WalkPosition& p1, const BWAPI::WalkPosition& p2) const
	{
		return (getRegionNumber(p1.x, p1.y) == getRegionNumber(p2.x, p2.y));
	}

	struct TileOrderingClearance;
	struct RegionOrderingSize;
};